/* $Id: alloca.c,v 1.2 1995/06/21 18:50:16 stuart Exp $ */

/*
 * File  : alloca.c
 * Author: Stuart Menefy
 * Date  : 29/12/92
 *
 * Implementation of alloca() for the transputer.
 *
 * Revisions:
 */

#include <sys/types.h>
#include <stdlib.h>

/* We do not include <alloca.h>, because for lcc the prototype for the
 * called function is different from the prototype of the real function.
 * See implementation notes on alloca for details.
 */

/*
 * This structure is allocated just before the memory returned by alloca().
 * The blocks form a stack, with the prev pointer pointing to the previous
 * block (and NULL when there are no more blocks). The return_address is the
 * address which the calling function would normally have returned to after
 * the allocated blocks have been freed, or NULL if alloca() has been called
 * previously in this function.
 */
struct _alloca_block
{
  struct _alloca_block *prev; /* address of previous block or 0.           */
  int return_address;         /* 0 for more blocks or return address else. */
                              /* The allocated data comes here.            */
};

/*
 * Declare pointers defined in allocaptrs.c which are shared with
 * setjmp()/longjmp() code.
 */
extern struct _alloca_block* _alloca_last;
extern void (*_alloca_longjmp)(struct _alloca_block*);
extern void _alloca_free(void);
static void longjmp_tidyup(struct _alloca_block*);

/*
 * This is the alloca() function called by the user. The first paramter is
 * a pointer to the calling functions params, so that the return address can
 * be examined and altered if necessary. The size is the number of bytes to be
 * allocated.
 */
void *_alloca(volatile const int *params, size_t size)
{
    int return_address;
    struct _alloca_block *block;

    block = malloc(sizeof(struct _alloca_block) + size);
    if (block == NULL)
        return NULL;
    
    /* Check if the return address has already been set.
     * If not, set it, and store it in the structure,
     * otherwise, simply store a NULL. */
    return_address = *params;
    if (return_address != (int)_alloca_free)
        *((int*)params) = (int)_alloca_free;
    else
        return_address = 0;

    block->return_address = return_address;
    block->prev = _alloca_last;
    _alloca_last = block;

    /* Store the _alloca_longjmp pointer */
    _alloca_longjmp = longjmp_tidyup;
        
    return block+1;
}


/*
 * This function is called from _alloca_free when the function which
 * called alloca() returns.
 *
 * Its purpose is simply to remove allocated blocks from the stack until
 * the first allocated block is reached.
 */
int _alloca_helper()
{
    struct _alloca_block* block, *prev;
    int result;

    for (block = _alloca_last;
         (block->return_address == 0);
         block = prev)
    {
        prev = block->prev;
        free(block);
    }

    result = block->return_address;
    _alloca_last = block->prev;
    free(block);

    return result;
}

/*
 * This function is called from longjmp() when a program has at some stage
 * called alloca().
 *
 * Its purpose is simply to remove allocated blocks from the stack until
 * the stack level present when setjmp() was called is reached.
 */
static void longjmp_tidyup(struct _alloca_block* setjmp_block)
{
    struct _alloca_block* block, *prev;

    for (block = _alloca_last;
         block != setjmp_block;
         block = prev)
    {
        prev = block->prev;
        free(block);
    }

    _alloca_last = block;
}

