#!/bin/sh

#
# Name source and target files
#
MAKEFILEG=../GMakefile
MAKEFILE=../Makefile
ERRLISTG=../errlist.g
ERRLISTH=../errlist.h
POWERSG=../powers.g
POWERSH=../powers.h
LMRG=../lmr.g
LMRH=../lmr.h
SITEG=../site.g
SITEH=../site.h
ANSWERFILE=answer.log
COMMENTCHAR="#"

#
# Check for DOS environment
#
if echo ${PATH} | grep ';' >&- ; then
  NULL="nul"
  RM=`which rm`
  SYS="DOS"
  EXE=".exe"
else
  NULL="/dev/null"
  RM="/bin/rm"
  SYS="UNIX"
  EXE=""
fi

#
# Check for Minix environment
#
MINIX=0
if test -d /usr/include/minix ; then
  MINIX=1
fi

#
# Determine how to echo without trailing new line
#
if echo '\c' | grep . >${NULL} ; then
  ECHO='echo -n'
  NONL=
else
  ECHO=echo
  NONL='\c'
fi

#
# IFS manipulation
#
SETIFS='eval SIFS="${IFS}";IFS="${NIFS}"'
RESETIFS=`echo 'eval^IFS="${SIFS}"' | tr '^' '\037'`
NIFS=`${ECHO} "^${NONL}" | tr '^' '\037' ; echo "${COMMENTCHAR}"`
IFS="${IFS}${COMMENTCHAR}"

#
# Check for non-interactive mode and generate answer file
#
RESPONSE="eval echo"
TAG="\\${COMMENTCHAR}"
if tty -s ; then
  LOGANSWER=">> ${ANSWERFILE}"
else
  LOGANSWER="| tee -a ${ANSWERFILE} | sed -e 's/^\([^${COMMENTCHAR}]*\)\\${COMMENTCHAR}.*/\1/'"
fi
${ECHO} "${NONL}" > ${ANSWERFILE}

#
# Nice heading to show what's about to happen
#
echo ""
echo "Stdio Installation Script"

echo ""
${ECHO} "Do you wish to use a preconfigured site description? [yn] ${NONL}"
read YN COMMENT
${RESPONSE} "'${YN}'${TAG} Use preconfigured site description${TAG} ${LOGANSWER}"
if test "${YN}" = "y" ; then
  echo ""
  for f in `ls *.sit` ; do
    if grep \^`basename $f .sit` site.txt ; then :
    else
      echo "$f	No description"
    fi
  done
  echo ""
  DESC=""
  while test "${DESC}" = "" ; do
    ${ECHO} "Select a site description: ${NONL}"
    read DESC COMMENT
    ${RESPONSE} "'${DESC}'${TAG} Site description${TAG} ${LOGANSWER}"
    if test ! -f "${DESC}.sit" ; then
      echo "${DESC}: invalid site name"
      DESC=""
    fi
  done
  echo ""
  echo "Generating ${SITEH}..."
  ${ECHO} "${NONL}" > ${SITEH}
  echo "Generating ${MAKEFILE}..."
  ${ECHO} "${NONL}" > ${MAKEFILE}
  echo "Generating ${ERRLISTH}..."
  ${ECHO} "${NONL}" > ${ERRLISTH}
  echo "Generating ${POWERSH}..."
  ${ECHO} "${NONL}" > ${POWERSH}
  echo "Generating ${LMRH}..."
  ${ECHO} "${NONL}" > ${LMRH}
  patch -d.. -s + -s + -s + -s + -s < ${DESC}.sit
  exit 0
fi

echo ""
${ECHO} "Do you wish to create a new site description? [yn] ${NONL}"
read YN COMMENT
${RESPONSE} "'${YN}'${TAG} Create new site description${TAG} ${LOGANSWER}"
if test "${YN}" != "y" ; then
  exit 0
fi

echo ""
echo "Respond with ? for help at any prompt"
echo ""

#
# Get the name of this site
#
SITENAME="?"
while test "${SITENAME}" = "?" ; do
  ${ECHO} "Type the name to label this site description: [?] ${NONL}"
  read SITENAME COMMENT
  ${RESPONSE} "'${SITENAME}'${TAG} Site name${TAG} ${LOGANSWER}"
  if test "${SITENAME}" = "?" ; then
    cat <<EOF
This is the name which will be used to generate a .sit file. The makefile
will be edited so that make site will generate a site description of the
correct name.
EOF
  fi
done

#
# Find the makefile
#
echo ""
echo "Makefiles available:"
echo ""
if test "${MINIX}" -ne 0 ; then
  BASEMAKE=makefile.min
else
  BASEMAKE=makefile.nix
fi
(cd ..; ls makefile.*) | grep -v makefile.cpp | pr -3 -l1 -t -w60
echo ""
${ECHO} "Enter your choice of makefile (default: ${BASEMAKE}): ${NONL}"
read YN COMMENT
if test ! -z "${YN}" ; then
  BASEMAKE="${YN}"
fi
${RESPONSE} "'${BASEMAKE}'${TAG} Makefile${TAG} ${LOGANSWER}"
${RM} -f ${MAKEFILEG}
ln ../${BASEMAKE} ${MAKEFILEG}

#
# Determine where to find include files
#
echo ""
echo "Searching for the target include directory..."
INCLUDE=/usr/include
if test -d ${INCLUDE} 2>${NULL} ; then
  YN="?"
  while test "${YN}" = "?" ; do
    ${ECHO} "Is it acceptable to use ${INCLUDE}? [yn?] ${NONL}"
    read YN COMMENT
    ${RESPONSE} "'${YN}'${TAG} Use '${INCLUDE}'${TAG} ${LOGANSWER}"
    if test "${YN}" = "?" ; then
      cat <<EOF
This is the directory in which the include files for the
target environment (not the host environment).
EOF
    fi
  done
else
  YN="n"
fi
if test "${YN}" != "y" ; then
  INCLUDE="?"
  while test "${INCLUDE}" = "?" ; do
    ${ECHO} "Type in your include directory: [?] ${NONL}"
    read INCLUDE COMMENT
    ${RESPONSE} "'${INCLUDE}'${TAG} Include directory${TAG} ${LOGANSWER}"
    if test "${INCLUDE}" = "?" ; then
      cat <<EOF
This is the include directory for the target environment. For
example, when cross compiling for Minix under DOS, the DOS
include directory is in c:/tc/include, and the Minix include
directory is in /usr/include. I would respond with '/usr/include'
to the prompt when installing stdio into Minix and 'c:/tc/include'
when installing stdio into DOS.
EOF
    else
      if test ! -d "${INCLUDE}" ; then
	echo "Directory ${INCLUDE} is not accessible"
	INCLUDE="?"
      fi
    fi
  done
fi

#
# Determine where to find C preprocessor
#
echo ""
echo "Searching for host's cpp..."
if test -f /lib/cpp ; then
  CPP="/lib/cpp -P"
else
  CPP="/usr/lib/cpp -P"
fi
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to use ${CPP}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Use '${CPP}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
The C preprocessor is used to strip comments from header
files when trying to determine whether some features are
available in the host environment. The preprocessor should
be able to accept input from stdin and produce output on
stdout.
EOF
  fi
done
if test "${YN}" != "y" ; then
  CPP="?"
  while test "${CPP}" = "?" ; do
    ${ECHO} "Type in your cpp command: [?] ${NONL}"
    ${SETIFS}
    read CPP COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${CPP}'${TAG} CPP invocation${TAG} ${LOGANSWER}"
    if test "${CPP}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to run
the C preprocessor with input from stdin and output sent
to stdout. Typically a command like:

	/lib/cpp

would be used to do this.
EOF
    fi
  done
fi

#
# Determine where to find host C compiler
#
echo ""
echo "Searching for host's cc..."
CC="cc -o "
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to compile and load using ${CC}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Host compile and load '${CC}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
The host cc compiler is the one used to compile and load C programs
to be run on this machine under this operating system.
EOF
  fi
done
if test "${YN}" != "y" ; then
  CC="?"
  while test "${CC}" = "?" ; do
    ${ECHO} "Type in your host's cc command: [?] ${NONL}"
    ${SETIFS}
    read CC COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${CC}'${TAG} Host compile and load${TAG} ${LOGANSWER}"
    if test "${CC}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to compile
foo.c to foo. Typically the command

	cc -o foo foo.c
	
would be used to do this. If this were the case, you would
respond with 'cc -o ' to the prompt (note the space).

Under the DOS environment I use the command:

	tcc -A -Ic:/tc/include -Lc:/tc/lib -efoo foo.c

so I respond to this prompt with 'tcc -A -Ic:/tc/include -Lc:/tc/lib -e'
(note the lack of a space).
EOF
    fi
  done
fi

#
# Determine where to find target C compiler
#
echo ""
echo "Searching for target's cc..."
TCC="${CC}"
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to compile and load using ${TCC}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Target compile and load '${TCC}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
The target cc compiler is the one used to compile programs
for the target machine. It should be the same physical
program as the host compiler but with different switches.

If stdio is being installed for this operating system, then
the target and host compiler commands will be the same. If
you are cross compiling, the target command will probably
specify different include directories and the like.
EOF
  fi
done
if test "${YN}" != "y" ; then
  TCC="?"
  while test "${TCC}" = "?" ; do
    ${ECHO} "Type in your target's compile and load cc command: [?] ${NONL}"
    ${SETIFS}
    read TCC COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${TCC}'${TAG} Target compile and load${TAG} ${LOGANSWER}"
    if test "${TCC}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to compile
foo.c in this environment to foo for execution in the target
environment. Typically a command like:

	cc -I/cross/include -o foo foo.c
	
would be used to do this. If this were the case, you would
respond with 'cc -I/cross/include -o ' to the prompt (note
the space). The resulting executable will not be executed.
This command will only be used to check that programs can
be compiled successfully.

Under the DOS environment that I use to cross compile for
Minix, I use

	mcc -o foo foo.c

to compile Minix programs, so I would respond to the prompt
with 'mcc -o ' (note the space).
EOF
    fi
  done
fi

#
# Determine how to compile to intermediate code
#
TCCc=`echo "${TCC}" | sed -e 's/ *-[a-z] *$/ -c /'`
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to compile only using ${TCCc}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Target compile '${TCCc}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
This is the command given to the target C compiler to compile
but not load, a program.
EOF
  fi
done
if test "${YN}" != "y" ; then
  TCCc="?"
  while test "${TCCc}" = "?" ; do
    ${ECHO} "Type in your target's compile only cc command: [?] ${NONL}"
    ${SETIFS}
    read TCCc COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${TCCc}'${TAG} Target compile${TAG} ${LOGANSWER}"
    if test "${TCCc}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to compile
foo.c to foo.o. Typically the command

	cc -c foo.c
	
would be used to do this. If this were the case, you would
respond with 'cc -c ' to the prompt (note the space).

Under the DOS environment I use the command:

	tcc -A -Ic:/tc/include -Lc:/tc/lib -c foo.c

so I respond to this prompt with 'tcc -A -Ic:/tc/include -Lc:/tc/lib -c '
(note the space).
EOF
    fi
  done
fi

#
# Determine if floating point support required
#
echo ""
echo "Floating point support..."
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Do you want floating point support? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Floating point support${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
If floating point support is required, extra floating point modules will
be compiled. It may be necessary to place these in a special maths library
which is included at link time.
EOF
  fi
done
FPCODE="${YN}"

#
# Determine how to load floating point library code
#
echo ""
echo "Searching for host's floating point library..."
if test -f /lib/libm.a -o -f /usr/lib/libm.a ; then
  FPLIB="-lm"
else
  FPLIB=""
fi
YN="?"
if test "${FPCODE}" != "y" ; then
  echo "No floating point library."
  YN="y"
  FPLIB=""
fi
while test "${YN}" = "?" ; do
  if test -z "${FPLIB}" ; then
    ${ECHO} "No floating point library. Is this acceptable? [yn?] ${NONL}"
    read YN COMMENT
    ${RESPONSE} "'${YN}'${TAG} No floating point library${TAG} ${LOGANSWER}"
  else
    ${ECHO} "Is it acceptable to use ${FPLIB}? [yn?] ${NONL}"
    read YN COMMENT
    ${RESPONSE} "'${YN}'${TAG} Use '${FPLIB}'${TAG} ${LOGANSWER}"
  fi
  if test "${YN}" = "?" ; then
    cat <<EOF
This is the command line argument required to be concatenated to the end
of a compiler invocation command to link in the floating point library code
to a program.
EOF
  fi
done
if test "${YN}" != "y" ; then
  FPLIB="?"
  while test "${FPLIB}" = "?" ; do
    ${ECHO} "Type the floating point library command line switch: [?] ${NONL}"
    ${SETIFS}
    read FPLIB COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${FPLIB}'${TAG} Floating point load switch${TAG} ${LOGANSWER}"
    if test "${FPLIB}" = "?" ; then
      cat <<EOF
This is the command line switch required to load the floating point library
code. On most systems it would be -lm. Enter an empty line if there is no
floating point library to load.
EOF
    fi
  done
fi

#
# Determine how to compile floating point
#
echo ""
echo "Searching for host's floating point cc..."
if test "${MINIX}" -ne 0 ; then
  FPCC="cc -f -o "
else
  FPCC="${CC}"
fi
YN="?"
if test "${FPCODE}" != "y" ; then
  echo "No floating point support."
  YN="y"
  FPCC=""
fi
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to compile and load using ${FPCC}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Host fp compile and load '${FPCC}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
This is the command line require to compile and load floating point
C code to be run on this machine under this operating system.
EOF
  fi
done
if test "${YN}" != "y" ; then
  FPCC="?"
  while test "${FPCC}" = "?" ; do
    ${ECHO} "Type in your host's floating point compile and load cc command: [?] ${NONL}"
    ${SETIFS}
    read FPCC COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${FPCC}'${TAG} Host floating point compile and load${TAG} ${LOGANSWER}"
    if test "${FPCC}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to compile
and load a program that contains floating point code.
EOF
    fi
  done
fi

#
# Determine target's fp compile command
#
echo ""
echo "Searching for target's floating point cc..."
if test "${FPCC}" = "${CC}" ; then
  TFPCC="${TCC}"
else
  TFPCC="${FPCC}"
fi
YN="?"
if test "${FPCODE}" != "y" ; then
  echo "No floating point support."
  YN="y"
  TFPCC=""
fi
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to compile and load using ${TFPCC}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Target fp compile and load '${TFPCC}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
This is the command line require to compile and load floating point
C code to be run on the target machine.
EOF
  fi
done
if test "${YN}" != "y" ; then
  TFPCC="?"
  while test "${TFPCC}" = "?" ; do
    ${ECHO} "Type in your target's floating point compile and load cc command: [?] ${NONL}"
    ${SETIFS}
    read TFPCC COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${TFPCC}'${TAG} Target floating point compile and load${TAG} ${LOGANSWER}"
    if test "${TFPCC}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to compile and load
a program that contains floating point code for the target machine.
EOF
    fi
  done
fi
if test "${FPCC}" = "${CC}" ; then
  TFPCCc=`echo "${TCCc}" | sed -e 's/ *-[a-z] *$/ -c /'`
else
  TFPCCc=`echo "${FPCC}" | sed -e 's/ *-[a-z] *$/ -c /'`
fi
YN="?"
if test "${FPCODE}" != "y" ; then
  YN="y"
  TFPCCc=""
fi
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to compile only using ${TFPCCc}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Target fp compile '${TFPCCc}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
This is the command line require to compile floating point but not load
C code to be run on the target machine.
EOF
  fi
done
if test "${YN}" != "y" ; then
  TFPCCc="?"
  while test "${TFPCCc}" = "?" ; do
    ${ECHO} "Type in your target's floating point compile only cc command: [?] ${NONL}"
    ${SETIFS}
    read TFPCCc COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${TFPCCc}'${TAG} Target floating point compile${TAG} ${LOGANSWER}"
    if test "${TFPCCc}" = "?" ; then
      cat <<EOF
This is the command you would type if you wanted to compile, but not load
a program that contains floating point code for the target machine.
EOF
    fi
  done
fi

#
# Ask for any additional definitions
#
echo ""
echo "Additional definitions..."
if test "${MINIX}" -ne 0 ; then
  DEFINES="_MINIX"
else
  DEFINES=`sed -n -e '/^#'${SITENAME}' / {
s/^#[^ ]* //
p
n
}' < ${MAKEFILEG}`
fi
if test -z "${DEFINES}" ; then
  DEFINES=-D_`echo ${SITENAME} | tr '[a-z]' '[A-Z]'`
else
  DEFINES="-D${DEFINES}"
fi
DEFINES="${DEFINES} -D_POSIX_SOURCE"
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to use ${DEFINES}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Use '${DEFINES}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
The additional definitions will be used when the C preprocessor
is invoked by the makefile.
EOF
  fi
done
if test "${YN}" != "y" ; then
  DEFINES="?"
  while test "${DEFINES}" = "?" ; do
    ${ECHO} "Type in the definitions: [?] ${NONL}"
    ${SETIFS}
    read DEFINES COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${DEFINES}'${TAG} Macro definitions${TAG} ${LOGANSWER}"
    if test "${DEFINES}" = "?" ; then
      cat <<EOF
Extra definitions will probably include things like a feature
test macro for the current site. For example, _MSDOS may be
required to test for a DOS site, so '-D_MSDOS' would be
typed.
EOF
    fi
  done
fi

#
# Ask how to build a library
#
echo ""
echo "Checking on library construction..."
echo ""
echo "    1  Use ar and ranlib"
echo "    2  Use ar, lorder and tsort"
echo "    3  Use ar only"
echo ""
if test -f /bin/ranlib -o -f /usr/bin/ranlib ; then
  LIBTYPE=1
else
  if test -f /bin/lorder -o -f /usr/bin/lorder ; then
    LIBTYPE=2
  else
    LIBTYPE=3
  fi
fi
echo "Your site appears to use ${LIBTYPE}."
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is this acceptable? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Use library type '${LIBTYPE}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
This information will be used to modify the makefile to suit this site. The
makefile will then be able to libraries containing stdio. If the guess is
incorrect, respond in the negative and you will be prompted for the correct
combination.
EOF
  fi
done
if test "${YN}" != "y" ; then
  LIBTYPE="?"
  while test "${LIBTYPE}" = "?" ; do
    ${ECHO} "Enter the combination code: ${NONL}"
    read LIBTYPE COMMENT
    ${RESPONSE} "'${LIBTYPE}'${TAG} Library type${TAG} ${LOGANSWER}"
    if test "${LIBTYPE}" != "3" -a "${LIBTYPE}" != "2" -a "${LIBTYPE}" != "1"
    then
      LIBTYPE="?"
      cat <<EOF
The only permissible combinations are the ones listed above. If your site
has different requirements you will have to edit the makefile manually.
EOF
    fi
  done
fi
if test "${LIBTYPE}" = "1" ; then
  RANLIB=ranlib
  LORDER=echo
  TSORT=cat
fi
if test "${LIBTYPE}" = "2" ; then
  RANLIB=":"
  LORDER=lorder
  TSORT=tsort
fi
if test "${LIBTYPE}" = "3" ; then
  RANLIB=":"
  LORDER=echo
  TSORT=cat
fi

#
# Ask for the ack hack
#
if test "${MINIX}" -ne 0 ; then
  ACKHACK="?"
  echo ""
  echo "Checking for ACK NULL hack..."
  while test "${ACKHACK}" = "?" ; do
    ${ECHO} "Do you want stdio.h to #undef NULL? [yn?] ${NONL}"
    read ACKHACK COMMENT
    ${RESPONSE} "'${ACKHACK}'${TAG} NULL ACK Hack${TAG} ${LOGANSWER}"
    if test "${ACKHACK}" = "?" ; then
      cat <<EOF
The Minix ack compiler doesn't understand about ((void *) 0). Users of this
compiler will have to force NULL to be defined as (0). This is done by
including stdio.h as the last file in all programs and allowing stdio.h to
redefine any existing version of NULL.
EOF
    fi
  done
fi

#
# Find a good definition for NULL
#
echo ""
echo "Searching for a definition for NULL..."
STDIONULL="((void *) 0)"
if ${CPP} -DUSESTDDEF <ynull.h >${NULL} 2>${NULL} ; then
  STDIONULL=`${CPP} -DUSESTDDEF <ynull.h | grep '^Null' | sed -e 's/^Null //'`
else
  if ${CPP} -DUSESTDIO <ynull.h >${NULL} 2>${NULL} ; then
    STDIONULL=`${CPP} -DUSESTDIO <ynull.h | grep '^Null' | sed -e 's/^Null //'`
  fi
fi
YN="?"
while test "${YN}" = "?" ; do
  ${ECHO} "Is it acceptable to use ${STDIONULL}? [yn?] ${NONL}"
  read YN COMMENT
  ${RESPONSE} "'${YN}'${TAG} Use '${STDIONULL}'${TAG} ${LOGANSWER}"
  if test "${YN}" = "?" ; then
    cat <<EOF
NULL is used as an alias for the null pointer. The definition used by
stdio.h should be compatible with those in the other include files.
EOF
  fi
done
if test "${YN}" != "y" ; then
  STDIONULL="?"
  while test "${STDIONULL}" = "?" ; do
    ${ECHO} "Type in your definition for NULL: [?] ${NONL}"
    ${SETIFS}
    read STDIONULL COMMENT
    ${RESETIFS}
    ${RESPONSE} "'${STDIONULL}'${TAG} Definition for NULL${TAG} ${LOGANSWER}"
    if test "${STDIONULL}" = "?" ; then
      cat <<EOF
This definition for NULL will be inserted into the stdio.h file. It should
be compatible with the definitions in the other include files.
EOF
    fi
  done
fi

#
# Ask for debugging assertions
#
DEBUG="?"
echo ""
echo "Debugging assertions..."
while test "${DEBUG}" = "?" ; do
  ${ECHO} "Do you want to generate debugging assertions? [yn?] ${NONL}"
  read DEBUG COMMENT
  ${RESPONSE} "'${DEBUG}'${TAG} Generate debugging assertions${TAG} ${LOGANSWER}"
  if test "${DEBUG}" = "?" ; then
    cat <<EOF
The ASSERT() macro is used to generate assertions in the library. These
are useful while debugging the stdio library to ensure assumed conditions
do indeed hold. Stable versions can have this code removed so that
executables will be smaller and faster.
EOF
   fi
done

#
# Confirm that cc and cpp produce exit status
#
echo ""
echo "Checking compiler exit status..."
YN="y"
if ${CPP} <ykillcpp.c 1>${NULL} 2>${NULL}; then
  echo "Warning: ${CPP} does not return exit status"
  YN="n"
fi
if ${CC}ykillcc ykillcc.c 1>${NULL} 2>${NULL}; then
  echo "Warning: ${CC} does not return exit status"
  YN="n"
fi
if ${TCC}ykillcc ykillcc.c 1>${NULL} 2>${NULL}; then
  echo "Warning: ${TCC} does not return exit status"
  YN="n"
fi
if ${FPCC}ykillcc ykillcc.c 1>${NULL} 2>${NULL}; then
  echo "Warning: ${FPCC} does not return exit status"
  YN="n"
fi
if ${TFPCC}ykillcc ykillcc.c 1>${NULL} 2>${NULL}; then
  echo "Warning: ${TFPCC} does not return exit status"
  YN="n"
fi
if test "${YN}" = "y" ; then
  echo "Exit status appears to be ok"
else
  cat <<EOF
Failure to return exit status on error means that it is impossible
to determine if a compilation failed. Strange errors may result.
EOF
fi

#
# Edit the Makefile
#
echo ""
echo "Editing ${MAKEFILE}..."
if test -f /bin/cdiff ; then
  CDIFF="cdiff"
else
  CDIFF="diff -c"
fi
if cmp -s ${MAKEFILEG} ../makefile.nix ; then
  sed -e "s~__SITE__~${SITENAME}~g
s~__CPP__~${CPP}~g
s~__CC__~${TCCc}~g
s~__FPCC__~${TFPCCc}~g
s~__LD__~${TCC}~g
s~__FPLD__~${TFPCC}~g
s~__DEFINES__~${DEFINES}~g
s~__MATHLIB__~${FPLIB}~g
s~__RANLIB__~${RANLIB}~g
s~__LORDER__~${LORDER}~g
s~__TSORT__~${TSORT}~g
s~__CDIFF__~${CDIFF}~g" <${MAKEFILEG} >makefile.tmp
  ${RM} -f ${MAKEFILE} ${MAKEFILEG}
  mv makefile.tmp ${MAKEFILE}
else
  ${RM} -f ${MAKEFILE}
  cp ${MAKEFILEG} ${MAKEFILE}
  ${RM} -f ${MAKEFILEG}
fi

#
# Initialise configuration specification
#
${ECHO} "${NONL}" >${SITEH}

#
# Generate debugging flag
#
echo ""
echo "Include debugging assertions..."
if test "${DEBUG}" = "y" ; then
  echo "/*efine NDEBUG*/" | tee -a ${SITEH}
else
  echo "#define NDEBUG" | tee -a ${SITEH}
fi

#
# Generate definition for NULL
#
echo ""
echo "Generate definition for NULL..."
echo "#define NULLPOINTER ${STDIONULL}" | tee -a ${SITEH}

#
# Generate ACK compiler NULL hack flag
#
echo ""
echo "Generate ACK compiler null hack..."
if test "${ACKHACK}" = "y" ; then
  echo "#define ACKHACK" | tee -a ${SITEH}
else
  echo "/*efine ACKHACK*/" | tee -a ${SITEH}
fi

#
# Generate floating point support flag
#
echo ""
echo "Floating point support..."
if test "${FPCODE}" = "y" ; then
  echo "/*efine NOFLOAT*/" | tee -a ${SITEH}
else
  echo "#define NOFLOAT" | tee -a ${SITEH}
fi

#
# System error message table
#
echo ""
echo "Checking for sys_errlist[] and perror() clash..."
YN="y"
${TCC}yperror yperror.c 1>${NULL} 2>${NULL} && test -r yperror${EXE}
if test $? -eq 0 ; then
  ${TCC}yserror yserror.c 1>${NULL} 2>${NULL} && test -r yserror${EXE}
  if test $? -ne 0 ; then
    YN="n"
  fi
fi
if test "${MINIX}" -ne 0 ; then
  YN="y"
fi
if test "${YN}" = "n" ; then
  echo "/*efine ERRLIST*/" | tee -a ${SITEH}
else
  if test "${TCC}" = "${CC}" ; then
    if ${CC}ygerror ygerror.c 1>${NULL} && test -r ygerror${EXE} ; then
      :
    else
      exit 1
    fi
    sed -n -f ygerror.sed < ${INCLUDE}/errno.h | ./ygerror >${ERRLISTH}
  else
    sed -n -f yperror.sed < ${INCLUDE}/errno.h > yerrlist.dat
    if ${CC}yerrlist yerrlist.c 1>${NULL} && test -r yerrlist${EXE} ; then
      :
    else
      exit 1
    fi
    ./yerrlist >${ERRLISTH}
  fi
  echo "#define ERRLIST" | tee -a ${SITEH}
fi

#
# System errno variable
#
echo ""
echo "Checking for errno and perror() clash..."
YN="y"
${TCC}yerrno yerrno.c >${NULL} 2>${NULL} && test -r yerrno${EXE}
if test $? -eq 0 ; then
  ${TCC}yserrno yserrno.c >${NULL} 2>${NULL} && test -r yserrno${EXE}
  if test $? -ne 0 ; then
    YN="n"
  fi
fi
if test "${YN}" = "n" ; then
  echo "/*efine ERRNO*/" | tee -a ${SITEH}
else
  echo "#define ERRNO" | tee -a ${SITEH}
fi

#
# Check out the system
#
echo ""
echo "Checking for ANSI conformant compiler environment..."
STDC=0
${CC}ystdc ystdc.c 1>${NULL} 2>${NULL} && test -r ystdc${EXE}
if test $? -eq 0 ; then
  STDC=1
  cat <<EOF | tee -a ${SITEH}
#define STDARG
#define LIMITS
#define FLOAT
EOF
fi
if test ${STDC} -eq 0 ; then
  if ${CPP} -DUSESTDARG <ystdarg.h >${NULL} 2>${NULL} ; then
    echo "#define STDARG" | tee -a ${SITEH}
  else
    echo "/*efine STDARG*/" | tee -a ${SITEH}
  fi
fi
if test ${STDC} -eq 0 ; then
  if ${CC}ylimits ylimits.c 1>${NULL} && test -r ylimits${EXE} ; then
    :
  else
    exit 1
  fi
  if ${CPP} <ylimits.h >${NULL} 2>${NULL} ; then
    echo "#define LIMITS" | tee -a ${SITEH}
    for f in `${CPP} < ylimits.h | grep '^X' | sed -e 's/^X//'` ; do
      ./ylimits $f | tee -a ${SITEH}
    done
  else
    echo "/*efine LIMITS*/" | tee -a ${SITEH}
    ./ylimits | tee -a ${SITEH}
  fi
fi
if test "${FPCODE}" = "y" ; then
  if ${FPCC}yfloat yfloat.c 1>${NULL} && test -r yfloat${EXE} ; then
    :
  else
    exit 1
  fi
  if ${FPCC}yldbl yldbl.c 1>${NULL} 2>${NULL} && test -r yldbl${EXE} && ./yldbl ; then
    :
    echo "#define LONGDOUBLE" | tee -a ${SITEH}
  else
    echo "/*efine LONGDOUBLE*/" | tee -a ${SITEH}
  fi
  if ${CPP} <yfloat.h >${NULL} 2>${NULL} ; then
    if test ${STDC} -eq 0 ; then
      echo "#define FLOAT" | tee -a ${SITEH}
    fi
    for f in `${CPP} < yfloat.h | grep '^X[^L]' | sed -e 's/^X//'` ; do
      ./yfloat $f | tee -a ${SITEH}
    done
    for f in `${CPP} < yfloat.h | grep '^XL' | sed -e 's/^X//'` ; do
      d=`echo $f | sed -e 's/^L//'`
      echo "#define $f $d" | tee -a ${SITEH}
    done
  else
    echo "/*efine FLOAT*/" | tee -a ${SITEH}
    ./yfloat | tee -a ${SITEH}
  fi
fi

echo ""
${ECHO} "Checking for va_list in ${NONL}"
if ${CPP} -DUSESTDARG <ystdarg.h >${NULL} 2>${NULL} ; then
  echo "stdarg.h..."
  YARG="USESTDARG"
else
  echo "varargs.h..."
  YARG="USEVARARG"
fi
VALIST=`${CPP} -D${YARG} <ystdarg.h | sed -n -e '
/^[ 	]*typedef[ 	].*[^A-Za-z0-9_]va_list/ {
s/[ 	]*typedef[ 	]*\(.*[^A-Za-z0-9_]\)va_list[^A-Za-z0-9_]*/\1/
p
q
}'`
echo "#define VA_LIST_T ${VALIST}" | tee -a ${SITEH}

echo ""
echo "Checking for size_t in sys/types.h..."
if ${CC}ysizet ysizet.c 1>${NULL} && test -r ysizet${EXE} ; then
  :
else
  exit 1
fi
SIZET=`${CPP} <ytypes.h | sed -n -e '
/typedef.*size_t/ {
s/[ 	]*typedef[ 	]*\([^;]*\)[ 	]size_t[ 	]*;/\1/
p
q
}'`
USIZET=`./ysizet`
echo "#define SIZE_T ${SIZET}" | tee -a ${SITEH}
echo "#define USIZE_T ${USIZET}" | tee -a ${SITEH}

echo ""
echo "Checking for HUGE_VAL in math.h..."
if ${CPP} <ymath.h | grep -s '^huge_val$' ; then
  echo "/*efine HUGE_VAL*/" | tee -a ${SITEH}
else
  if ${CPP} <ymath.h | grep -s '^huge$' ; then
    echo "#define HUGE_VAL HUGE" | tee -a ${SITEH}
  else
    echo "#define HUGE_VAL LDBL_MAX" | tee -a ${SITEH}
  fi
fi

echo ""
echo "Checking compiler for correct (int) ((unsigned char) (x)) casts..."
if ${CC}yuchar yuchar.c 1>${NULL} && test -r yuchar${EXE} ; then
  :
else
  exit 1
fi
if ${CC}yumask yumask.c 1>${NULL} && test -r yumask${EXE} ; then
  :
else
  exit 1
fi
if ./yuchar ; then
  echo "#define UNSIGNEDCHAR 0" | tee -a ${SITEH}
else
  UCHAR=`./yumask`
  echo "#define UNSIGNEDCHAR ${UCHAR}" | tee -a ${SITEH}
fi

echo ""
echo "Checking for hidden system library..."
${TCC}yhidden yhidden.c >${NULL} 2>${NULL} && test -r yhidden${EXE}
if test $? -eq 0 ; then
  echo "#define HIDDENLIBC" | tee -a ${SITEH}
else
  echo "/*efine HIDDENLIBC*/" | tee -a ${SITEH}
fi

echo ""
echo "Checking for memcpy(), memchr() and memset() functions..."
if ${CPP} <ymemory.h >${NULL} 2>${NULL} ; then
  cat <<EOF | tee -a ${SITEH}
#include <memory.h>
#define MEMORY
/*efine MYMEMCPY*/
/*efine MYMEMCHR*/
/*efine MYMEMSET*/
EOF
else
  if ${CPP} <ystring.h | grep -s memcpy ; then
  cat <<EOF | tee -a ${SITEH}
#define MEMORY
/*efine MYMEMCPY*/
/*efine MYMEMCHR*/
/*efine MYMEMSET*/
EOF
  else
    echo "/*efine MEMORY*/" | tee -a ${SITEH}
    ${TCC}ybcopy ybcopy.c 1>${NULL} 2>${NULL} && test -r ybcopy${EXE}
    if test $? -eq 0 ; then
      cat <<EOF | tee -a ${SITEH}
void bcopy();
#define MEMCPY(a,b,c) bcopy((b),(a),(c))
/*efine MYMEMCPY*/
EOF
    else
      echo "#define MYMEMCPY" | tee -a ${SITEH}
    fi
    cat <<EOF | tee -a ${SITEH}
#define MYMEMCHR
#define MYMEMSET
EOF
  fi
fi

echo ""
echo "Checking for string.h functions..."
if ${CPP} <ystring.h >${NULL} 2>${NULL} ; then
  echo "#define STRING" | tee -a ${SITEH}
else
  echo "/*efine STRING*/" | tee -a ${SITEH}
  if ${CPP} <ystrings.h >${NULL} 2>${NULL} ; then
    echo "#include <strings.h>" | tee -a ${SITEH}
  fi
fi

echo ""
echo "Checking for careless tolower..."
${CC}ytolower ytolower.c 1>${NULL} 2>${NULL} && test -r ytolower${EXE}
if ${CPP} <ytolower.h | grep -s '^TOLOWER$' ; then
  if ${CPP} <ytolower.h | grep -s 'tolower' ; then
    :
  else
    if test "${STDC}" -eq 0 ; then
      echo "int tolower();" | tee -a ${SITEH}
    else
      echo "int tolower(int);" | tee -a ${SITEH}
    fi
  fi
fi
if ./ytolower ; then
  echo "#define TOLOWER(c) tolower(c)" | tee -a ${SITEH}
else
  echo "#define TOLOWER(c) (isupper((c))?tolower((c)):(c))" | tee -a ${SITEH}
fi

echo ""
echo "Checking for 3 argument opens..."
YOPEN="/*efine OPEN3*/"
if ${CPP} <yfcntl.h | grep -s '^open3$' ; then
  YOPEN="#define OPEN3"
fi
echo "${YOPEN}" | tee -a ${SITEH}

echo ""
echo "Checking for dup2 function..."
${TCC}ydup2 ydup2.c 1>${NULL} 2>${NULL} && test -r ydup2${EXE}
if test $? -eq 0 ; then
  echo "#define DUP2" | tee -a ${SITEH}
else
  echo "/*efine DUP2*/" | tee -a ${SITEH}
fi

echo ""
echo "Checking for rename function..."
${TCC}yrename yrename.c 1>${NULL} 2>${NULL} && test -r yrename${EXE}
if test $? -eq 0 ; then
  echo "#define RENAME" | tee -a ${SITEH}
else
  echo "/*efine RENAME*/" | tee -a ${SITEH}
fi

echo ""
echo "Checking for STD*_FILENO in unistd.h..."
${CPP} <yunistd.h >yunistd.dat
if grep -s _FILENO <yunistd.dat ; then
  echo "/*efine UNISTD*/" | tee -a ${SITEH}
  grep _FILENO <yunistd.dat | sed -e 's/^/#define /' | tee -a ${SITEH}
else
  echo "#define UNISTD" | tee -a ${SITEH}
fi

echo ""
echo "Checking for mode_t in fcntl.h..."
if ${CPP} <yfcntl.h | grep creat | grep -s mode_t ; then
  echo "#define MODE_T mode_t" | tee -a ${SITEH}
else
  echo "/*efine MODE_T mode_t*/" | tee -a ${SITEH}
fi

echo ""
echo "Checking for uid_t in pwd.h..."
if ${CPP} <ypwd.h | grep getpwuid | grep -s uid_t ; then
  echo "#define UID_T uid_t" | tee -a ${SITEH}
else
  echo "/*efine UID_T uid_t*/" | tee -a ${SITEH}
fi

echo ""
echo "Checking for void (*signal())()..."
if ${CPP} <ysignal.h | grep -s '^sig_dfl .*void' ; then
  echo "#define VOIDSIGNAL" | tee -a ${SITEH}
else
  echo "/*efine VOIDSIGNAL*/" | tee -a ${SITEH}
fi

echo ""
echo "Checking sys/types.h and time.h for Posix typedefs..."
${ECHO} "${NONL}" > ytypes.dat
if ${CPP} <ytypes.h >${NULL} 2>${NULL} ; then
  echo "#define SYSTYPES" | tee -a ${SITEH}
else
  echo "/*efine SYSTYPES*/" | tee -a ${SITEH}
fi
${CPP} <ytypes.h | grep -s off_t
if test $? -ne 0 ; then
  echo "typedef long off_t;			/* best guess */" >>ytypes.dat
fi
${CPP} <ytypes.h | grep -s pid_t
if test $? -ne 0 ; then
  echo "typedef int pid_t;			/* best guess */" >>ytypes.dat
fi
${CPP} <ytypes.h | grep -s uid_t
if test $? -ne 0 ; then
  echo "typedef int uid_t;			/* best guess */" >>ytypes.dat
fi
${CPP} <ytypes.h | grep -s time_t
if test $? -ne 0 ; then
  if ${CPP} <ytime.h | grep -s time_t ; then
    echo "#define TIME" >>ytypes.dat
  else
    echo "/*efine TIME*/" >>ytypes.dat
    echo "typedef long time_t;		/* best guess */" >>ytypes.dat
  fi
fi
${CPP} <ytypes.h | grep -s mode_t
if test $? -ne 0 ; then
  echo "typedef int mode_t;		/* best guess */" >>ytypes.dat
fi
if ${CPP} <ytypes.h | grep -s size_t ; then
  if ${CPP} <ytypes.h | grep -s X_SIZE_T ; then
    echo "#define _SIZE_T" >>ytypes.dat
  fi
fi
tee -a ${SITEH} <ytypes.dat

if test "${FPCODE}" != "y" ; then
  echo "/*efine DENORMAL*/" | tee -a ${SITEH}
  echo "/*efine TRUNCATE*/" | tee -a ${SITEH}
  echo "/*efine LMR*/"      | tee -a ${SITEH}
  echo "/*efine POWERS*/"   | tee -a ${SITEH}
fi

if test "${FPCODE}" = "y" ; then
  echo ""
  echo "Checking floating point denormalisation..."
  if ${FPCC}ydenorm ydenorm.c ${FPLIB} 1>${NULL} && test -r ydenorm${EXE} ; then
    :
  else
    exit 1
  fi
  if ./ydenorm ; then
    echo "#define DENORMAL" | tee -a ${SITEH}
  else
    echo "/*efine DENORMAL*/" | tee -a ${SITEH}
  fi
fi

if test "${FPCODE}" = "y" ; then
  echo ""
  echo "Checking floating point to integer conversion..."
  if ${FPCC}ytrunc ytrunc.c ${FPLIB} 1>${NULL} && test -r ytrunc${EXE} ; then
    :
  else
    exit 1
  fi
  if ./ytrunc ; then
    echo "#define TRUNCATE" | tee -a ${SITEH}
  else
    echo "/*efine TRUNCATE*/" | tee -a ${SITEH}
  fi
fi

if test "${FPCODE}" = "y" ; then
  echo ""
  echo "Generating LDBL_MAX / FLT_RADIX..."
  if ${FPCC}ylmr ylmr.c ${FPLIB} 1>${NULL} && test -r ylmr${EXE} ; then
    :
  else
    exit 1
  fi
  ./ylmr > ${LMRH}
  echo "#define LMR" | tee -a ${SITEH}
fi

if test "${FPCODE}" = "y" ; then
  echo ""
  echo "Generating power table..."
  if ${FPCC}ypow10 ypow10.c ${FPLIB} 1>${NULL} && test -r ypow10${EXE} ; then
    :
  else
    exit 1
  fi
  SHIFT=`./ypow10 0 0 1 0 | sed -n -e '/bit/ !d
s/[^0-9]*\([1-9][0-9]*\) bit.*/\1/
p
'`
  SHIFT=`expr ${SHIFT} \* 3 / 2`
  echo "#define NEGEXPONENT	-${SHIFT}" > ${POWERSH}
  echo "#define POSEXPONENT	0"         >> ${POWERSH}
  echo "#ifdef SMALL" >> ${POWERSH}
  echo "#undef SMALL" >> ${POWERSH}
  ./ypow10 g -1 -15 -1 0 >> ${POWERSH}
  ./ypow10 0 15 1 0 >> ${POWERSH}
  echo "#endif" >> ${POWERSH}
  echo "#ifdef LARGE" >> ${POWERSH}
  echo "#undef LARGE" >> ${POWERSH}
fi
cat ${SITEH} > ypow10.dat
cat <<'EOF' >> ypow10.dat
#ifdef	FLOAT
#include <float.h>
#endif
#if	LDBL_MAX_10_EXP-0 != 0
P LDBL_MAX_10_EXP
N -LDBL_MAX_10_EXP
#else
P DBL_MAX_10_EXP
N -DBL_MAX_10_EXP
#endif
EOF
if test "${FPCODE}" = "y" ; then
  ./ypow10 g -16 \
	     `${CPP} < ypow10.dat | grep '^N' | \
	      sed -e 's/^N//' -e '$s/[ 	]//g'` \
	     -16 ${SHIFT} >> ${POWERSH}
  ./ypow10 g 16 \
	     `${CPP} < ypow10.dat | grep '^P' | \
	      sed -e 's/^P//' -e '$s/[ 	]//g'` \
	   16 0 >> ${POWERSH}
fi
if test "${FPCODE}" = "y" ; then
  echo "#endif" >> ${POWERSH}
  echo "#define POWERS" | tee -a ${SITEH}
fi

#
# Cleanup
#
${RM} -f *.s *.o *.exe *.dat `ls | grep -v '\.'`

exit 0
