/*
 * Function: 	strftime
 * Purpose:	produce a string from a format string and a time
 *		structure.
 * In:		maxsize	...	maximum size of the resulting string
 *				(incl. \0)
 *		format	...	format string. all characters are copied
 *				verbatim to s except for the following
 *				format specifiers:
 *				%a	abbr. weekday
 *				%A	full weekday
 *				%b	abbr. month name
 *				%B	full month name
 *				%c	date and time
 *				%d	day of month (01--31)
 *				%H	hour (00--23)
 *				%I	hour (01--12)
 *				%m	month (01-12)
 *				%M	minute (00-59)
 *				%p	am/pm
 *				%S	second (00-61)
 *				%U	week of year, starting at sunday
 *					(00-52)
 *				%w	weekday as number (0--6)
 *				%W	week of year, starting at monday
 *					(00-52)
 *				%x	Date
 *				%X	Time
 *				%y	year (00--99)
 *				%Y	year 
 *				%Z	timezone
 *				%%	%
 *		timeptr	...	time information
 * Out:		s	...	resulting string
 * Returns:	length of resulting string or 0 if maxsize would be
 * 		exceeded.
 *
 * $Log: strftime.c,v $
 * Revision 1.1  1995/07/18  16:51:14  stuart
 * Origional 1.6.25 version
 *
 *
 * Modified by Michael Haardt to work with setlocale(), also added all
 * the locale tables with help of some friendly people on IRC.
 *
 * Revision 1.4  1994/08/02  08:12:29  hjp
 * implemented time zones
 *
 * Revision 1.3  1994/08/01  20:33:30  hjp
 * added a hook for locales.
 *
 * Revision 1.2  1992/12/17  05:58:44  hjp
 * bug fix: %d should work now
 *
 * Revision 1.1  1992/05/30  22:54:38  hjp
 * Initial revision
 *
 */

#include <assert.h>
#include <string.h>
#include <time.h>
#include <minix/locale.h>

extern char *_tzname[2];

size_t strftime (s, maxsize, format, timeptr)
	char		*s;
	size_t		maxsize;
	const char	*format;
	const struct tm *timeptr;
{
	size_t	i = 0;
	size_t	l;
	int	idata;
	char	*sdata = NULL;
	enum {
		NOTHING, STRING, INT, COMPLEX
	}	type;
	int	j;

	while (*format) {
		#ifdef DEBUG
		printf ("*format = `%c'\n", *format);
		#endif
		if (*format == '%') {
			switch (* ++ format) {
			case 'a':
				sdata = __strftime_locale.abbrweekday[timeptr->tm_wday];
				type = STRING;
				break;
			case 'A':
				sdata = __strftime_locale.fullweekday[timeptr->tm_wday];
				type = STRING;
				break;
			case 'b':
				sdata = __strftime_locale.abbrmonth[timeptr->tm_mon];
				type = STRING;
				break;
			case 'B':
				sdata = __strftime_locale.fullmonth[timeptr->tm_mon];
				type = STRING;
				break;
			case 'c':
				l = strftime (s, maxsize - i, __strftime_locale.c_format, timeptr);
				type = COMPLEX;
				break;
			case 'd':
				idata = timeptr->tm_mday;
				l = 2;
				type = INT;
				break;
			case 'H':
				idata = timeptr->tm_hour;
				l = 2;
				type = INT;
				break;
			case 'I':
				idata = ((timeptr->tm_hour + 11) % 12) + 1;
				l = 2;
				type = INT;
				break;
			case 'j':
				idata = timeptr->tm_yday + 1;
				l = 3;
				type = INT;
				break;
			case 'm':
				idata = timeptr->tm_mon + 1;
				l = 2;
				type = INT;
				break;
			case 'M':
				idata = timeptr->tm_min;
				l = 2;
				type = INT;
				break;
			case 'p':
				sdata = timeptr->tm_hour >= 12 ? __strftime_locale.pm : __strftime_locale.am;
				type = STRING;
				break;
			case 'S':
				idata = timeptr->tm_sec;
				l = 2;
				type = INT;
				break;
			case 'U':
				idata = (timeptr->tm_yday - timeptr->tm_wday) / 7;
				l = 2;
				type = INT;
				break;
			case 'w':
				idata = timeptr->tm_wday;
				l = 1;
				type = INT;
				break;
			case 'W':
				idata = (timeptr->tm_yday - timeptr->tm_wday + 1) / 7;
				l = 2;
				type = INT;
				break;
			case 'x':
				l = strftime (s, maxsize - i, __strftime_locale.x_format, timeptr);
				type = COMPLEX;
				break;
			case 'X':
				l = strftime (s, maxsize - i, __strftime_locale.X_format, timeptr);
				type = COMPLEX;
				break;
			case 'y':
				idata = timeptr->tm_year;
				l = 2;
				type = INT;
				break;
			case 'Y':
				idata = timeptr->tm_year + 1900;
				l = 4;
				type = INT;
				break;
			case 'Z':
				sdata = _tzname[timeptr->tm_isdst];
				type = STRING;
				break;
			case '%':
				sdata = "%";
				type = STRING;
				break;
			default:
				/* ignore illegal format specifiers */
				type = NOTHING;
				break;
			}
			#ifdef DEBUG
			printf ("type = %d, idata = %d, sdata = `%s', l = %d\n",
				type, idata, sdata, l);
			#endif
			switch (type) {
			case NOTHING:
				break;
			case INT:
				if (i + l >= maxsize) return 0;
				s += l;
				for (j = 1; j <= l; j ++) {
					s [-j] = idata % 10 + '0';
					idata /= 10;
				}
				i += l;
				break;
			case STRING:
				l = strlen (sdata);
				if (i + l >= maxsize) return 0;
				memcpy (s, sdata, l);
				s += l;
				i += l;
				break;
			case COMPLEX:
				if (l == 0) return 0;
				i += l;
				s += l;
				break;
			default:
				assert (0);
				break;
			}
		} else {
			* s ++ = * format;
			i ++;
		}
		format ++;
	}
	if (i >= maxsize) return 0;
	*s = '\0';
	return i;
}
