/* #includes */ /*{{{C}}}*//*{{{*/
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
/*}}}*/
/* #defines */ /*{{{*/
#define BOOTLEN 12 /* length of boot-neighbour message header */
#define APPLEN  16 /* length of boot-application message header */
#undef DEBUG
/*}}}*/

/* types */ /*{{{*/
struct Proc
{
  int used;
  long ram;
  int link[4];
  char *boot;
  unsigned int bootlen;
  char app[128];
  unsigned int applen;
};

struct Bootdir
{
  char *dir;
  struct Bootdir *next;
};
/*}}}*/
/* variables */ /*{{{*/
static struct Proc *proc;
static int *boot;
static struct Bootdir *bootdir=(struct Bootdir*)0;
static struct Bootdir **bootdirtail=&bootdir;
/*}}}*/

/* putword */ /*{{{*/
static void putword(FILE *fp, long word)
{
#ifdef DEBUG
  fprintf(stderr,"%ld:4",word);
#endif
  fputc(word&0xff,fp);
  fputc((word>>8)&0xff,fp);
  fputc((word>>16)&0xff,fp);
  fputc((word>>24)&0xff,fp);
}
/*}}}*/
/* putfile */ /*{{{*/
static void putfile(FILE *fp, const char *file)
{
  int fd;
#ifdef DEBUG
  int len=0;
#endif

  if ((fd=open(file,O_RDONLY))==-1)
  {
    fprintf(stderr,"btl: can not open %s: %s\n",file,strerror(errno));
    exit(1);
  }
  else
  {
    char buffer[1024];
    ssize_t r;

    while ((r=read(fd,buffer,sizeof(buffer)))>0)
    {
      fwrite(buffer,r,1,fp);
#ifdef DEBUG
      len+=r;
#endif
    }
    close(fd);
  }
#ifdef DEBUG
  fprintf(stderr,"%s:%d",file,len);
#endif
}
/*}}}*/
/* findboot */ /*{{{*/
static char *findboot(const char *name)
{
  struct Bootdir *p;

  for (p=bootdir; p; p=p->next)
  {
    static char ln[256];

    strcpy(ln,bootdir->dir);
    strcat(ln,"/");
    strcat(ln,name);
    if (access(ln,R_OK)==0) return strcpy(malloc(strlen(ln)+1),ln);
  }
  return (char*)0;
}
/*}}}*/
/* appboot */ /*{{{*/
static void appboot(const char *name)
{
  assert(name!=(const char*)0);
  *bootdirtail=malloc(sizeof(struct Bootdir));
  (*bootdirtail)->dir=strcpy(malloc(strlen(name)+1),name);
  bootdirtail=&((*bootdirtail)->next);
}
/*}}}*/
/* bootproc */ /*{{{*/
static void bootproc(int node, int hops)
{
  int i;

  ++proc[node].used;
  for (i=0; i<hops; ++i)
  {
    putword(stdout,1);
#ifdef DEBUG
    fprintf(stderr," link ");
#endif
    putword(stdout,boot[i]);
#ifdef DEBUG
    fprintf(stderr," len ");
#endif
    putword(stdout,proc[node].bootlen+(hops-i>1 ? hops-i-1 : 0)*BOOTLEN);
#ifdef DEBUG
    fprintf(stderr,"\n");
#endif
  }
#ifdef DEBUG
  fprintf(stderr,"boot ");
#endif
  putfile(stdout,proc[node].boot);
#ifdef DEBUG
  fprintf(stderr,"\n");
#endif
  for (i=0; i<4; ++i) if (proc[node].link[i]>=0 && proc[proc[node].link[i]].used==1)
  {
    boot[hops]=i;
    bootproc(proc[node].link[i],hops+1);
  }
  for (i=0; i<hops; ++i)
  {
    putword(stdout,1);
#ifdef DEBUG
    fprintf(stderr," link ");
#endif
    putword(stdout,boot[i]);
#ifdef DEBUG
    fprintf(stderr," len ");
#endif
    putword(stdout,proc[node].applen+APPLEN+(hops-i>1 ? hops-i-1 : 0)*BOOTLEN);
#ifdef DEBUG
    fprintf(stderr,"\n");
#endif
  }
  putword(stdout,0);
#ifdef DEBUG
  fprintf(stderr," ram ");
#endif
  putword(stdout,proc[node].ram*1024);
#ifdef DEBUG
  fprintf(stderr," proc ");
#endif
  putword(stdout,node);
#ifdef DEBUG
  fprintf(stderr," len ");
#endif
  putword(stdout,proc[node].applen);
#ifdef DEBUG
  fprintf(stderr," app ");
#endif
  putfile(stdout,proc[node].app);
#ifdef DEBUG
  fprintf(stderr,"\n");
#endif
}
/*}}}*/

/* main */ /*{{{*/
int main(int argc, char *argv[])
{
  /* variables */ /*{{{*/
  char ln[128];
  char link[4][3];
  char bootfile[128],appfile[128];
  int line,node,ram,i,procs;
  /*}}}*/

  /* parse options */ /*{{{*/
  appboot("/usr/lcc/boot");
  /*}}}*/
  /* read config file */ /*{{{*/
  procs=0;
  proc=(struct Proc*)0;
  for (line=1; fgets(ln,sizeof(ln),stdin); ++line) if (ln[0]!='\n' && ln[0]!='#')
  {
    if (sscanf(ln,"%d %d %s %s %s %s %s %s",&node,&ram,link[0],link[1],link[2],link[3],bootfile,appfile)!=8)
    {
      fprintf(stderr,"btl:%d:incomplete specification\n",line);
      exit(1);
    }
    if (node>=0)
    {
      struct stat buf;

      if (node>=procs)
      {
        int oldprocs;

        oldprocs=procs;
        proc=realloc(proc,(procs=(((node+8)>>3)<<3))*sizeof(struct Proc));
        for (; oldprocs<procs; ++oldprocs) proc[oldprocs].used=0;
      }
      proc[node].used=1;
      if ((proc[node].boot=findboot(bootfile))==(char*)0)
      {
        fprintf(stderr,"btl:can not find %s\n",bootfile);
        exit(1);
      }
      if (stat(proc[node].boot,&buf)==-1)
      {
        fprintf(stderr,"btl:can not open %s:%s\n",proc[node].boot,strerror(errno));
        exit(1);
      }
      else proc[node].bootlen=buf.st_size;
      strcpy(proc[node].app,appfile);
      if (stat(appfile,&buf)==-1)
      {
        fprintf(stderr,"btl:stat of %s failed:%s\n",appfile,strerror(errno));
        exit(1);
      }
      else proc[node].applen=buf.st_size;
      if (ram>0) proc[node].ram=ram;
      else
      {
        fprintf(stderr,"btl:%d:invalid memory size\n",line);
        exit(1);
      }
      for (i=0; i<4; ++i)
      {
        if (link[i][0]=='-') proc[node].link[i]=-1;
        else
        {
          char *end;

          proc[node].link[i]=strtol(link[i],&end,0);
          if (end==link[i] || proc[node].link[i]<0)
          {
            fprintf(stderr,"btl:%d:invalid link number for link %d\n",line,i);
            exit(1);
          }
        }
      }
    }
    else
    {
      fprintf(stderr,"btl:%d:invalid node number\n",line);
      exit(1);
    }
  }
  boot=malloc(procs*sizeof(int));
  /*}}}*/
  bootproc(0,0);
  return 0;
}
/*}}}*/
