/*{{{}}}*/
/*{{{  #includes*/
#undef  _MINIX
#define _MINIX          1
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE   1
#undef  _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 2

#ifdef _XOPEN_SOURCE
#include <nl_types.h>
#else
typedef long nl_catd;
static nl_catd catopen(char *name, int oflag) { return 0; }
static char *catgets(nl_catd catd, int set_id, int msg_id, char *msg) { return msg; }
static void catclose(nl_catd catd) { }
#endif

#include <a.out.h>

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <locale.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "../common/number.h"
/*}}}  */

/*{{{  types*/
/* Used for reducing shifted instructions to more complex instructions */
typedef enum { DONTCARE, PREFIX, LDC, SPECIAL } Instr;
typedef enum { FALSE, TRUE } Bool;
/* How is an address referenced? */
typedef enum { NONE, CALL, JUMP, LDPI, DATA } Reference;
/* If and how should an operand be printed? */
typedef enum { NOT, SIGNED, UNSIGNED } Opvalid;

typedef unsigned long Address;

typedef struct
{
  Reference reference;
  Address address;
  char *str;
  unsigned int number;
  struct Symbol *next;
} Symbol;
/*}}}  */
/*{{{  variables*/
nl_catd catd;

static struct
{
  Address pc;
  Instr instr;
  char s[10];
  int nibble;
  long op;
  Opvalid opvalid;
  Reference reference;
} putbuf[16];

static int putbufi=-1;

static char *oprstr[]=
{
  /* 0x00 */ "rev",
  /* 0x01 */ "lb",
  /* 0x02 */ "bsub",
  /* 0x03 */ "endp",
  /* 0x04 */ "diff",
  /* 0x05 */ "add",
  /* 0x06 */ "gcall",
  /* 0x07 */ "in",

  /* 0x08 */ "prod",
  /* 0x09 */ "gt",
  /* 0x0a */ "wsub",
  /* 0x0b */ "out",
  /* 0x0c */ "sub",
  /* 0x0d */ "startp",
  /* 0x0e */ "outbyte",
  /* 0x0f */ "outword",

  /* 0x10 */ "seterr",
  /* 0x11 */ (char*)0,
  /* 0x12 */ "resetch",
  /* 0x13 */ "csub0",
  /* 0x14 */ (char*)0,
  /* 0x15 */ "stopp",
  /* 0x16 */ "ladd",
  /* 0x17 */ "stlb",

  /* 0x18 */ "sthf",
  /* 0x19 */ "norm",
  /* 0x1a */ "ldiv",
  /* 0x1b */ (char*)0, /* ldpi */
  /* 0x1c */ "stlf",
  /* 0x1d */ "xdble",
  /* 0x1e */ "ldpri",
  /* 0x1f */ "rem",

  /* 0x20 */ "ret",
  /* 0x21 */ (char*)0, /* lend */
  /* 0x22 */ "ldtimer",
  /* 0x23 */ (char*)0,
  /* 0x24 */ (char*)0,
  /* 0x25 */ (char*)0,
  /* 0x26 */ (char*)0,
  /* 0x27 */ (char*)0,

  /* 0x28 */ (char*)0,
  /* 0x29 */ "testerr",
  /* 0x2a */ "testpranal",
  /* 0x2b */ "tin",
  /* 0x2c */ "div",
  /* 0x2d */ (char*)0,
  /* 0x2e */ "dist",
  /* 0x2f */ "disc",

  /* 0x30 */ "diss",
  /* 0x31 */ "lmul",
  /* 0x32 */ "not",
  /* 0x33 */ "xor",
  /* 0x34 */ "bcnt",
  /* 0x35 */ "lshr",
  /* 0x36 */ "lshl",
  /* 0x37 */ "lsum",

  /* 0x38 */ "lsub",
  /* 0x39 */ "runp",
  /* 0x3a */ "xword",
  /* 0x3b */ "sb",
  /* 0x3c */ "gajw",
  /* 0x3d */ "savel",
  /* 0x3e */ "saveh",
  /* 0x3f */ "wcnt",

  /* 0x40 */ "shr",
  /* 0x41 */ "shl",
  /* 0x42 */ "mint",
  /* 0x43 */ "alt",
  /* 0x44 */ "altwt",
  /* 0x45 */ "altend",
  /* 0x46 */ "and",
  /* 0x47 */ "enbt",

  /* 0x48 */ "enbc",
  /* 0x49 */ "enbs",
  /* 0x4a */ "move",
  /* 0x4b */ "or",
  /* 0x4c */ "csngl",
  /* 0x4d */ "ccnt1",
  /* 0x4e */ "talt",
  /* 0x4f */ "ldiff",

  /* 0x50 */ "sthb",
  /* 0x51 */ "taltwt",
  /* 0x52 */ "sum",
  /* 0x53 */ "mul",
  /* 0x54 */ "sttimer",
  /* 0x55 */ "stoperr",
  /* 0x56 */ "cword",
  /* 0x57 */ "clrhalterr",

  /* 0x58 */ "sethalterr",
  /* 0x59 */ "testhalterr",
  /* 0x5a */ "dup",
  /* 0x5b */ "move2dinit",
  /* 0x5c */ "move2dall",
  /* 0x5d */ "move2dnonzero",
  /* 0x5e */ "move2dzero",
  /* 0x5f */ (char*)0,

  /* 0x60 */ (char*)0,
  /* 0x61 */ (char*)0,
  /* 0x62 */ (char*)0,
  /* 0x63 */ "unpacksn",
  /* 0x64 */ (char*)0,
  /* 0x65 */ (char*)0,
  /* 0x66 */ (char*)0,
  /* 0x67 */ (char*)0,

  /* 0x68 */ (char*)0,
  /* 0x69 */ (char*)0,
  /* 0x6a */ (char*)0,
  /* 0x6b */ (char*)0,
  /* 0x6c */ "postnormsn",
  /* 0x6d */ "roundsn",
  /* 0x6e */ (char*)0,
  /* 0x6f */ (char*)0,

  /* 0x70 */ (char*)0,
  /* 0x71 */ "ldinf",
  /* 0x72 */ "fmul",
  /* 0x73 */ "cflerr",
  /* 0x74 */ "crcword",
  /* 0x75 */ "crcbyte",
  /* 0x76 */ "bitcnt",
  /* 0x77 */ "bitrevword",

  /* 0x78 */ "bitrevnbits",
  /* 0x79 */ (char*)0,
  /* 0x7a */ (char*)0,
  /* 0x7b */ (char*)0,
  /* 0x7c */ (char*)0,
  /* 0x7d */ (char*)0,
  /* 0x7e */ (char*)0,
  /* 0x7f */ (char*)0,

  /* 0x80 */ "fpsttest",
  /* 0x81 */ "wsubdb",
  /* 0x82 */ "fpldnldbi",
  /* 0x83 */ "fpchkerr",
  /* 0x84 */ "fpstnldb",
  /* 0x85 */ "fpldtest",
  /* 0x86 */ "fpldnlsni",
  /* 0x87 */ "fpadd",

  /* 0x88 */ "fpstnlsn",
  /* 0x89 */ "fpsub",
  /* 0x8a */ "fpldnldb",
  /* 0x8b */ "fpmul",
  /* 0x8c */ "fpdiv",
  /* 0x8d */ (char*)0,
  /* 0x8e */ "fpldnlsn",
  /* 0x8f */ "fpremfirst",

  /* 0x90 */ "fpremstep",
  /* 0x91 */ "fpnan",
  /* 0x92 */ "fpordered",
  /* 0x93 */ "fpnotfinite",
  /* 0x94 */ "fpgt",
  /* 0x95 */ "fpeq",
  /* 0x96 */ "fpi32tor32",
  /* 0x97 */ (char*)0,

  /* 0x98 */ "fpi32tor64",
  /* 0x99 */ (char*)0,
  /* 0x9a */ "fpb32tor64",
  /* 0x9b */ (char*)0,
  /* 0x9c */ "fptesterr",
  /* 0x9d */ "fprtoi32",
  /* 0x9e */ "fpstnli32",
  /* 0x9f */ "fpldzerosn",

  /* 0xa0 */ "fpldzerodb",
  /* 0xa1 */ "fpint",
  /* 0xa2 */ (char*)0,
  /* 0xa3 */ "fpdup",
  /* 0xa4 */ "fprev",
  /* 0xa5 */ (char*)0,
  /* 0xa6 */ "fpldnladddb",
  /* 0xa7 */ (char*)0,

  /* 0xa8 */ "fpldnlmuldb",
  /* 0xa9 */ (char*)0,
  /* 0xaa */ "fpldnladdsn",
  /* 0xab */ (char*)0, /* fpentry */
  /* 0xac */ "fpldnlmulsn",
};

Bool mksym=FALSE;

Symbol *hash[1403];
/*}}}  */

/*{{{  add_sym*/
static void add_sym(Reference reference, Address address, char *str)
{
  static unsigned int number=0;
  Symbol **run;

  run=&hash[address%(sizeof(hash)/sizeof(hash[0]))];
  while (*run!=(Symbol*)0 && ((*run)->address!=address || (*run)->reference!=reference)) run=(Symbol**)&((*run)->next);
  if (*run==(Symbol*)0)
  {
    *run=malloc(sizeof(Symbol));
    assert(*run!=(Symbol*)0);
    (*run)->next=(struct Symbol*)0;
    (*run)->reference=reference;
    (*run)->address=address;
    (*run)->number=++number;
  }
}
/*}}}  */
/*{{{  load_sym*/
static void load_sym(FILE *symfp)
{
  /*{{{  variables*/
  char ln[128],*end;
  /*}}}  */

  while (fgets(ln,sizeof(ln),symfp)!=(char*)0)
  {
    /*{{{  variables*/
    Address address;
    Reference reference;
    /*}}}  */

    address=number(ln,&end);
    if (ln==end)
    {
      fprintf(stderr,"dis: Invalid address at line: %s",ln);
      exit(1);
    }
    while (*end==' ') ++end;
    switch (*end)
    {
      case 'j': reference=JUMP; break;
      case 'c': reference=CALL; break;
      case 'l': reference=LDPI; break;
      case 'd': reference=DATA; break;
      default: fprintf(stderr,"dis: Invalid token at line: %s",ln); exit(1);
    }
    while (*end==' ') ++end;
    add_sym(reference,address,end);
  }
}
/*}}}  */
/*{{{  lookup_sym*/
Symbol *lookup_sym(Address address)
{
  /*{{{  variables*/
  Symbol *run;
  /*}}}  */

  run=hash[address%(sizeof(hash)/sizeof(hash[0]))];
  while (run!=(Symbol*)0 && run->address!=address) run=(Symbol*)run->next;
  return run;
}
/*}}}  */
/*{{{  label_sym*/
const char *label_sym(Reference reference, Address address)
{
  /*{{{  variables*/
  Symbol *run;
  static char s[32];
  /*}}}  */

  for (run=lookup_sym(address); run!=(Symbol*)0 && (run->address!=address || run->reference!=reference); run=(Symbol*)run->next);
  if (run!=(Symbol*)0) sprintf(s,"_%u",run->number);
  else sprintf(s,"0x%x",address);
  return s;  
}
/*}}}  */
/*{{{  dump_sym*/
static void dump_sym(void)
{
  /*{{{  variables*/
  Symbol *run;
  int i;
  /*}}}  */
  
  for (i=0; i<(sizeof(hash)/sizeof(hash[0])); ++i)
  {
    for (run=hash[i]; run!=(Symbol*)0; run=(Symbol*)run->next) switch (run->reference)
    {
      case JUMP: 
      {
        printf("0x%08x j _%u\n",run->address,run->number); 
        break;
      }
      case LDPI: 
      {
        printf("0x%08x l _%u\n",run->address,run->number); 
        break;
      }
      case CALL: 
      {
        printf("0x%08x c _%u\n",run->address,run->number); 
        break;
      }
      case DATA:
      {
        /* There is little use in giving a hint of a data region with length 0 */
        if (*label_sym(JUMP,run->address)=='0' && *label_sym(CALL,run->address)=='0') printf("0x%08x d\n",run->address);
        break;
      }
      default:   assert(0);
    }
  }
}
/*}}}  */
/*{{{  printline*/
static void printline(int i)
{
  unsigned long value;

  if (mksym) return;
  if (1) printf("0x%08lx ",putbuf[i].pc); else printf("      ");
  printf("%s",putbuf[i].s);
  if (putbuf[i].opvalid!=NOT)
  {
    if (putbuf[i].instr==PREFIX) value=putbuf[i].nibble; else value=putbuf[i].op;
    switch (putbuf[i].reference)
    {
      case NONE: 
      {
        switch (putbuf[i].opvalid)
	{
	  case UNSIGNED: printf(" 0x%lx",value); break;
	  case SIGNED:   printf(" %ld",value); break;
	  default: assert(0);
	}
	break;
      }
      case JUMP:
      case CALL:
      case LDPI: printf(" %s",label_sym(putbuf[i].reference,putbuf[i].op)); break;
    }  
  }
  putchar('\n');
}
/*}}}  */
/*{{{  putflush*/
static void putflush(void)
{
  int i;

  for (i=0; i<=putbufi; ++i) printline(i);
  putbufi=-1;
}
/*}}}  */
/*{{{  delprefix*/
static void delprefix(void)
{
  int i=putbufi;

  while (i>0 && putbuf[i-1].instr==PREFIX) --i;
  if (i!=putbufi)
  {
    putbuf[putbufi].pc=putbuf[i].pc;
    memcpy(&putbuf[i],&putbuf[putbufi],sizeof(putbuf[0]));
    putbufi=i;
  }
}
/*}}}  */
/*{{{  put*/
static void put(Address pc, char *s, Reference reference, Instr instr, int nibble, long op, Opvalid opvalid)
{
  if (putbufi==sizeof(putbuf)/sizeof(putbuf[0])-1)
  {
    printline(0);
    memmove(&putbuf[0],&putbuf[1],(sizeof(putbuf)/sizeof(putbuf[0])-1)*sizeof(putbuf[0]));
  }
  else ++putbufi;
  putbuf[putbufi].pc=pc;
  putbuf[putbufi].instr=instr;
  strcpy(putbuf[putbufi].s,s);
  putbuf[putbufi].nibble=nibble;
  putbuf[putbufi].op=op;
  putbuf[putbufi].opvalid=opvalid;
  putbuf[putbufi].reference=reference;
  if (instr!=PREFIX) delprefix();
  if (instr==DONTCARE) putflush();
}
/*}}}  */
/*{{{  lastinstr*/
static Instr lastinstr(void)
{
  if (putbufi<0) return DONTCARE;
  else return putbuf[putbufi].instr;
}
/*}}}  */

/*{{{  main*/
int main(int argc, char *argv[], char *env[])
{
  /*{{{  variables*/
  long op=0;
  Address pc=0x80000070;
  long textlen;
  int c;
  int instr=0;
  FILE *f;
  Bool usage=FALSE, boot=FALSE, executable=FALSE;
  Bool isdata=FALSE;
  /*}}}  */

  /*{{{  set locale*/
  assert(sizeof(oprstr)/sizeof(oprstr[0])==0xac+1);
  setlocale(LC_ALL,"");
  catd=catopen("dis",0);
  /*}}}  */
  /*{{{  parse arguments*/
  while ((c=getopt(argc,argv,"ea:bhSs:"))!=EOF)
  switch (c)
  {
    /*{{{  a address*/
    case 'a':
    {
      char *end;
    
      pc=number(optarg,&end);
      if (end==optarg)
      {
        fprintf(stderr,catgets(catd,1,1,"dis: Invalid adress %s\n"),optarg);
        usage=TRUE;
      }
      executable=FALSE;
      break;
    }
    /*}}}  */
    /*{{{  b*/
    case 'b': boot=TRUE; executable=FALSE; break;
    /*}}}  */
    /*{{{  e*/
    case 'e':
    {
      executable=TRUE;
      pc=0;
      boot=FALSE;
      break;
    }
    /*}}}  */
    /*{{{  S*/
    case 'S': mksym=TRUE; break;  
    /*}}}  */
    /*{{{  s file*/
    case 's': 
    {
      FILE *symfp;
    
      if ((symfp=fopen(optarg,"r"))==(FILE*)0)
      {
        fprintf(stderr,"dis: Can not open symbol table file %s: %s\n",optarg,strerror(errno));
        exit(1);
      }
      load_sym(symfp);
      fclose(symfp);
      break;
    }
    /*}}}  */
    /*{{{  h, ?*/
    case 'h':
    case '?':
    {
      usage=TRUE;
      break;
    }
    /*}}}  */
  }
  /*{{{  open input file*/
  if (optind>=argc) usage=TRUE;
  else if ((f=fopen(argv[optind],"r"))==(FILE*)0)
  {
    fprintf(stderr,catgets(catd,1,2,"dis: can't open %s\n"), argv[optind]);
    usage=TRUE;
  }
  /*}}}  */
  /*}}}  */
  /*{{{  check usage*/
  if (usage)
  {
    fprintf(stderr,catgets(catd,1,3,"Usage: dis [[-e]|[-a hexaddr] [-b]] [-S] [-s filename] filename\n"));
    exit(1);
  }
  /*}}}  */
  /*{{{  output a.out header, if wanted*/
  if (executable)
  {
    /*{{{  variables*/
    struct exec a_out;
    /*}}}  */
  
    pc=0;
    /*{{{  read header*/
    fread(&a_out,A_MINHDR,1,f);
    if (a_out.a_hdrlen>A_MINHDR) fread(&(a_out.a_trsize),a_out.a_hdrlen-A_MINHDR,1,f);
    textlen=a_out.a_text;
    /*}}}  */
    if (!mksym)
    {
      /*{{{  magic*/
      printf("; Magic number  [%02x][%02x]   = %s\n",a_out.a_magic[0],a_out.a_magic[1],BADMAG(a_out) ? "invalid" : "valid");
      /*}}}  */
      /*{{{  cpu*/
      printf("; CPU           [%02x]       = ",a_out.a_cpu);
      switch (a_out.a_cpu)
      {
        case A_I8086: printf("Intel i8086 compatible\n"); break;
        case A_M68K: printf("Motorola m68000\n"); break;
        case A_NS16K: printf("NSC 16032\n"); break;
        case A_I80386: printf("Intel i80386 compatible\n"); break;
        case A_SPARC: printf("Sun SPARC\n"); break;
        case A_IMSTx: printf("INMOS Transputer\n"); break;
        default: printf("unknown cpu\n"); break;
      }
      /*}}}  */
      /*{{{  flags*/
      printf("; Flags         [%02x]       = ",a_out.a_flags);
      printf("%s zero page",a_out.a_flags&A_UZP ? "unmapped" : "mapped");
      if (a_out.a_flags&A_EXEC) printf(", executable");
      printf(", %s",a_out.a_flags&A_SEP ? "seperate I/D" : "common I/D");
      if (a_out.a_flags&A_PURE) printf(", pure text");
      if (a_out.a_flags&A_TOVLY) printf(", text overlay");
      printf("\n");
      /*}}}  */
      /*{{{  unused*/
      printf("; Unused byte   [%02x]",a_out.a_unused);
      if (a_out.a_cpu==A_IMSTx && a_out.a_unused & A_TP_MBS)
      {
        int comma=0;
      
        printf("       = needs ");
        if (a_out.a_unused & A_TP_FPU) 
        {
          printf("FPU");
          comma=1;
        }
        if (a_out.a_unused & A_TP_FPSUPPORT)
        {
          printf("%sFP support", comma ? ", " : "");
          comma=1;
        }
        if (a_out.a_unused & A_TP_EXTRAINST)
        {
          printf("%sdup etc.", comma ? ", " : "");
          comma=1;
        }
        if (a_out.a_unused & A_TP_DBGSUPPORT)
        {
          printf("%spop etc.", comma ? ", " : "");
          comma=1;
        }
      }
      printf("\n");
      /*}}}  */
      /*{{{  version*/
      printf("; Version stamp [%04x]\n",a_out.a_version);
      /*}}}  */
      /*{{{  text*/
      printf("; Text          [%08lx] = %ld bytes\n",a_out.a_text,a_out.a_text);
      /*}}}  */
      /*{{{  data*/
      printf("; Data          [%08lx] = %ld bytes\n",a_out.a_data,a_out.a_data);
      /*}}}  */
      /*{{{  bss*/
      printf("; BSS           [%08lx] = %ld bytes\n",a_out.a_bss,a_out.a_bss);
      /*}}}  */
      /*{{{  total*/
      printf("; Total         [%08lx] = %ld bytes\n",a_out.a_total,a_out.a_total);
      /*}}}  */
      /*{{{  stack/heap*/
      printf("; Stack/Heap               = %ld bytes\n",a_out.a_total-a_out.a_data-a_out.a_bss-(a_out.a_flags&A_SEP ? 0 : a_out.a_text));
      /*}}}  */
      /*{{{  entry*/
      printf("; Entry         [%08lx] = address %ld\n",a_out.a_entry,a_out.a_entry);
      /*}}}  */
      /*{{{  symbols*/
      printf("; Symbols       [%08lx] = %ld bytes = %ld entries\n",a_out.a_syms,a_out.a_syms,a_out.a_syms/sizeof(struct nlist));
      /*}}}  */
      if (a_out.a_hdrlen>A_MINHDR)
      {
        /*{{{  text relocation*/
        printf("; Text reloc    [%08lx] = %ld bytes = %ld entries\n",a_out.a_trsize,a_out.a_trsize,a_out.a_trsize/sizeof(struct reloc));
        /*}}}  */
      }
    }
  }
  /*}}}  */
  /*{{{  output boot length, if wanted*/
  if (boot)
  {
    if (mksym) fgetc(f); else put(pc-1,"d8",NONE,DONTCARE,0,fgetc(f),TRUE);
  }
  /*}}}  */
  /*{{{  disassemble text*/
  if (executable && !mksym) printf("; text segment\n");
  while ((c=fgetc(f))!=EOF && (!executable || textlen>0))
  {
    /*{{{  variables*/
    int high=((unsigned int)c)&0xf0;
    long low=((unsigned int)c)&0x0f;
    Symbol *run;
    Bool iscode=FALSE;
    /*}}}  */
  
    /*{{{  process any references to the current pc*/
    for (run=lookup_sym(pc); run!=(Symbol*)0; run=(Symbol*)run->next)
    {
      switch (run->reference)
      {
        case JUMP:
        case CALL: iscode=TRUE;
        case LDPI: putflush(); if (!mksym) printf("_%u\n",run->number); break;
        case DATA: isdata=TRUE; break;
      }
    }
    if (iscode) isdata=FALSE;
    /*}}}  */
    --textlen;
    if (isdata)
    /*{{{  output d8*/
    {
        put(pc,"d8",NONE,DONTCARE,0,c,UNSIGNED);
        putflush();
    }
    /*}}}  */
    else
    /*{{{  disassemble*/
    {
      if (lastinstr()==PREFIX) op=(op<<4)|low; else op=low;
      switch (high)
      {
        /*{{{  j*/
        case 0x00: 
        {
          put(pc,"j",JUMP,DONTCARE,0,pc+op+1,UNSIGNED);
          if (mksym) 
          {
            add_sym(JUMP,pc+op+1,(char*)0); 
            add_sym(DATA,pc+1,(char*)0);
          }
          break;
        }
        /*}}}  */
        /*{{{  ldlp*/
        case 0x10: put(pc,"ldlp",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  pfix*/
        case 0x20: put(pc,"pfix",NONE,PREFIX,low,op,UNSIGNED); break;
        /*}}}  */
        /*{{{  ldnl*/
        case 0x30: put(pc,"ldnl",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  ldc*/
        case 0x40: put(pc,"ldc",NONE,LDC,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  ldnlp*/
        case 0x50: put(pc,"ldnlp",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  nfix*/
        case 0x60: op=~op; put(pc,"nfix",NONE,PREFIX,low,op,UNSIGNED); break;
        /*}}}  */
        /*{{{  ldl*/
        case 0x70: put(pc,"ldl",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  adc*/
        case 0x80: put(pc,"adc",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  call*/
        case 0x90: put(pc,"call",CALL,DONTCARE,0,pc+op+1,UNSIGNED); if (mksym) add_sym(CALL,pc+op+1,(char*)0); break;
        /*}}}  */
        /*{{{  cj*/
        case 0xa0: put(pc,"cj",JUMP,DONTCARE,0,pc+op+1,UNSIGNED); if (mksym) add_sym(JUMP,pc+op+1,(char*)0); break;
        /*}}}  */
        /*{{{  ajw*/
        case 0xb0: put(pc,"ajw",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  eqc*/
        case 0xc0: put(pc,"eqc",NONE,DONTCARE,0,op,UNSIGNED); break;
        /*}}}  */
        /*{{{  stl*/
        case 0xd0: put(pc,"stl",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  stnl*/
        case 0xe0: put(pc,"stnl",NONE,DONTCARE,0,op,SIGNED); break;
        /*}}}  */
        /*{{{  opr*/
        case 0xf0:
        {
          switch (op)
          {
            case 0x1b:
            put(pc,"ldpi",NONE,SPECIAL,0,0,NOT);
            if (putbufi-1>=0 && putbuf[putbufi-1].instr==LDC)
            {
              --putbufi;
              strcpy(putbuf[putbufi].s,"ldpi");
              putbuf[putbufi].op=pc+putbuf[putbufi].op+1;
              putbuf[putbufi].opvalid=UNSIGNED;
              putbuf[putbufi].instr=DONTCARE;
              putbuf[putbufi].reference=LDPI;
              if (mksym) add_sym(LDPI,putbuf[putbufi].op,(char*)0);
              putflush();
            }
            break;
            case 0x21:
            put(pc,"lend",NONE,SPECIAL,0,0,NOT);
            if (putbufi-1>=0 && putbuf[putbufi-1].instr==LDC)
            {
              --putbufi;
              strcpy(putbuf[putbufi].s,"lend");
              putbuf[putbufi].op=pc-putbuf[putbufi].op+1;
              putbuf[putbufi].opvalid=UNSIGNED;
              putbuf[putbufi].instr=DONTCARE;
              putbuf[putbufi].reference=JUMP;
              if (mksym) add_sym(JUMP,putbuf[putbufi].op,(char*)0);
              putflush();
            }
            break;
            case 0xab:
            put(pc,"fpentry",NONE,SPECIAL,0,0,NOT);
            if (putbufi-1>=0 && putbuf[putbufi-1].instr==LDC)
            {
              char *s;
          
              switch (putbuf[putbufi-1].op)
              {
                case 0x01: s="fpusqrtfirst"; goto eek;
                case 0x02: s="fpusqrtstep"; goto eek;
                case 0x03: s="fpufpusqrtlast"; goto eek;
                case 0x04: s="fpurp"; goto eek;
                case 0x05: s="fpurm"; goto eek;
                case 0x06: s="fpurz"; goto eek;
                case 0x07: s="fpur32tor64"; goto eek;
                case 0x08: s="fpur64tor32"; goto eek;
                case 0x09: s="fpuexpdec32"; goto eek;
                case 0x0a: s="fpuexpinc32"; goto eek;
                case 0x0b: s="fpuabs"; goto eek;
                case 0x0d: s="fpunoround"; goto eek;
                case 0x0e: s="fpuchki32"; goto eek;
                case 0x0f: s="fpuchki64"; goto eek;
                case 0x11: s="fpudivby2"; goto eek;
                case 0x12: s="fpumulby2"; goto eek;
                case 0x22: s="fpurn"; goto eek;
                case 0x23: s="fpuseterr"; goto eek;
                case 0x2c: s="fpusave"; goto eek;
                case 0x30: s="fpurestore"; goto eek;
                case 0x9c: s="fpuclrerr"; goto eek;
              }
              break;
              eek:
              --putbufi;
              strcpy(putbuf[putbufi].s,s);
              putbuf[putbufi].opvalid=NOT;
              putbuf[putbufi].instr=DONTCARE;
              putbuf[putbufi].reference=NONE;
              putflush();
            }
            break;
            default:
            {
              if (mksym && op==0x20) add_sym(DATA,pc+1,(char*)0);
                if (op>0xac || oprstr[op]==(char*)0) put(pc,"opr",NONE,DONTCARE,0,op,UNSIGNED);
                else put(pc,oprstr[op],NONE,DONTCARE,0,0,NOT);
              break;
            }
          }
          break;
        }
        /*}}}  */
      }
      instr=high;
    }
    /*}}}  */
    ++pc;
  }
  if (mksym) dump_sym(); else putflush();
  /*}}}  */
  /*{{{  close file*/
  fclose(f);
  fflush(stdout);
  /*}}}  */
  return 0;
}
/*}}}  */
