/*
 *  $Id: smpshort.c,v 1.7 2000/09/08 06:54:18 loic Exp $
 *
 *  (C) 1995 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

#include "mpid.h"
#include "mpiddev.h"
#include "mpimem.h"
#include "reqalloc.h"
#include "gm_stbar.h"
#include "smpi.h"

/* Prototype definitions */
int MPID_SMP_Eagerb_send_short ANSI_ARGS(( void *, int, int, int, int, int, 
					  MPID_Msgrep_t ));
int MPID_SMP_Eagerb_isend_short ANSI_ARGS(( void *, int, int, int, int, int, 
					   MPID_Msgrep_t, MPIR_SHANDLE * ));
int MPID_SMP_Eagerb_recv_short ANSI_ARGS(( MPIR_RHANDLE *, int, void * ));
int MPID_SMP_Eagerb_save_short ANSI_ARGS(( MPIR_RHANDLE *, int, void *));
int MPID_SMP_Eagerb_unxrecv_start_short ANSI_ARGS(( MPIR_RHANDLE *, void * ));
void MPID_SMP_Eagerb_short_delete ANSI_ARGS(( MPID_Protocol * ));
int MPID_SMP_Cancel_send ANSI_ARGS(( int, int, int, int, MPID_Aint ));
int MPID_SMP_Cancel_recv ANSI_ARGS(( void *, int, int *, MPID_Aint * ));

/*
 * Definitions of the actual functions
 */

int MPID_SMP_Eagerb_send_short( buf, len, src_lrank, tag, context_id, dest,
			       msgrep )
void          *buf;
int           len, tag, context_id, src_lrank, dest;
MPID_Msgrep_t msgrep;
{
  int destination = smpi.local_nodes[dest];

  DEBUG_PRINT_MSG("S starting MPID_SMP_Eagerb_send_short");
  /* Flow control : is there enough place on the shared memory area */
  if (!smpi.send_fifo_head && smpi_able_to_send(destination, len+sizeof(SMPI_PKT_HEAD_T)))
    /* let's go to send it ! */
    smpi_post_send_bufferinplace(buf,len,src_lrank,tag, context_id, destination, NULL);
  else {
    /* not enough place, we will send it later */
    SMPI_PKT_SHORT_T * pkt_p = (SMPI_PKT_SHORT_T *)gmpi_xmalloc(len + sizeof(SMPI_PKT_HEAD_T),
								"MPI/smp_plug:send_short");
    smpi_assert(pkt_p);

    /* These references are ordered to match the order they appear in the 
       structure */
    pkt_p->mode	      = MPID_PKT_SHORT;
    pkt_p->context_id = context_id;
    pkt_p->lrank      = src_lrank;
    pkt_p->tag	      = tag;
    pkt_p->len	      = len;
 
    smpi_assert(dest < MPID_MyWorldSize);
    smpi_assert(len <= SMPI_PKT_MAX_LOCALDATA_SIZE);
    smpi_assert(destination > -1);
    smpi_assert(destination < smpi.num_local_nodes);


    DEBUG_PRINT_SMP_SEND_PKT("S Sending in the send_queue", pkt_p);
    
    if (len > 0) {
      MEMCPY( pkt_p->buffer, buf, len);
      DEBUG_PRINT_PKT_DATA("S Getting data from buf", pkt_p);
    }
    
    DEBUG_PRINT_MSG("S Sending message in the send_queue");
    smpi_queue_send(pkt_p, 0, (len + sizeof(SMPI_PKT_HEAD_T)), destination, 0);
    DEBUG_PRINT_MSG("S Sent message in the send_queue");
  }
    
  return MPI_SUCCESS;
}

int MPID_SMP_Eagerb_isend_short( buf, len, src_lrank, tag, context_id, dest,
			 msgrep, shandle )
void          *buf;
int           len, tag, context_id, src_lrank, dest;
MPID_Msgrep_t msgrep;
MPIR_SHANDLE *shandle;
{
  int mpi_errno;
  int destination = smpi.local_nodes[dest];
  
  shandle->is_complete = 1;
  /* Store partners rank in request in case message is cancelled */
  shandle->partner     = dest;
  
  DEBUG_PRINT_MSG("S starting MPID_SMP_Eagerb_isend_short");
  /* Flow control : is there enough place on the shared memory area */
  if (!smpi.send_fifo_head && smpi_able_to_send(destination, len+sizeof(SMPI_PKT_HEAD_T)))
    /* let's go to send it ! */
    smpi_post_send_bufferinplace(buf,len,src_lrank,tag, context_id, destination, shandle);
  else {
    /* not enough place, we will send it later */
    SMPI_PKT_SHORT_T * pkt_p = (SMPI_PKT_SHORT_T *)gmpi_xmalloc(len + sizeof(SMPI_PKT_HEAD_T),
								"MPI/smp_plug:isend_short");
    smpi_assert(pkt_p);
    
    /* These references are ordered to match the order they appear in the 
       structure */
    pkt_p->mode	      = MPID_PKT_SHORT;
    pkt_p->context_id = context_id;
    pkt_p->lrank      = src_lrank;
    pkt_p->tag	      = tag;
    pkt_p->len	      = len;
 
    smpi_assert(dest < MPID_MyWorldSize);
    smpi_assert(len <= SMPI_PKT_MAX_LOCALDATA_SIZE);
    smpi_assert(destination > -1);
    smpi_assert(destination < smpi.num_local_nodes);
     
    DEBUG_PRINT_SMP_SEND_PKT("S Sending in the send_queue", pkt_p);

    if (len > 0) {
      MEMCPY( pkt_p->buffer, buf, len);
      DEBUG_PRINT_PKT_DATA("S Getting data from buf", pkt_p);
    }
    
    DEBUG_PRINT_MSG("S Sending message in the send_queue");
    smpi_queue_send(pkt_p, shandle, (len + sizeof(SMPI_PKT_HEAD_T)), destination, 0);
    DEBUG_PRINT_MSG("S Sent message in the send_queue");
  }
  
  /* Instead of this, the calling code should test from not-complete,
     and set finish if needed */
  mpi_errno = shandle->s.MPI_ERROR;
  if (shandle->finish)
    (shandle->finish)( shandle );

  return mpi_errno;
}

int MPID_SMP_Eagerb_recv_short( rhandle, from_grank, in_pkt )
MPIR_RHANDLE *rhandle;
int          from_grank;
void         *in_pkt;
{
    SMPI_PKT_SHORT_T *pkt = (SMPI_PKT_SHORT_T *)in_pkt;
    int          msglen;
    int          err = MPI_SUCCESS;
    int my_id = smpi.my_local_id;
    
    DEBUG_PRINT_MSG("R Starting Eagerb_recv_short");
    msglen		  = pkt->len;
    rhandle->s.MPI_TAG	  = pkt->tag;
    rhandle->s.MPI_SOURCE = pkt->lrank;
    MPID_CHK_MSGLEN(rhandle,msglen,err);
    
    if (msglen > 0)
      MEMCPY( rhandle->buf, pkt->buffer, msglen ); 
  
    rhandle->s.count	  = msglen;
    rhandle->s.MPI_ERROR  = err;
    rhandle->send_id = 0;
    if (rhandle->finish) {
      (rhandle->finish)( rhandle );
    }
    
    /* Flow control */
    smpi_complete_recv(from_grank, my_id, (pkt->len + sizeof(SMPI_PKT_HEAD_T)));
   
    rhandle->is_complete = 1;
    return err;
}

/* 
 * This routine is called when it is time to receive an unexpected
 * message
 */
int MPID_SMP_Eagerb_unxrecv_start_short( rhandle, in_runex )
MPIR_RHANDLE *rhandle;
void         *in_runex;
{
    MPIR_RHANDLE *runex = (MPIR_RHANDLE *)in_runex;
    int          msglen, err = 0;

    msglen = runex->s.count;
    DEBUG_PRINT_MSG("R Starting Eagerb_unxrecv_start_short");
    MPID_CHK_MSGLEN(rhandle,msglen,err);
    /* Copy the data from the local area and free that area */
    if (runex->s.count > 0) {
	MEMCPY( rhandle->buf, runex->start, msglen );
	FREE( runex->start );
    }
    rhandle->s		 = runex->s;
    rhandle->s.count     = msglen;
    rhandle->s.MPI_ERROR = err;
    rhandle->wait	 = 0;
    rhandle->test	 = 0;
    rhandle->push	 = 0;
    rhandle->is_complete = 1;
    if (rhandle->finish) 
	(rhandle->finish)( rhandle );
    MPID_RecvFree( runex );

    return err;
}

/* Save an unexpected message in rhandle */
int MPID_SMP_Eagerb_save_short( rhandle, from, in_pkt )
MPIR_RHANDLE *rhandle;
int          from;
void         *in_pkt;
{
    int my_id = smpi.my_local_id;
    SMPI_PKT_SHORT_T   *pkt = (SMPI_PKT_SHORT_T *)in_pkt;

    DEBUG_PRINT_MSG("R Starting Eagerb_save_short");
    rhandle->s.MPI_TAG	  = pkt->tag;
    rhandle->s.MPI_SOURCE = pkt->lrank;
    rhandle->s.MPI_ERROR  = 0;
    rhandle->s.count      = pkt->len;
    rhandle->is_complete  = 1;
    rhandle->send_id = 0;
    if (pkt->len > 0) {
	rhandle->start	  = (void *)MALLOC( pkt->len );
	if (!rhandle->start) {
	    rhandle->s.MPI_ERROR = MPI_ERR_INTERN;
	    return 1;
	}
	MEMCPY( rhandle->start, pkt->buffer, pkt->len );
    }
    rhandle->push = MPID_SMP_Eagerb_unxrecv_start_short;
    
    /* flow control */
    smpi_complete_recv(from, my_id, (pkt->len + sizeof(SMPI_PKT_HEAD_T)));
			
    return 0;
}

void MPID_SMP_Eagerb_short_delete( p )
MPID_Protocol *p;
{
    FREE( p );
}


int MPID_SMP_Cancel_send( src_lrank, dest, cancel, anti_send_ok, send_id )
int           src_lrank, dest, cancel, anti_send_ok;
MPID_Aint     send_id;
{

  volatile void * ptr_volatile;
  void * ptr;
  SMPI_PKT_ANTI_SEND_T * pkt;
  int destination = smpi.local_nodes[dest];
  int my_id = smpi.my_local_id;
  
  smpi_assert(dest > -1);
  smpi_assert(dest < MPID_MyWorldSize);
  smpi_assert(destination > -1);
  smpi_assert(destination < smpi.num_local_nodes);

  DEBUG_PRINT_MSG("S starting MPID_SMP_Cancel_send");
  /* Flow control : is there enough place on the shared memory area */
  if (!smpi.send_fifo_head && smpi_able_to_send(destination,sizeof(SMPI_PKT_ANTI_SEND_T))) {
    /* let's go to send it ! */
    gmpi_lock();
    
    /* build the packet */
    ptr_volatile = (void *)((&smpi_shmem->pool)+ SMPI_NEXT(my_id,destination));
    ptr = (void *)ptr_volatile;
    pkt = (SMPI_PKT_ANTI_SEND_T *)ptr;
    if (anti_send_ok)
      pkt->mode  = MPID_PKT_ANTI_SEND_OK;
    else
      pkt->mode  = MPID_PKT_ANTI_SEND;
    pkt->lrank = src_lrank;
    pkt->cancel  = cancel;
    pkt->send_id = send_id;
    
    if (anti_send_ok) {
      DEBUG_PRINT_MSG("S Sending anti_send_ok message\n");
    } else {
      DEBUG_PRINT_MSG("S Sending anti_send message\n");
    } 
    
    /* update flow control */
    smpi_complete_send(my_id, destination, sizeof(SMPI_PKT_ANTI_SEND_T));

    gmpi_unlock();
  } else {
    /* not enough place, we will send it later */
    SMPI_PKT_ANTI_SEND_T * pkt_p = (SMPI_PKT_ANTI_SEND_T *)gmpi_xmalloc(sizeof(SMPI_PKT_ANTI_SEND_T),
									"MPI/smp_plug:cancel_send");
    smpi_assert(pkt_p);
    
    if (anti_send_ok)
      pkt_p->mode  = MPID_PKT_ANTI_SEND_OK;
    else
      pkt_p->mode  = MPID_PKT_ANTI_SEND;
    pkt_p->lrank = src_lrank;
    pkt_p->cancel  = cancel;
    pkt_p->send_id = send_id;
    
    if (anti_send_ok) {
      DEBUG_PRINT_MSG("S Sending anti_send_ok message in the send_queue\n");
    } else {
      DEBUG_PRINT_MSG("S Sending anti_send message in the send_queue\n");
    } 
    smpi_queue_send(pkt_p, 0, sizeof(SMPI_PKT_ANTI_SEND_T), destination, 0);
    DEBUG_PRINT_MSG("S Sent message in the send_queue");
  }

  return MPI_SUCCESS;
}


int MPID_SMP_Cancel_recv( in_pkt, from, cancel_ptr, send_id_ptr )
void *      in_pkt;
int         from;
int *       cancel_ptr;
MPID_Aint * send_id_ptr;
{
  SMPI_PKT_ANTI_SEND_T *pkt = (SMPI_PKT_ANTI_SEND_T *)in_pkt;
  gmpi_lock();
  
  *send_id_ptr = pkt->send_id;
  *cancel_ptr = pkt->cancel;

  gmpi_unlock();
  return MPI_SUCCESS;
}


MPID_Protocol *MPID_SMP_Short_setup(void)
{
    MPID_Protocol *p;

    p = (MPID_Protocol *) MALLOC( sizeof(MPID_Protocol) );
    if (!p) return 0;
    p->send	   = MPID_SMP_Eagerb_send_short;
    p->recv	   = MPID_SMP_Eagerb_recv_short;
    p->isend	   = MPID_SMP_Eagerb_isend_short;
    p->wait_send   = 0;
    p->push_send   = 0;
    p->cancel_send = MPID_SMP_Cancel_send;
    p->irecv	   = 0;
    p->wait_recv   = 0;
    p->push_recv   = 0;
    p->cancel_recv = MPID_SMP_Cancel_recv;
    p->do_ack      = 0;
    p->unex        = MPID_SMP_Eagerb_save_short;
    p->delete      = MPID_SMP_Eagerb_short_delete;

    return p;
}
