/*
 *  $Id: chshort.c,v 1.12 2000/04/19 07:10:54 patrick Exp $
 *
 *  (C) 1995 by Argonne National Laboratory and Mississipi State University.
 *      All rights reserved.  See COPYRIGHT in top-level directory.
 */

#include "mpid.h"
#include "mpiddev.h"
#include "mpimem.h"
#include "reqalloc.h"
#include "flow.h"
#include "chpackflow.h"
#include "gmpi.h"

/* Prototype definitions */
int MPID_CH_Eagerb_send_short ANSI_ARGS(( void *, int, int, int, int, int, 
					  MPID_Msgrep_t ));
int MPID_CH_Eagerb_isend_short ANSI_ARGS(( void *, int, int, int, int, int, 
					   MPID_Msgrep_t, MPIR_SHANDLE * ));
int MPID_CH_Eagerb_recv_short ANSI_ARGS(( MPIR_RHANDLE *, int, void * ));
int MPID_CH_Eagerb_save_short ANSI_ARGS(( MPIR_RHANDLE *, int, void *));
int MPID_CH_Eagerb_unxrecv_start_short ANSI_ARGS(( MPIR_RHANDLE *, void * ));
void MPID_CH_Eagerb_short_delete ANSI_ARGS(( MPID_Protocol * ));
int MPID_CH_Cancel_send ANSI_ARGS(( int, int, int, int, MPID_Aint ));
int MPID_CH_Cancel_recv ANSI_ARGS(( void *, int, int *, MPID_Aint * ));

/*
 * Definitions of the actual functions
 */

static int max_queued = 0;

int MPID_CH_Eagerb_send_short( buf, len, src_lrank, tag, context_id, dest,
			       msgrep )
void          *buf;
int           len, tag, context_id, src_lrank, dest;
MPID_Msgrep_t msgrep;
{
    int pkt_len;
#define pkt (*pkt_p)
    MPID_PKT_SHORT_T *pkt_p = gmpi_ctl_alloc(sizeof(*pkt_p));
    char* pktp;

    DEBUG_PRINT_MSG("S starting MPID_CH_Eagerb_send_short");

    gmpi_lock();
#ifdef MPID_PACK_CONTROL

    while (!MPID_PACKET_CHECK_OK(dest)) {  /* begin while !ok loop */
	/* Wait for a protocol ACK packet */

#ifdef MPID_DEBUG_ALL

	if (MPID_DebugFlag || MPID_DebugFlow) {
		FPRINTF(MPID_DEBUG_FILE,
   "[%d] S Waiting for a protocol ACK packet (in eagerb_send_short) from %d\n",
			MPID_MyWorldRank, dest);
	}

#endif
	MPID_DeviceCheck( MPID_BLOCKING );
    }  /* end while !ok loop */

    MPID_PACKET_ADD_SENT(MPID_MyWorldRank, dest)
#endif

    /* These references are ordered to match the order they appear in the 
       structure */
    pkt_len        = sizeof(MPID_PKT_HEAD_T) + sizeof(MPID_Aint);
    pkt.mode	   = MPID_PKT_SHORT;
    pkt.context_id = context_id;
    pkt.lrank	   = src_lrank;
    pkt.to         = dest;
    pkt.seqnum     = len + pkt_len; 
    pkt.src        = MPID_MyWorldRank;
    pkt.tag	   = tag;
    pkt.len	   = len;
    MPID_DO_HETERO(pkt.msgrep = (int)msgrep);

    DEBUG_PRINT_SEND_PKT("S Sending",&pkt);
    MPID_PKT_PACK( &pkt, pkt_len, dest );

    if (len > 0) {
	MEMCPY( pkt.buffer, buf, len );
	DEBUG_PRINT_PKT_DATA("S Getting data from buf",&pkt);
    }
    /* Always use a blocking send for short messages.
       (May fail with systems that do not provide adequate
       buffering.  These systems should switch to non-blocking sends)
     */

    DEBUG_PRINT_SEND_PKT("S Sending message in a single packet",&pkt);

#if 0
	pktp = (char *)&pkt;
	fprintf( stdout, "gmpi: Packet to send dump: 0x%x%x%x%x%x%x%x%x\n", pktp[0],pktp[1],pktp[2],pktp[3],pktp[4],pktp[5],pktp[6],pktp[7]);
#endif

    /* In case the message is marked as non-blocking, indicate that we don't
       need to wait on it.  We may also want to use nonblocking operations
       to send the envelopes.... */
    MPID_DRAIN_INCOMING_FOR_TINY(1);
    MPID_SendControlBlock( pkt_p, len + pkt_len, dest );
    DEBUG_PRINT_MSG("S Sent message in a single packet");
#undef pkt

    /* added this to prevent running out of dma-able memory when sending lots */
    while (gmpi.num_sends_queued > 100)
        MPID_DRAIN_INCOMING;

/*
    if (gmpi.num_sends_queued > max_queued) {
	max_queued = gmpi.num_sends_queued;
    }
*/

    gmpi_unlock();

    return MPI_SUCCESS;
}


int MPID_CH_Eagerb_isend_short( buf, len, src_lrank, tag, context_id, dest,
			 msgrep, shandle )
void          *buf;
int           len, tag, context_id, src_lrank, dest;
MPID_Msgrep_t msgrep;
MPIR_SHANDLE *shandle;
{
    int pkt_len;
#define pkt (*pkt_p)
    MPID_PKT_SHORT_T *pkt_p = gmpi_ctl_alloc(sizeof(*pkt_p));
    char* pktp;

    gmpi_lock();
    DEBUG_PRINT_MSG("S Starting Eagerb_isend_short");
#ifdef MPID_PACK_CONTROL
    while (!MPID_PACKET_CHECK_OK(dest)) {  /* begin while !ok loop */
	/* Wait for a protocol ACK packet */
#ifdef MPID_DEBUG_ALL
	if (MPID_DebugFlag || MPID_DebugFlow) {
		FPRINTF(MPID_DEBUG_FILE,
   "[%d] S Waiting for a protocol ACK packet (in eagerb_send_short) from %d\n",
			MPID_MyWorldRank, dest);
	}
#endif
	MPID_DeviceCheck( MPID_BLOCKING );
    }  /* end while !ok loop */

    MPID_PACKET_ADD_SENT(MPID_MyWorldRank, dest)
#endif

    /* These references are ordered to match the order they appear in the 
       structure */
    pkt_len        = sizeof(MPID_PKT_HEAD_T) + sizeof(MPID_Aint);
    pkt.mode	   = MPID_PKT_SHORT;
    pkt.context_id = context_id;
    pkt.lrank	   = src_lrank;
    pkt.to         = dest;
    pkt.seqnum     = len + pkt_len;
    pkt.src        = MPID_MyWorldRank;
    pkt.tag	   = tag;
    pkt.len	   = len;
    MPID_DO_HETERO(pkt.msgrep = (int)msgrep);

    /* We save the address of the send handle in the packet; the receiver
       will return this to us */
    MPID_AINT_SET(pkt.send_id,shandle);
    
    /* Store partners rank in request in case message is cancelled */
    shandle->partner     = dest;
    shandle->is_complete = 1;
    /* Instead of this, the calling code should test from not-complete,
       and set finish if needed */
/* #ifdef FOO
    if (shandle->finish) 
	(shandle->finish)( shandle );
#endif */
    
    DEBUG_PRINT_SEND_PKT("S Sending",&pkt);
    MPID_PKT_PACK( &pkt, sizeof(pkt), dest );

    if (len > 0) {
	MEMCPY( pkt.buffer, buf, len );
	DEBUG_PRINT_PKT_DATA("S Getting data from buf",&pkt);
    }
    /* Always use a blocking send for short messages.
       (May fail with systems that do not provide adequate
       buffering.  These systems should switch to non-blocking sends)
     */
    DEBUG_PRINT_SEND_PKT("S Sending message in a single packet",&pkt);

    /* In case the message is marked as non-blocking, indicate that we don't
       need to wait on it.  We may also want to use nonblocking operations
       to send the envelopes.... */
    MPID_DRAIN_INCOMING_FOR_TINY(1);
    MPID_SendControlBlock( pkt_p, len + pkt_len, dest );

    DEBUG_PRINT_MSG("S Sent message in a single packet");
#undef pkt
    gmpi_unlock();
    return MPI_SUCCESS;

}


int MPID_CH_Eagerb_recv_short( rhandle, from_grank, in_pkt )
MPIR_RHANDLE *rhandle;
int          from_grank;
void         *in_pkt;
{
    MPID_PKT_SHORT_T *pkt = (MPID_PKT_SHORT_T *)in_pkt;
    int          msglen;
    int          err = MPI_SUCCESS;
    
    msglen		  = pkt->len;

    DEBUG_PRINT_MSG("R Starting Eagerb_recv_short");

#ifdef MPID_PACK_CONTROL
    if (MPID_PACKET_RCVD_GET(pkt->src)) {
	MPID_SendProtoAck(pkt->to, pkt->src);
    }
    MPID_PACKET_ADD_RCVD(pkt->to, pkt->src);
#endif

    rhandle->s.MPI_TAG	  = pkt->tag;
    rhandle->s.MPI_SOURCE = pkt->lrank;

    MPID_CHK_MSGLEN(rhandle,msglen,err);
    if (msglen > 0) {
	MEMCPY( rhandle->buf, pkt->buffer, msglen ); 
    }
    rhandle->s.count	  = msglen;
    rhandle->s.MPI_ERROR  = err;

    if (rhandle->finish) {
	MPID_DO_HETERO(rhandle->msgrep = (MPID_Msgrep_t)pkt->msgrep);
	(rhandle->finish)( rhandle );
    }
    rhandle->is_complete = 1;

    return err;
}

/* 
 * This routine is called when it is time to receive an unexpected
 * message
 */
int MPID_CH_Eagerb_unxrecv_start_short( rhandle, in_runex )
MPIR_RHANDLE *rhandle;
void         *in_runex;
{
    MPIR_RHANDLE *runex = (MPIR_RHANDLE *)in_runex;
    int          msglen, err = 0;

    msglen = runex->s.count;
    DEBUG_PRINT_MSG("R Starting Eagerb_unxrecv_start_short");
    MPID_CHK_MSGLEN(rhandle,msglen,err);
    /* Copy the data from the local area and free that area */
    if (runex->s.count > 0) {
	MEMCPY( rhandle->buf, runex->start, msglen );
	FREE( runex->start );
    }
    MPID_DO_HETERO(rhandle->msgrep = runex->msgrep);
    rhandle->s		 = runex->s;
    rhandle->s.count     = msglen;
    rhandle->s.MPI_ERROR = err;
    rhandle->wait	 = 0;
    rhandle->test	 = 0;
    rhandle->push	 = 0;
    rhandle->is_complete = 1;
    if (rhandle->finish) 
	(rhandle->finish)( rhandle );
    MPID_RecvFree( runex );

    return err;
}

/* Save an unexpected message in rhandle */
int MPID_CH_Eagerb_save_short( rhandle, from, in_pkt )
MPIR_RHANDLE *rhandle;
int          from;
void         *in_pkt;
{
    MPID_PKT_SHORT_T   *pkt = (MPID_PKT_SHORT_T *)in_pkt;

    DEBUG_PRINT_MSG("R Starting Eagerb_save_short");
#ifdef MPID_PACK_CONTROL
    if (MPID_PACKET_RCVD_GET(pkt->src)) {
	MPID_SendProtoAck(pkt->to, pkt->src);
    }
    MPID_PACKET_ADD_RCVD(pkt->to, pkt->src);
#endif

    rhandle->s.MPI_TAG	  = pkt->tag;
    rhandle->s.MPI_SOURCE = pkt->lrank;
    rhandle->s.MPI_ERROR  = 0;
    rhandle->from         = from;
    rhandle->partner      = pkt->to;
    rhandle->s.count      = pkt->len;
    /* rhandle->is_complete  = 1; */
    /* Need to save msgrep for heterogeneous systems */
    MPID_DO_HETERO(rhandle->msgrep = (MPID_Msgrep_t)pkt->msgrep);
    if (pkt->len > 0) {
	rhandle->start	  = (void *)MALLOC( pkt->len );
	if (!rhandle->start) {
	    rhandle->s.MPI_ERROR = MPI_ERR_INTERN;
	    return 1;
	}
	MEMCPY( rhandle->start, pkt->buffer, pkt->len );
    }
    rhandle->push = MPID_CH_Eagerb_unxrecv_start_short;
    return 0;
}

void MPID_CH_Eagerb_short_delete( p )
MPID_Protocol *p;
{
    FREE( p );
}


int MPID_CH_Cancel_send( src_lrank, dest, cancel, anti_send_ok, send_id )
int           src_lrank, dest, cancel, anti_send_ok;
MPID_Aint     send_id;
{
  MPID_PKT_ANTI_SEND_T *pkt = gmpi_ctl_alloc(sizeof(MPID_PKT_ANTI_SEND_T));
  
  DEBUG_PRINT_MSG("S starting MPID_CH_Cancel_send");
  gmpi_lock();
    
  /* These references are ordered to match the order 
     they appear in the structure */
  if (anti_send_ok)
    pkt->mode  = MPID_PKT_ANTI_SEND_OK;
  else
    pkt->mode  = MPID_PKT_ANTI_SEND;
  pkt->lrank   = src_lrank;
  pkt->to      = dest;
  pkt->cancel  = cancel;
  pkt->send_id = send_id;
  
  if (anti_send_ok) {
    DEBUG_PRINT_BASIC_SEND_PKT("S Sending anti_send_ok message\n", pkt);
  } else {
    DEBUG_PRINT_BASIC_SEND_PKT("S Sending anti_send message\n", pkt);
  }
  MPID_PKT_PACK( pkt, sizeof(MPID_PKT_ANTI_SEND_T), pkt->to );
  MPID_SendControlBlock( pkt, sizeof(MPID_PKT_ANTI_SEND_T), pkt->to ); 

#ifdef FELDY_NEED_THIS
  /* added this to prevent running out of dma-able memory when sending lots */
  while (gmpi.num_sends_queued > 100)
    MPID_DRAIN_INCOMING;
  
  if (gmpi.num_sends_queued > max_queued) {
    max_queued = gmpi.num_sends_queued;
  }
#endif /* FELDY_NEED_THIS */

  gmpi_unlock();

  return MPI_SUCCESS;
}

int MPID_CH_Cancel_recv( in_pkt, from, cancel_ptr, send_id_ptr )
void *      in_pkt;
int         from;
int *       cancel_ptr;
MPID_Aint * send_id_ptr;
{
  MPID_PKT_ANTI_SEND_T *pkt = (MPID_PKT_ANTI_SEND_T *)in_pkt;
  gmpi_lock();
  
  /* A cancel packet is a little larger than the basic packet size and 
     may need to be unpacked (in the heterogeneous case) */
  MPID_PKT_UNPACK( (MPID_PKT_HEAD_T *)in_pkt + 1,
		   sizeof(MPID_PKT_ANTI_SEND_T) - sizeof(MPID_PKT_HEAD_T),
		   from ); 

  *send_id_ptr = pkt->send_id;
  *cancel_ptr = pkt->cancel;

  gmpi_unlock();
  return MPI_SUCCESS;
}


MPID_Protocol *MPID_CH_Short_setup()
{
    MPID_Protocol *p;

    p = (MPID_Protocol *) MALLOC( sizeof(MPID_Protocol) );
    if (!p) return 0;
    p->send	   = MPID_CH_Eagerb_send_short;
    p->recv	   = MPID_CH_Eagerb_recv_short;
    p->isend	   = MPID_CH_Eagerb_isend_short;
    p->wait_send   = 0;
    p->push_send   = 0;
    p->cancel_send = MPID_CH_Cancel_send;
    p->irecv	   = 0;
    p->wait_recv   = 0;
    p->push_recv   = 0;
    p->cancel_recv = MPID_CH_Cancel_recv;
    p->do_ack      = 0;
    p->unex        = MPID_CH_Eagerb_save_short;
    p->delete      = MPID_CH_Eagerb_short_delete;

    return p;
}
