#!/usr/bin/wish
#
# file init.tcl
#
# there are no more user configurable parts in this file
# please customise $HOME/.bWatchrc.tcl

# everything starts here ...

# version number
set bWatchVersion 1.0.2

# debugging level
set debug 0

#------------------------------------------------------------------------
# nothing beyond here should have to be changed
#------------------------------------------------------------------------

# set the homeDir variable from the enviroment variable $HOME
# but catch all errors !
set errorCode [catch {set homeDir    $env(HOME)} errorMessage]
if {$errorCode} {
    puts "init.tcl : Can't find your home directory.  Giving up."
    exit
}
set bWatchDir  "$homeDir"

# window manager options
# let us catch this because the DISPLAY variable might not
# be set or we might be trying to run bWatch in a text console

set errorCode [catch {wm title . "bWatch $bWatchVersion"} errorMessage]
if {$errorCode} {
    puts "init.tcl : I can't set the window title.  Giving up."
    exit
}

# other window manager options ...
wm iconname . "bWatch $bWatchVersion"


# remove any old tmp files ( this was for version 1.0.0 )
# catch {exec /bin/rm -f /tmp/bWatch*}

# show the console window
# all error messgages will be logged there

showConsole

# we now set the default values.  If $HOME/.bWatchrc.tcl exists
# it will be loaded later and these values will be over written

# get the name of the machine on which bWatch is running
set errorCode [catch {set hostName [exec hostname]} errorMessage]

if {$debug && $errorCode} {
    puts "init.tcl : $errorMessage"
} else { 
    regexp {[a-zA-Z0-9]+} $hostName localHost 
    set listOfHosts "$localHost"
}

# set the columns which should be displayed
#
# 1 - information will be displayed for every host
# 0 - column will not be shown

# number of users on the system
set display(numUsers)         1 

# system time
set display(time)             1

# 1 minute average load
set display(load1)            1

# 5 minutes average load
set display(load5)            1

# 15 minutes average load
set display(load15)           1

# number of processes
set display(numProcesses)     1

# total RAM 
set display(totalMemory)      1

# available free RAM
set display(freeMemory)       1

# shared RAM
set display(sharedMemory)     1

# buffered memory
set display(buffers)          1

# cached memory
set display(cache)            1

# total swap space
set display(totalSwap)        1

# available swap space
set display(freeSwap)         1


# set commands
#
# these are the commands used to obtain information about the hosts

set command(ping)               "/bin/ping"
set command(uptime)             "/usr/bin/uptime "
set command(rsh)                "/usr/bin/rsh"
set command(date)               "/bin/date" 
set command(numberOfProcesses)  "/bin/ps ax | wc"
set command(cat)                "/bin/cat"
set command(memInfo)            "cat /proc/meminfo"


# set colours

# foreground and background coulours
# of the heading of the table

set colour(headingBG)         #066
set colour(headingFG)         #FFF

# foreground and background colours
# of the first column showing host names

set colour(hostNameBG)        #006
set colour(hostNameFG)        #FFF

# the system load, memory etc is divided into 5 operating states represented
# by different colours :
#
# idle           - idle state, almost no memory usage (not implemented yet)
# neutral        - normal load conditions, CPU crunching away doing normal work
# firstWarning   - quite high load, low memory.  Not a major problem
# secondWarning  - overloaded, running out of memory.  Should investigate
# error          - load very high or running out of swap.  Can cause problems eg
#                  processes might start dying 
  
set colour(neutralFG)         black
set colour(neutralBG)         white
set colour(firstWarningFG)    orange
set colour(secondWarningFG)   #A00
set colour(errorFG)           #F00
set colour(errorBG)           #FFF


# The five different operating states give us information
# about the system and how it is performing.
# Please modify the numbers to suit your system.
# numbers below represent my choice for topcat 

# load limits

# idle is up to neutral e.g 0.0 - 0.5 (not yet used)
#set loadsLimit(neutral)                   0.5
set loadLimit(firstWarning)                1.5
set loadLimit(secondWarning)               2.5 
set loadLimit(error)                       5.0

# memory limits in KB
#set freeMemLimit(neutral)
set freeMemLimit(firstWarning)            1024
set freeMemLimit(secondWarning)            512
set freeMemLimit(error)                      1

# swap limits in KB

# give me first warning when 
# swap less then 20 MB
set freeSwapLimit(firstWarning)          20480

# give me socond warning when
# swap less then 10 MB
set freeSwapLimit(secondWarning)         10240

# give me third warning when 
# swap less then 5 MB
set freeSwapLimit(error)                  5120     



# initial display 

# what should be displayed when
# information cannot be obtained
set blank                            "-------"

# default width of all the cells
set cell(width)                              8


# We now load the resource file if there is one
# if the we succeed then our default values just set will
# be overwritten

loadrc


#----------------------------------------------------------------------------
# procedure getHostInfo
# 
# creates a toplevel window and displays more detailed
# information about the host
#
# NOT USED YET
#----------------------------------------------------------------------------

proc getHostInfo {host} {

    toplevel ."$host"
    wm title ."$host" "Information about $host"
    
    set allProcesses [getAllProcesses $host]
    
    text ."$host".test \
	-width 120 \
	-height 40
    
    pack ."$host".test \
	
    
    ."$host".test insert end $allProcesses
    

    # build button
    
    frame ."$host".buttonFrame
    
    pack ."$host".buttonFrame \
	-expand 1 \
	-fill x \
	-in ."$host" 
    
    button ."$host".refreshButton \
	-command {} \
	-text "?????" 
    
    pack ."$host".refreshButton \
	-in ."$host".buttonFrame \
	-side left \
	-fill x
    
    
    button ."$host".exitButton \
	-text {Kill Me} \
	-command {}
    
    
    pack ."$host".exitButton \
	-in ."$host".buttonFrame \
	-side left \
	-fill x
    
    bind ."$host".exitButton <ButtonPress-1> {
	set blah [regexp {\.\"[0-9a-zA-Z]+\"} %W myname]
	destroy $myname
    } 
}


#----------------------------------------------------------------------------
# procedure setError
#
#        this procedure sets the error for a particular $host by changing
#        all its columns to $blank with coulour $colour(errorFG)
#
# arguments
#        host - the host for which the error should be set
#
#----------------------------------------------------------------------------

proc setError {host} {
    global colour blank numUsers time load1 load5 load15 numProcesses \
	totalmemory freeMemory sharedMemory buffers cache totalSwap FreeSwap \
	error

    # set the error flag to true
    set error($host) 1

    ."$host"HostName configure \
	-fg $colour(errorBG) \
	-bg $colour(errorFG)
    

    catch {."$host"numUsers configure \
	       -fg $colour(errorFG)}

    catch {."$host"time configure \
	       -fg $colour(errorFG)}

    catch {."$host"load1 configure \
	       -fg $colour(errorFG)}

    catch {."$host"load5 configure \
	       -fg $colour(errorFG)}

    catch {."$host"load15 configure \
	       -fg $colour(errorFG)}

    catch {."$host"numProcesses configure \
	       -fg $colour(errorFG)}

    catch {."$host"totalMemory configure \
	       -fg $colour(errorFG)}

    catch {."$host"freeMemory configure \
	       -fg $colour(errorFG)}

    catch {."$host"sharedMemory configure \
	       -fg $colour(errorFG)}

    catch {."$host"buffers configure \
	       -fg $colour(errorFG)}

    catch {."$host"cache configure \
	       -fg $colour(errorFG)}

    catch {."$host"totalSwap configure \
	       -fg $colour(errorFG)}

    catch {."$host"freeSwap configure \
	       -fg $colour(errorFG)}
}

#----------------------------------------------------------------------------
# procedure unSetError
#
#        this procedure unsets the error for a particular $host by changing
#        all its columns 
#
# arguments
#        host - the host for which the error should be unset
#
#----------------------------------------------------------------------------

proc unSetError {host} {
    global colour blank numUsers time load1 load5 load15 numProcesses \
	totalmemory freeMemory sharedMemory buffers cache totalSwap FreeSwap \
	error

    # set the error flag to true
    set error($host) 0

    ."$host"HostName configure \
	-fg $colour(neutralFG) \
	-bg $colour(neutralBG)
    

    catch {."$host"numUsers configure \
	       -fg $colour(neutralFG)}

    catch {."$host"time configure \
	       -fg $colour(neutralFG)}

    catch {."$host"load1 configure \
	       -fg $colour(neutralFG)}

    catch {."$host"load5 configure \
	       -fg $colour(neutralFG)}

    catch {."$host"load15 configure \
	       -fg $colour(neutralFG)}

    catch {."$host"numProcesses configure \
	       -fg $colour(neutralFG)}

    catch {."$host"totalMemory configure \
	       -fg $colour(neutralFG)}

    catch {."$host"freeMemory configure \
	       -fg $colour(neutralFG)}

    catch {."$host"sharedMemory configure \
	       -fg $colour(neutralFG)}

    catch {."$host"buffers configure \
	       -fg $colour(neutralFG)}

    catch {."$host"cache configure \
	       -fg $colour(neutralFG)}

    catch {."$host"totalSwap configure \
	       -fg $colour(neutralFG)}

    catch {."$host"freeSwap configure \
	       -fg $colour(neutralFG)}
}

#----------------------------------------------------------------------------
# procedure getDate
# 
#        procedure executes the command(date) and returns its result
#        no error checking is done
#
#----------------------------------------------------------------------------

proc getDate {} {

    global command
    set date [exec $command(date)]
    return $date
}


#----------------------------------------------------------------------------
# procedure pingHost
#
#        procedure sends one ping to $host using the $command(ping)
#        if an error occurs pingHost will write this error to the console
#        
# Arguments :
#        $host - the name of the host to ping
#
# Results :
#        procedure returns 0 if the host replied to the ping or
#        returns the error code which was returned by command(ping) 
# 
#----------------------------------------------------------------------------

proc pingHost {host} {
    
    global command
    set errorCode \
	[catch {set pingResult \
		    [exec $command(ping) -c 1 $host]} errorMessage]
    
    if {$errorCode} {
	set date [getDate]
	.console.messageList insert end \
	    "$date :: pingHost($host) : $errorMessage"
	return $errorCode
    } else {
	return 0
    }
}


