/*									tab:4
 *
 * stat.sc - Routines to calculate confidence intervals for tests
 *
 * "Copyright (c) 1993 and The Regents of the University 
 * of California.  All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."
 *
 * Author: 			Andrea C. Dusseau
 * Version:			73
 * Creation Date:	Mon Jun 28 13:59:51 1993
 * Filename:		stat.sc
 * History:
 */
static char _version_string_[] = "\nVersion:1:stat.sc\0\n";

#include <math.h>
#include <stdio.h>
#include "gm.h"			/* needed for 'inline' in logp_main.h */
#include "logp_stat.h"
#include "logp_main.h"

int verbose = 1;

static double mean_time[MAX_BUFFER];   /* mean time */
static double size[MAX_BUFFER];   /* mean time */
static double sum_time[MAX_BUFFER];
static double sqrd_time[MAX_BUFFER];
static double var_time[MAX_BUFFER];    /* variance */
static double std_time[MAX_BUFFER];    /* standard deviation */
static int    count[MAX_BUFFER];	/* count the # of iterations of each size */
static int bogus;		/* Number of bogus points thrown away */

static double mean_time1;
static double sum_time1;
static double sqrd_time1;
static double var_time1;
static double std_time1;
static int    count1;

static int n;

void stats_initialize (void)
{
    int i = 0;

    GM_PARAMETER_MAY_BE_UNUSED (_version_string_);
    
    for (; i<MAX_BUFFER; i++) {
		mean_time[i] = 0.0;
		sum_time[i] = 0.0;
		sqrd_time[i] = 0.0;
		var_time[i] = 0.0;
		std_time[i] = 0.0;
		count[i] = 0;
    }	
    bogus = 0;
    n = 0;
    mean_time1 = 0.0;
    sum_time1 = 0.0;
    sqrd_time1 = 0.0;
    var_time1 = 0.0;
    std_time1 = 0.0;
    count1 = 0;
}

/* 
 * Collecting and processing the statistical data.
 * return 1 -- continue to collect data points 
 *        0 -- stop collecting data
 */
int stats_get(int this_size, int s, int iter, double user_time)
{
    double range;

	if (user_time <= 0)  {
#if 1
		printf("# ERROR: Reported time of %f for size %f on iteration %d\n",
			   user_time, size[s], iter);
#endif
		return 1;
    }

    if (verbose)    
		printf("%d: s: %d, size: %f, time: %f\n", iter, s, size[s], user_time);

    /* Throw away the first iteration */
    if (iter == 0) {
		/* How many points are we dealing with? */
		if (s > n) n = s;
		size[s] = this_size;	
		return 1;
    }

    if (size[s] != this_size) {
		printf("# ERROR: Expected size for designator %d to be %f instead of %d.\n",
			   s, size[s], this_size);
		return 0;
    }

    /* 
     * The next three iterations are used only to get an approximation
     * to the mean so that we can know which later values are bogus and
     * throw out those points (maybe due to context switch).
     */
    if (iter < 3) {
		sum_time[s] += user_time;
		return 1;
    }
    
    if (iter == 3) {			/* Get the estimated mean value */
		sum_time[s] += user_time;
		mean_time[s] = sum_time[s] / (double)iter;
		sum_time[s] = 0;
		return 1;
    }

    if (iter >= MAX_ITER) {
		/*      printf("#max iterations exceed\n");*/
		return 0;
    }

    /* Now starting the 4th iteration ... throw away bogus point */
    if (s >= n) {
		if (user_time > 3*mean_time[s]) {
			if (verbose)
				printf("**** iter %d: throw away data point for %f bytes (MAX) (%8.4f).\n", 
					   iter, size[s], user_time);	
			goto check_cnf;
		}
    }
    else {
		if (user_time > 3*mean_time[s+1] || user_time > 3*mean_time[s]) {
			if (verbose)
				printf("**** iter %d: throw away data point for %f bytes (%8.4f).\n", 
					   iter, size[s], user_time);
			goto check_cnf;
		}
    }


    count[s]++;
    sum_time[s] += user_time;
    sqrd_time[s] += user_time * user_time;
    mean_time[s] = sum_time[s]/count[s];


  check_cnf:

    /*
     * We need at least 30 iterations for Central Limit Theorem to apply
     */
    if (count[s] <= 100)  return 1;

    mean_time[s] = sum_time[s] / (double)count[s];
    /* Recall that var(X) = E(^2) - [E(X)]^2 */
    var_time[s] = sqrd_time[s] / (double)count[s]- 
		mean_time[s] * mean_time[s];
    std_time[s] = sqrt(var_time[s]);
    
    if (verbose) {
		printf("Size: %f, Mean (us): %8.4f, Var: %8.4f, Std: %8.4f\n", size[s], 
			   mean_time[s], var_time[s], std_time[s]);
    }

    /* The interval around the mean time */
    range = 2 * Z_CONFIDENCE * std_time[s] / sqrt((double)count[s]);
    if (range <=  0.10 * mean_time[s]) {
		if (verbose)
			printf("Conf intv (small enough): %f\n", range);
		return 0;
    }

    if (verbose)
		printf("Conf intv (__too large__): %f\n", range);
    return 1;
}

/* 
 * Stats_final() finds a linear regression model with least-square criterion
 * and zero overall error.
 * Reference: Chapter 14 in Jain's book 
 */
void stats_final(double *start, double *start_cnf, double *slope, double *slope_cnf)
{
    double sum_prod = 0, xmean = 0, ymean = 0, sum_x_sqrd = 0;
    double b0, b1, sb0, sb1;
    double denom, temp;
    double sse, se;
    int i;

    for (i = 0; i < n; i++) {
		sum_prod += mean_time[i] * size[i];
		xmean += (double) size[i];
		ymean += mean_time[i];
		sum_x_sqrd += (double)size[i] * (double)size[i];

    }

    xmean /= (double)n;
    ymean /= (double)n;

    if (verbose)
		printf(" n: %d, Sum xy: %.4e, Sum X^2: %.4e\n E[x]: %8.4f, E[y]: %8.4f\n",
			   n, sum_prod, sum_x_sqrd, xmean, ymean);
    denom = sum_x_sqrd - n * xmean * xmean;
    b1 = (sum_prod - n*xmean*ymean) / denom;
    b0 = ymean - b1 * xmean;

    if (verbose) 
		printf("Linear model: predicted time = %8.4f (us) + %8.4f (us/byte)*X.\n", b0, b1);
	
    sse = 0;
    for (i = 0; i < n; i++) {
		temp = mean_time[i] - b0 - b1*size[i];
		sse += temp * temp;
    }
    if (verbose) printf("Sum of Squared Errors: %10.4f\n", sse);
    se = sqrt((double)(sse/(n-2)));
    sb0 = se * ((double)(1.0/n + xmean*xmean/denom));
    sb1 = se / sqrt((double)denom);
    if (verbose) {
		printf("sb0: %8.4f, sb1: %8.4f\n", sb0, sb1);
		printf("Confidence intervals:\tb0 (-/+ %f), b1 (-/+ %f)\n", 
			   T_VARIATE * sb0, T_VARIATE * sb1);
    }

    *start = b0; *slope = b1;
    *start_cnf = T_VARIATE * sb0; *slope_cnf = T_VARIATE * sb1;
}

/* 
 * Collecting and processing the statistical data for global comm. routines:
 * 	broadcasts, reductions, scans.
 * return 1 -- continue to collect data points 
 *        0 -- stop collecting data
 */
int stats_global(int iter, double user_time, double *mean_result, double *cnf_int)
{
    double range;

    if (user_time <= 0) {
#if 1
		printf("# ERROR: Reported time of %f on iteration %d\n", user_time, iter);
#endif
		return 1;
    }

    if (verbose){
      if (iter <2) 
	printf("%d: time: %f\n", iter, user_time);
    }
    /* Throw away the first iteration */
    if (iter == 0) return 1;

    /* 
     * The next three iterations are used only to get an approximation
     * to the mean so that we can know which later values are bogus and
     * throw out those points (maybe due to context switch).
     */
    if (iter < 3) {
		sum_time1 += user_time;
		return 1;
    }
    
    if (iter == 3) {			/* Get the estimated mean value */
		sum_time1 += user_time;
		mean_time1 = sum_time1 / (double)iter;
		sum_time1 = 0;
		return 1;
    }

    if (iter >= MAX_ITER) {
		/* printf("#max iterations exceed\n");*/
		return -1;
    }

    /* Now starting the 4th iteration ... throw away bogus point */
    if (user_time > 3*mean_time1) {
		if (verbose)
			printf("**** iter %d: throw away this iter. point (%8.4f).\n", 
				   iter, user_time);
		bogus++;
		return 1;
    }

    count1++;
    sum_time1 += user_time;
    sqrd_time1 += user_time * user_time;
    mean_time1 = sum_time1/count1;

    /*
     * We need at least 30 iterations for Central Limit Theorem to apply
     */
    if (count1 <= 100)  return 1;

    /* Recall that var(X) = E(X^2) - [E(X)]^2 */
    mean_time1 = sum_time1 / (double)count1;
    /* Recall that var(X) = E(X^2) - [E(X)]^2 */
    var_time1 = sqrd_time1 / (double)count1 - mean_time1 * mean_time1;
    std_time1 = sqrt(var_time1);

    if (verbose) {
		printf("Iter: %d, Mean (us): %8.4f, Var: %8.4f, Std: %8.4f\n", 
			   iter, mean_time1, var_time1, std_time1);
    }

    /* The interval around the mean time */
    range = 2 * Z_CONFIDENCE * std_time1 / sqrt((double)count1);
    if (range <=  0.10 * mean_time1) {
		if (verbose)
			printf("Conf intv (small enough): %f\n", range);
		*mean_result = mean_time1;
		*cnf_int = range;
		return 0;
    }

    if (verbose)
		printf("Conf intv (__too large__): %f\n", range);

    return 1;
}

static
void stats_print (void)
{

    int i;

    printf("\n%-s %s %s\n\n", "size", "mean time", "bandwidth");

    for (i = 0; i <= n; i++) {
        printf("%d %.8f +/- %.8f %.8f\n", (int)size[i],
			   mean_time[i], std_time[i], size[i]/mean_time[i]);
    }
    printf("\n");

}
