/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>

#include "gm.h"
#include "gm_internal.h"
#include "stats.h"
#include "math.h"

enum
{
  MILLION = 1000 * 1000,
  NUM_BUFFERS = 10,
  SIZE = 7,
  PORT = 2,
  UNIT = 0,
  DEBUG_FAULTS = 1
};

unsigned int this_node_id;

void *rbuf[NUM_BUFFERS];
void *sbuf[NUM_BUFFERS];

/****************************************************************
 * error generation
 ****************************************************************/

#if 0
gm_u8_t _good_route[] = { 0x80 };
gm_u8_t _bad_route[] = { 0x81 };

static gm_status_t
bad_route (struct gm_port *p)
{
  return _gm_set_route (p, this_node_id, 1, (char *) _bad_route);
}

static gm_status_t
good_route (struct gm_port *p)
{
  return _gm_set_route (p, this_node_id, 1, (char *) _good_route);
}
#endif

/****************************************************************
 * testing infrastructure
 *
 * These functions provide a sort of test scripting interface.
 ****************************************************************/

static void
test_timed_out (void *context)
{
  char *test_name;

  test_name = (char *) context;
  fprintf (stderr, "*** Test \"%s\" timed out. ***\n", test_name);
  gm_exit (GM_FAILURE);
}

struct send_record
{
  struct send_record *next;
  unsigned int tag;
}
 *first_send_record, *last_send_record;


static void
_sent (gm_port_t * port, void *_context, gm_status_t status, char *extra)
{
  struct send_record *r, *context;
  GM_PARAMETER_MAY_BE_UNUSED (port);
  GM_PARAMETER_MAY_BE_UNUSED (status);

  context = _context;
  r = first_send_record;
  if (!r)
    {
      fprintf (stderr, "send completed when none expected\n");
      gm_exit (GM_FAILURE);
    }

  if (r != context)
    {
      fprintf (stderr, "send %u completed.  expected send %u to complete.\n",
	       context->tag, r->tag);
      gm_exit (GM_FAILURE);
    }

  if (DEBUG_FAULTS)
    printf ("completed send %u %s\n", context->tag, extra);

  first_send_record = r->next;
  gm_lookaside_free (r);
}

static void
sent (gm_port_t * port, void *context, gm_status_t status)
{
  if (status != GM_SUCCESS)
    gm_perror ("expected good send failed", status), gm_exit (GM_FAILURE);

  _sent (port, context, status, "normally");
}

static jmp_buf test_env;

static void
last_sent (gm_port_t * port, void *context, gm_status_t status)
{
  sent (port, context, status);
  longjmp (test_env, 1);
}

static void
dropped (gm_port_t * port, void *context, gm_status_t status)
{
  if (status != GM_SEND_DROPPED)
    {
      gm_perror ("send was not dropped", status);
      gm_exit (GM_FAILURE);
    }
  _sent (port, context, status, "(dropped)");
}

static void
freed (gm_port_t * port, void *context, gm_status_t status)
{
  GM_PARAMETER_MAY_BE_UNUSED (port);
  GM_PARAMETER_MAY_BE_UNUSED (context);
  GM_PARAMETER_MAY_BE_UNUSED (status);
  
  printf ("freed()\n");
}

static void
resume (gm_port_t * port, void *context, gm_status_t status)
{
  if (status == GM_SUCCESS)
    {
      gm_perror ("expected bad send succeeded", status);
      gm_exit (GM_FAILURE);
    }
  _sent (port, context, status, "with error");
  gm_perror ("expected error", status);
  if (DEBUG_FAULTS)
    {
      printf ("resuming sends\n");
      fflush (stdout);
    }
  gm_resume_sending (port, GM_LOW_PRIORITY, this_node_id, PORT, freed, 0);
}

static void
drop_sends (gm_port_t * port, void *context, gm_status_t status)
{
  if (status == GM_SUCCESS)
    {
      gm_perror ("expected bad send succeeded", status);
      gm_exit (GM_FAILURE);
    }
  _sent (port, context, status, "with error");
  gm_perror ("expected error", status);
  if (DEBUG_FAULTS)
    {
      printf ("dropping sends\n");
      fflush (stdout);
    }
  gm_drop_sends (port, GM_LOW_PRIORITY, this_node_id, PORT, freed, 0);
}

static struct expect_record
{
  struct expect_record *next;
  unsigned int tag;
}
 *first_expect_record, *last_expect_record;

static void
notice_recv (unsigned int tag)
{
  if (!first_expect_record)
    {
      fprintf (stderr, "got receive %u when no receive was expected.\n", tag);
      gm_exit (GM_FAILURE);
    }

  if (first_expect_record->tag != tag)
    {
      fprintf (stderr, "got receive %u when receive %u was expected.",
	       tag, first_expect_record->tag);
      gm_exit (GM_FAILURE);
    }

  gm_lookaside_free (first_expect_record);
  first_expect_record = first_expect_record->next;
}

static void
run_test (gm_port_t * port, char *test_name)
{
  gm_alarm_t a;

  if (DEBUG_FAULTS)
    {
      printf ("running test \"%s\"\n", test_name);
      fflush (stdout);
    }

  if (setjmp (test_env) == 0)
    {
      gm_recv_event_t *e;

      gm_initialize_alarm (&a);
      gm_set_alarm (port, &a, (gm_u64_t) (66 * MILLION),
		    test_timed_out, test_name);

      while (1)
	{
	  e = gm_blocking_receive (port);
	  switch (gm_ntoh_u8 (e->recv.type))
	    {
	    default:
	      gm_unknown (port, e);
	      break;
	    case GM_RECV_EVENT:
	      {
		gm_u8_t *message;
		gm_u8_t tag;

		message = gm_ntohp (e->recv.message);
		tag = message[0];
		message[0] = 0;
		notice_recv (tag);
		if (gm_ntoh_u8 (e->recv.tag) == 4)
		  return;
	      }
	    case GM_NO_RECV_EVENT:
	      break;
	    }
	}
    }

  gm_cancel_alarm (&a);
}

static void
expect (gm_port_t * port, unsigned int tag)
{
  static struct gm_lookaside *lookaside;
  struct expect_record *r;

  /* init */

  if (!lookaside)
    {
      lookaside = gm_create_lookaside (sizeof (struct expect_record), 1);
      if (!lookaside)
	{
	  gm_perror ("could not create expect lookaside", GM_OUT_OF_MEMORY);
	  gm_exit (GM_OUT_OF_MEMORY);
	}
    }

  /* allocate a send record */

  r = (struct expect_record *) gm_lookaside_alloc (lookaside);
  if (!r)
    {
      gm_perror ("could not allocate send record", GM_OUT_OF_MEMORY);
      gm_exit (GM_OUT_OF_MEMORY);
    }

  /* provide receive buffer to allow the expected receive */

  gm_provide_receive_buffer (port, rbuf[tag], SIZE, GM_LOW_PRIORITY);

  /* record details of the expected receive. */

  r->tag = tag;
  r->next = 0;
  if (first_expect_record)
    last_expect_record = last_expect_record->next = r;
  else
    first_expect_record = last_expect_record = r;
}

static void
send (gm_port_t * port, unsigned int tag,
      gm_send_completion_callback_t callback)
{
  static struct gm_lookaside *lookaside;
  static struct send_record *r;

  /* init */

  if (!lookaside)
    {
      lookaside = gm_create_lookaside (sizeof (struct send_record), 1);
      if (!lookaside)
	{
	  gm_perror ("could not create send record lookaside",
		     GM_OUT_OF_MEMORY);
	  gm_exit (GM_OUT_OF_MEMORY);
	}
    }

  /* record the details of the send */

  r = gm_lookaside_alloc (lookaside);
  if (!r)
    {
      gm_perror ("could not allocate send record", GM_OUT_OF_MEMORY);
      gm_exit (GM_OUT_OF_MEMORY);
    }
  r->tag = tag;
  r->next = 0;
  if (first_send_record)
    last_send_record = last_send_record->next = r;
  else
    first_send_record = last_send_record = r;

  /* perform the send */

  gm_send_to_peer_with_callback (port, sbuf[tag], SIZE,
				 gm_max_length_for_size (SIZE),
				 GM_LOW_PRIORITY, this_node_id,
				 callback, (void *) r);
}

/****************************************************************
 * test out of recv buffers error handling
 ****************************************************************/

static void
resume_no_recv_buffer (gm_port_t * port, void *context, gm_status_t status)
{
  expect (port, 3);
  expect (port, 4);
  resume (port, context, status);
}

static void
drop_sends_no_recv_buffer (gm_port_t * port, void *context,
			   gm_status_t status)
{
  drop_sends (port, context, status);
  expect (port, 5);
  expect (port, 6);
  send (port, 5, sent);
  send (port, 6, last_sent);
}

static void
test_no_recv_buffer_drop (gm_port_t * port)
{
  /****
   * drop test
   ****/

  printf ("testing drop\n");
  fflush (stdout);

  expect (port, 0);
  expect (port, 1);

  send (port, 0, sent);
  send (port, 1, sent);
  send (port, 2, drop_sends_no_recv_buffer);
  send (port, 3, dropped);
  send (port, 4, dropped);

  run_test (port, "no recv buffer drop");
}

static void
test_no_recv_buffer_resume (gm_port_t * port)
{
  /****
   * resume test
   ****/

  printf ("testing resume\n");
  fflush (stdout);

  expect (port, 0);
  expect (port, 1);

  send (port, 0, sent);
  send (port, 1, sent);
  send (port, 2, resume_no_recv_buffer);
  send (port, 3, sent);
  send (port, 4, last_sent);

  run_test (port, "no recv buffer resume");
}

static int
gm_fault (int argc, char *argv[])
{
  int i;
  gm_status_t status;
  struct gm_port *port;
  GM_PARAMETER_MAY_BE_UNUSED (argc);
  GM_PARAMETER_MAY_BE_UNUSED (argv);
  
  /****************
   * initialize
   ****************/

  status = gm_init ();
  if (status != GM_SUCCESS)
    gm_perror ("could not initialize GM", status), gm_exit (status);

  status = gm_open (&port, UNIT, PORT, "gm_fault", GM_API_VERSION_1_1);
  if (status != GM_SUCCESS)
    gm_perror ("could not open GM port", status), gm_exit (status);

  for (i = 0; i < NUM_BUFFERS; i++)
    {
      sbuf[i] = gm_dma_malloc (port, gm_max_length_for_size (SIZE));
      if (!sbuf[i])
	{
	  gm_perror ("could not alloc send buffer", GM_OUT_OF_MEMORY);
	  gm_exit (GM_OUT_OF_MEMORY);
	}
      *(gm_u8_t *) sbuf[i] = (gm_u8_t) i;	/* tag the send buffer */
      rbuf[i] = gm_dma_malloc (port, gm_max_length_for_size (SIZE));
      if (!rbuf[i])
	{
	  gm_perror ("could not alloc recv buffer", GM_OUT_OF_MEMORY);
	  gm_exit (GM_OUT_OF_MEMORY);
	}
    }

  status = gm_set_acceptable_sizes (port, GM_LOW_PRIORITY, 1 << SIZE);
  if (status != GM_SUCCESS)
    gm_perror ("could not set acceptable sizes", status), gm_exit (status);

  status = gm_get_node_id (port, &this_node_id);
  if (status != GM_SUCCESS)
    gm_perror ("could not get node id", status), gm_exit (status);
  if (DEBUG_FAULTS)
    printf ("This is node %u.\n", this_node_id);

#if 0
  status = good_route (port);
  if (status != GM_SUCCESS)
    gm_perror ("could set good route", status), gm_exit (status);
#endif

  /****************
   * test
   ****************/

  while (1)
    {
      test_no_recv_buffer_drop (port);
      test_no_recv_buffer_resume (port);
    }

  return 0;
}

/****************************************************************
 * Entry point
 ****************************************************************/

#define GM_MAIN gm_fault
#include "gm_main.h"

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
