/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm.h"

#define MAX_RECORD_COUNT 100000
#define MAX_BUFFER_COUNT 100

/* randomize the contents of a buffer and return checksum */

gm_u8_t
fill (void *ptr, int len)
{
  gm_u8_t sum, val;

  sum = 0;
  while (len--)
    {
      val = (gm_u8_t) gm_rand ();
      ((gm_u8_t *)ptr)[len] = val;
      sum += val;
    }
  return sum;
}

/****************************************************************
 * Record handling
 ****************************************************************/

struct record
{
  void *buffer;
  int length;
  gm_u8_t checksum;
};

int alloc_count;
int free_count;

void
record_verify (struct record *r)
{
  gm_u8_t sum;
  int length;
  gm_u8_t *buffer;
  
  length = r->length;
  buffer = (gm_u8_t *) r->buffer;
  gm_always_assert (buffer);

  sum = 0;
  while (length--)
    {
      sum += buffer[length];
    }
  if (sum != r->checksum)
    {
      gm_printf ("record checksum is incorrect\n");
      gm_exit (GM_FAILURE);
    }
}

struct record *
record_alloc (struct gm_port *port)
{
  struct record *r;
  unsigned long length;
  void *b;
  
  length = (gm_rand () & 0x1fff) + 1;
  r = gm_malloc (sizeof (struct record));
  if (!r)
    {
      gm_perror ("could not allocate buffer record", GM_OUT_OF_MEMORY);
      gm_exit (GM_OUT_OF_MEMORY);
    }
  b = gm_dma_malloc (port, length);
  if (!b)
    {
      goto abort_with_record;
    }
  r->buffer = b;
  r->length = length;
  r->checksum = fill (b, length); /* simulate early user access */
  alloc_count++;
  return r;

 abort_with_record:
  gm_free (r);
  return 0;
}

void
record_free (struct record *r, struct gm_port *port)
{
  record_verify (r);
  gm_dma_free (port, r->buffer);
  gm_free (r);
  free_count++;
}

void
record_tickle (struct record *r)
{
  record_verify (r);
  r->checksum = fill (r->buffer, r->length);
}

/****************************************************************
 * Operations on random records
 ****************************************************************/

struct record *record[MAX_RECORD_COUNT];
int buffer_count;

/* Allocate a new buffer and append it to the buffer array, if there
   is space for another buffer in the buffer array. */

struct record *
allocate_a_record (struct gm_port *port)
{
  struct record *r;
  
  if (buffer_count >= MAX_RECORD_COUNT)
    return (struct record *)0;

  r = record_alloc (port);
  if (r)
    {
      record[buffer_count++] = r;
    }
  return r;
}

/* Remove a random buffer from the buffer array, if there is one. */

void
free_a_record (struct gm_port *port)
{
  if (buffer_count)
    {
      unsigned long pos;

      pos = gm_rand_mod (buffer_count);
      record_free (record[pos], port);
      record[pos] = record[--buffer_count];
    }
}

/* Write random values to a buffer */

void
tickle_a_record (struct gm_port *port)
{
  if (buffer_count)
    {
      unsigned long pos;

      pos = gm_rand_mod (buffer_count);
      record_tickle (record[pos]);
    }
}

/****************************************************************
 * Main test engine
 ****************************************************************/

int
main (int argc, char *argv[])
{
  struct gm_port *port;
  gm_status_t status;
  
  /* Initialize the GM library. */

  status = gm_open (&port, 0, 2, "gm_malloc test", GM_API_VERSION_1_3);
  if (status != GM_SUCCESS)
    {
      gm_perror ("Could not open port", status);
      gm_exit (status);
    }
  
  gm_printf ("Allocating buffers\n");
  do
    {
      int r;

      r = gm_rand() & 0xff;
      if (r >= 0x80)
	{
	  tickle_a_record (port);
	}
      else if (r > 0x40)
	{
	  free_a_record (port);
	}
      else
	{
	  allocate_a_record (port);
	}
    }
  while (buffer_count < MAX_BUFFER_COUNT);

#if 0
  gm_printf ("Exhausting DMAable memory\n");
  while (buffer_count < MAX_RECORD_COUNT)
    {
      if (allocate_a_record (port) == 0)
	{
	  gm_printf ("Exhausted DMAable memory\n");
	  break;
	}
    }

#if 0
  {
     int i;

     gm_printf ("Exercising exhausted allocator");
     for (i = 0; i<1000; i++)
       {
         int r;

         r = gm_rand() & 0xff;
         if (r >= 0x80)
	   {
	     tickle_a_record (port);
	   }
         else if (r > 0x40)
	   {
	     free_a_record (port);
	   }
         else
	   {
	     allocate_a_record (port);
	   }
       }
  }
#endif

  gm_printf ("Relieving exhausted memory allocator");
  while (buffer_count > MAX_BUFFER_COUNT)
    {
      free_a_record (port);
    }
#endif
  
  gm_printf ("Freeing buffers\n");
  do
    {
      int r;

      r = gm_rand() & 0xff;
      if (r >= 0x80)
	{
	  tickle_a_record (port);
	}
      else if (r > 0x40)
	{
	  allocate_a_record (port);
	}
      else
	{
	  free_a_record (port);
	}
    }
  while (buffer_count);
  
  /*  gm_printf ("%d allocs, %d frees\n", alloc_count, free_count); */
  gm_printf ("done\n");
  return 0;
}
