/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include <stdio.h>		/* -*-objc-*- */
#include "GM.h"

#if !GM_SIMULATE
#include <lanai_device.h>
#endif

#define REPORT_PROGRESS 0

#define TEST_PRIORITY GM_LOW_PRIORITY
#define MAX_TEST_SIZE 12
#define TEST_TIME 4
#define TEST_MESSAGE_CNT 1

enum test_message_types {
  test_ping_type = GM_FIRST_NONGM_TYPE,
  FIRST_ILLEGAL_TYPE,
};

typedef struct Chainable_Test_Message
{
  struct Chainable_Test_Message *next;
  gm_u32_t _reserved;
  gm_message_t gm_message;
} Chainable_Test_Message;

Chainable_Test_Message *free_test_messages[32] = {0};
/* int test_sends_pending; */
unsigned target_node_id;

	  
static inline
void
sanity_check ()
{
#if GM_DEBUG
  Chainable_Test_Message *foo;
  int i;

  for (i=0; i<32; i++)
    {
      foo = free_test_messages[i];
      while (foo)
	{
	  gm_assert (foo->gm_message.type);
	  foo = foo->next;
	}
    }
#endif
}

#include <signal.h>

void (*signal (int sig, void (*disp)(int)))(int);

/* int sigvec(int sig, struct sigvec *vec, struct sigvec *ovec); */
unsigned int alarm (unsigned int seconds);

int alarmed;

void
alarm_handler (int sig)
{
  printf ("# Got alarm signal.\n");
  alarmed = 1;
  signal (SIGALRM, &alarm_handler);
}

void
test_initialize (unsigned port, int argc, char **argv)
{
  gm_recv_token_t *r;
  gm_send_token_t *s;
  gm_message_t *p;
  const unsigned priority = TEST_PRIORITY;
  int i, size;
  gm_status_t status;

  status = gm_init_port (port, argc, argv);
  if (status != GM_SUCCESS)
    {
      fprintf (stderr, "Could not initialize GM.\n");
      gm_exit (status);
    }

  if (argc >= 3)
    {
      extern int atoi (char *);
      target_node_id = atoi (argv[2]);
    }
  else
    target_node_id = gm_globals.this_node_id;

				/* Setup recv tokens */
#if 0
  for (size = 5; size<=MAX_TEST_SIZE; size++)
#else
  for (size = MAX_TEST_SIZE; size<=MAX_TEST_SIZE; size++)
#endif
    {
      int tokens_of_size;
      if (size<8)
	tokens_of_size = 512/(1<<size);
      else
	tokens_of_size = 2;

      for (i = 0; i < tokens_of_size; i++)
	{
	  Chainable_Test_Message *ctp;
	  
	  p = (gm_message_t *)gm_dma_calloc (1, (1<<size)-1);
	  gm_always_assert (p);
      
	  *(gm_u8_t *)&p->size = size;
				/* Make sure message is aligned. */
	  gm_always_assert (! ( (int)p & 0x7));
#if !GM_SIMULATE
	  gm_assert ((int)p >= (int)UBLOCK[0]);
	  gm_assert ((int)p < ( (int)UBLOCK[0] + gm_globals.dma_mem_len ));
#endif
	  gm_provide_receive_buffer (p, size, TEST_PRIORITY);

				/* Alloc send tokens. */
	  ctp = (Chainable_Test_Message *)gm_dma_calloc (1, (1<<size)-1);
	  gm_always_assert (ctp);
	  ctp->next = free_test_messages[size];
	  free_test_messages[size] = ctp;
	  ctp->gm_message.type = gm_htons (test_ping_type);
	  *(gm_u8_t *)&ctp->gm_message.size = size;
	  ctp->gm_message.target_node_id = gm_htonl (target_node_id);
	}
    }

  signal (SIGALRM, &alarm_handler);
}

void
send_first_message (int size, int length)
{
  Chainable_Test_Message *p;
  int i;

  gm_always_assert (size <= MAX_TEST_SIZE);
  sanity_check ();
  if (gm_globals.this_node_id == 1)
    {
      gm_always_assert (free_test_messages[size]);
      free_test_messages[size]->gm_message.length = gm_htonl (length);
#ifdef CHECK_PAYLOAD
      test_randomize_payload (&free_test_messages[size]->gm_message);
      test_checksum (&free_test_messages[size]->gm_message, 1);
#endif
      gm_send (&free_test_messages[size]->gm_message, length, TEST_PRIORITY);
      free_test_messages[size] = free_test_messages[size]->next;
      gm_assert (!free_test_messages[size] ||
		 free_test_messages[size]->gm_message.type);
    }
  sanity_check ();
}

static /* inline  */
void
test_free_sent_message (gm_message_t *p)
{
  Chainable_Test_Message *ctp;

  gm_assert (p->type);

  ctp = (void *)p - 8;
  ctp->next = free_test_messages[p->size];
  free_test_messages[p->size] = ctp;
}

int
gm_bw (int argc, char **argv)
{
  int test_length;
  int recv_cnt, send_cnt, sleep_cnt, i, size;
  gm_message_t *received_message;
  gm_message_t *received_buffer;
  
  test_initialize (3, argc, argv);

#if 0
  size = 7;
  for (test_length = 64; test_length < GM_DEBUG?64:100; test_length+=4)
    {
      {
#elif 0
  for (size = 6; size <= MAX_TEST_SIZE; size++)
    {
      {
      test_length = (1<<size)-8;
#elif 1
  for (size = MAX_TEST_SIZE; size >= 6; size--)
    {
      {
      test_length = (1<<size)-8;
#elif 1
  for (size = 6; size <= MAX_TEST_SIZE; size++)
    {
      for (test_length = 1<<(size-1);
	   test_length <= (1<<size) - 8;
	   test_length = (test_length == ( 1 << (size-1) ) + 8
			  ? (1<<size)-8
			  : test_length + 8))
	{
#else
  for (test_length = 24; test_length < 2048; test_length+=8)
    {
      size = gm_size (test_length);
      {
#endif

	alarmed = recv_cnt = send_cnt = sleep_cnt = 0;
	send_first_message (size, test_length);

	sanity_check ();
      
	if (gm_globals.this_node_id == 1)
	  {
	    send_cnt++;
	    alarm (TEST_TIME);
	  }
	else
	  {
	    while (1)
	      {
		received_message = gm_blocking_receive (&received_buffer);
		gm_provide_receive_buffer (received_buffer,
					   received_message->size,
					   TEST_PRIORITY);
		/* printf ("# Got a message.\n"); */
		fflush (stdout);
	      }
	  }
	
	while (!alarmed)
	  {
	    if (!free_test_messages[size])
	      {
		gm_message_t *sent_message;
		
		while ( (sent_message = gm_get_sent_message ()) )
		  test_free_sent_message (sent_message);
		while (!free_test_messages[size])
		  {
		    gm_message_t *wake_message;

		    /* printf ("# sleeping..."); */
		    fflush (stdout);
		    gm_sleep ();
		    sleep_cnt++;
		    wake_message = gm_blocking_receive (&wake_message);
		    gm_always_assert (wake_message->type == gm_htons(gm_wake_type));
		    gm_wake ();
		    /* printf ("# woke.\n"); */
		    while ( (sent_message = gm_get_sent_message ()) )
		      test_free_sent_message (sent_message);
		    if (alarmed)
		      goto alarmed_abort;
		  }
	      }

	    free_test_messages[size]->gm_message.length =
	      gm_htonl (test_length);

	    gm_send (&free_test_messages[size]->gm_message, test_length,
		     TEST_PRIORITY);
	    send_cnt++;
	    free_test_messages[size] = free_test_messages[size]->next;
	    gm_assert (!free_test_messages[size] ||
		       free_test_messages[size]->gm_message.type);
	  }

    alarmed_abort:
	printf ("# Flushing network. \n");
	while (gm_globals.sleeping)
	  {
	    received_message = gm_blocking_receive (&received_buffer);
	    
	    if ( received_message->type == gm_htons (gm_wake_type) )
	      {
		gm_wake ();
		continue;
	      }
	    
	    abort ();
	  }

	sleep (1);

				/* Transfer all sent messages to host. */
	while (1)
	  {
	    gm_message_t *sent_message;

	    sent_message = gm_get_sent_message ();
	    if (!sent_message)
	      break;

	    {
	      Chainable_Test_Message *ctp;

	      ctp = (void *)sent_message - 8;
	      ctp->next = free_test_messages[size];
	      free_test_messages[size] = ctp;
	    }
	  }
	recv_cnt = send_cnt;
	
	{
	  float MBps, us;

	  us = 1000000.0 * (float)TEST_TIME / (float)recv_cnt;
	  MBps = (float)(test_length - sizeof(gm_message_t)) / us;

	  printf ("%d %f %f\n", test_length, us, MBps);
	  printf ("# test_length = %d, size = %d\n",
		  test_length, size);
	  printf ("# Received %d messages of payload size %d in %d seconds.\n",
		  recv_cnt, (int)(test_length - sizeof(gm_message_t)), TEST_TIME);

	  printf ("# TEST_MESSAGE_CNT = %d\n", TEST_MESSAGE_CNT);
	  printf ("#     GAP: %f us Bandwidth: %f MBps (%f Mbps)\n",
		  us, MBps, 8.0*MBps);
	      
	  printf ("#     sleep_cnt = %d\tnack_cnt = %d\tdrop_cnt = %d\n",
		  sleep_cnt, gm_ntohl (gm_globals.lanai->nack_cnt),
		  gm_ntohl (gm_globals.lanai->drop_cnt));
	  printf ("#     too_small_cnt = %d\t"
		  "bogus_header_cnt = %d\t"
		  "out_of_sequence_cnt = %d\n",
		  gm_ntohl (gm_globals.lanai->too_small_cnt),
		  gm_ntohl (gm_globals.lanai->bogus_header_cnt),
		  gm_ntohl (gm_globals.lanai->out_of_sequence_cnt));
	
	  gm_globals.lanai->too_small_cnt
	    = gm_globals.lanai->bogus_header_cnt
	    = gm_globals.lanai->out_of_sequence_cnt
	    = gm_globals.lanai->drop_cnt
	    = gm_globals.lanai->nack_cnt
	    = 0;
	}
      }
    }
  return 0;
}

#define GM_MAIN gm_bw
#include "gm_main.h"
