/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1999 by Myricom, Inc.                                   *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* This program uses undocumented software interfaces that are not
   intended for use in code written outside Myricom.  Hence the
   following undocumented includes. */

#include "gm_internal.h"
#include "myriInterface.h"
#include "myriProm.h"


static void
route_byte_string (unsigned char byte, char *route_string)
{
  int my_byte = (int) byte;

  if ((my_byte >= 0x80) && (my_byte <= 0x8F))
    {
      sprintf (route_string, "%d", my_byte & ~0x80);
      return;
    }

  if ((my_byte >= 0xB0) && (my_byte <= 0xBF))
    {
      /* negative route */
      sprintf (route_string, "%d", (my_byte - 0xC0));
      return;
    }

  sprintf (route_string, "%02x", my_byte);
}

#define CHECK_SIZING(type) do {						    \
  if (!GM_POWER_OF_TWO (sizeof (struct type))				    \
      && sizeof (struct type) != sizeof (struct type ## _fields))	    \
    {									    \
      printf								    \
	("sizeof " #type " struct is neither a power of two\n"		    \
	 "nor as small as possible, reducing performance and scalabity\n"); \
    }									    \
} while (0)

static void
usage (void)
{
  printf ("\nusage: gm_board_info [-B board] [-d] [-t]\n");

  printf ("-B #   specifies board number (default = 0)\n");
  printf ("-d     print route bytes in decimal\n");
  printf ("-t     print terse listing (default = verbose)\n");
  printf ("-help  print this help message\n");

  printf ("\n");
  gm_exit (GM_FAILURE);
}


static int
verbose_details_1(gm_port_t *port)
{
  gm_myrinet_eeprom_t eeprom;
  gm_status_t status;


  /* Copy the eeprom */
  status = _gm_get_eeprom (port, &eeprom);
  if (status != GM_SUCCESS)
    {
      gm_perror ("gm_board_info: error reading eeprom", status);
      return 1;
    }

  /* print the eeprom */
  printf ("  lanai_clockval    = 0x%08x\n",
	  (unsigned int) eeprom.lanai_clockval);
  printf ("  lanai_cpu_version = 0x%04x\n",
	  (unsigned int) eeprom.lanai_cpu_version);
  printf ("  lanai_board_id    = %02x:%02x:%02x:%02x:%02x:%02x\n",
	  eeprom.lanai_board_id[0] & 0xff,
	  eeprom.lanai_board_id[1] & 0xff,
	  eeprom.lanai_board_id[2] & 0xff,
	  eeprom.lanai_board_id[3] & 0xff,
	  eeprom.lanai_board_id[4] & 0xff,
	  eeprom.lanai_board_id[5] & 0xff);
  printf ("  lanai_sram_size   = 0x%08x (%dK bytes)\n",
	  (unsigned int) eeprom.lanai_sram_size,
	  (unsigned int) eeprom.lanai_sram_size >> 10);
  printf ("  fpga_version      = \"%s\"\n", eeprom.fpga_version);
  printf ("  more_version      = \"%s\"\n", eeprom.more_version);
  if (!((eeprom.max_lanai_speed == 0) ||
	(eeprom.max_lanai_speed == 0xFFFF)))
    printf ("  max_lanai_speed   = 0x%04x\n", eeprom.max_lanai_speed);
  printf ("  board_type        = 0x%04x (", eeprom.board_type);
  switch (eeprom.board_type)
    {
    case GM_MYRINET_BOARD_TYPE_1MEG_SRAM:
      printf ("GM_MYRINET_BOARD_TYPE_1MEG_SRAM)\n");
      break;
    case GM_MYRINET_BOARD_TYPE_FPGA:
      printf ("GM_MYRINET_BOARD_TYPE_FPGA)\n");
      break;
    case GM_MYRINET_BOARD_TYPE_L5:
      printf ("GM_MYRINET_BOARD_TYPE_L5)\n");
      break;
    case GM_MYRINET_BOARD_TYPE_NONE:
      printf ("GM_MYRINET_BOARD_TYPE_NONE)\n");
      break;
    default:
      printf ("unknown type)\n");
      break;
    }
  printf ("  bus_type          = 0x%04x (", eeprom.bus_type);
  switch (eeprom.bus_type)
    {
    case GM_MYRINET_BUS_UNKNOWN:
      printf ("GM_MYRINET_BUS_UNKNOWN)\n");
      break;
    case GM_MYRINET_BUS_SBUS:
      printf ("GM_MYRINET_BUS_SBUS)\n");
      break;
    case GM_MYRINET_BUS_PCI:
      printf ("GM_MYRINET_BUS_PCI)\n");
      break;
    case GM_MYRINET_BUS_GSC:
      printf ("GM_MYRINET_BUS_GSC)\n");
      break;
    case GM_MYRINET_BUS_FPGA:
      printf ("GM_MYRINET_BUS_FPGA)\n");
      break;
    case GM_MYRINET_BUS_NONE:
      printf ("GM_MYRINET_BUS_NONE)\n");
      break;
    default:
      printf ("unknown bus)\n");
    }
  printf ("  product_code      = 0x%04x\n",
	  (unsigned int) eeprom.product_code);
  printf ("  serial_number     = %d\n",
	  (unsigned int) eeprom.serial_number);
  if (PRODUCT_CODE_OK (eeprom.product_code))
    {
      printf ("    (should be labeled: \"");
      printf (boards[eeprom.product_code].name, eeprom.serial_number);
      printf ("\")\n");
    }
  else
    {
      if (isprint (eeprom.board_label[0])
	  && isprint (eeprom.board_label[1]))
	{
	  unsigned char boardname[32];
	  gm_bzero (boardname, 32);
	  strncpy ((char *) boardname, (char *) eeprom.board_label,
		   31);
	  printf ("    (should be labeled: \"%s\")\n", boardname);
	}
      else
		{
	  printf ("    (Unknown board label)\n");
	}
    }

  /* print the LANai time */

  {
    gm_u32_t high, low;
    gm_u64_t minutes;

    gm_u64_t ticks = gm_ticks (port);
    ticks = gm_ticks (port);
    high = (gm_u32_t) (ticks >> 32);
    low = (gm_u32_t) ticks;
    minutes = (ticks >> 27);	/* should be (ticks/60/2000000) */
    printf ("LANai time is 0x%0x%08x ticks,"
	    " or about %d minutes since reset.\n",
	    high, low, (int) minutes);
  }
  return (0);

} /* verbose_details_1() */


static void
verbose_details_2(gm_port_t *port)
{
  gm_pid_t opener_pids[GM_NUM_PORTS+1];
  int i;
  gm_pid_t my_pid = gm_getpid ();

  for (i = 0; i < GM_NUM_PORTS; i++)
    {
      opener_pids[i] = 0;
    }
  if (_gm_get_opener_pids (port, opener_pids) == GM_SUCCESS)
    {
      printf ("Port: Status  PID\n");
      for (i = 0; i < GM_NUM_PORTS; i++)
        {
	  if (opener_pids[i])
	    {
	      printf ("%4d: %6s %5d  ", i, "BUSY", opener_pids[i]);
	      if (opener_pids[i] == my_pid)
	        printf ("%s", "(this process [gm_board_info])");
	      if (opener_pids[i] == 1)
	        printf ("%s", "(ip-over-GM)");

	      printf ("\n");
	    }
/*
	  else
	    printf ("%4d: %6s\n", i, "free");
*/
	}
    }
}


static int
gm_board_info (int argc, char *argv[])
{
  int my_board_num = -1, board_num, port_num;
  gm_port_t *port;
  gm_status_t status;
  int found = 0;
  int hex = 1;
  int verbose = 1;
  int found_routes = 0;
  gm_u8_t mapper_id[8];
  unsigned int page_hash_cache_size=0;


  for (argc--, argv++; argc; argc--, argv++)
    {
      if (strcmp (*argv, "-d") == 0)
	{
	  hex = 0;
	  continue;
	}
      else if (strcmp (*argv, "-t") == 0)
	{
	  verbose = 0;
	  continue;
	}
      else if (strcmp (*argv, "--help") == 0)
	{
	  usage ();
	}
      else if (strcmp (*argv, "-B") == 0)
	{
	  argc--, argv++;
	  if (!argc)
	    {
	      printf ("Board number expected after '-B'.\n");
	      usage ();
	    }
	  if (sscanf (*argv, "%i", &my_board_num) != 1)
	    {
	      printf ("bad Board number: %s\n", *argv);
	      usage ();
	    }
	  continue;
	}
      else
	{
	  usage ();
	  break;
	}
    }

  printf ("GM build ID is \"%s.\"\n", _gm_build_id);

  /* Check some performance-related assumptions that do not effect
     correct operation of GM. */

  CHECK_SIZING (gm_port_protected_lanai_side);
  CHECK_SIZING (gm_connection);

  /* Scan for Myrinet interfaces and print their descriptions */
  for (board_num = 0; board_num < 4; board_num++)
    {
      if (my_board_num != -1)
	{
	  board_num = my_board_num;
	}
      /* Try to open a port */
      for (port_num = 0; port_num < GM_NUM_PORTS; port_num++)
	{
	  unsigned int node_id, max_node_id, max_node_id_inuse = 0;
	  unsigned int this_node = 0;
	  gm_u8_t my_name[GM_MAX_HOST_NAME_LEN + 1] = { 0 };
          int node_type;

	  if (_gm_open_common (&port, board_num, port_num, "gm_board_info",
			       GM_API_VERSION_1_0) != GM_SUCCESS)
	    continue;

	  found++;

	  printf ("\n\nBoard number %d:\n", board_num);
	  if (verbose)
	    {
	      if (verbose_details_1(port) != 0)
		{
		  gm_close (port);
	          return 1;
		}
	    }
	  
	  /* print the routes to the other nodes in the network */

	  if (gm_get_node_id (port, &this_node) != GM_SUCCESS)
	    fprintf (stderr, "Could not determine this node ID.\n");
	  gm_bzero (mapper_id, 6);
	  (void) gm_get_mapper_unique_id (port, (char *) mapper_id);
	  my_name[0] = 0;
	  (void) gm_get_host_name (port, (char *) my_name);
	  gm_get_node_type (port, &node_type);
	  printf ("This is node %d (%s)  node_type=%d\n",
		  (int) this_node, my_name, node_type);

	  if (!this_node)
	    {
	      printf ("   *** Node ID not set, mapper not yet run?\n");
	    }

	  status = gm_max_node_id (port, &max_node_id);
	  if (status != GM_SUCCESS)
	    {
	      fprintf (stderr, "Could not determine max node ID.\n");
	      gm_exit (status);
	    }

	  if (gm_max_node_id_inuse (port, &max_node_id_inuse) != GM_SUCCESS)
	    {
	      fprintf
		(stderr,
		 "\n******** Could not determine max_node_id_inuse. *****\n");
	      max_node_id_inuse = 16;
	      fprintf
		(stderr,
		 "******** Will only search for routes up to id = %d\n\n",
		 max_node_id_inuse);
	    }


	  _gm_get_page_hash_cache_size(port, &page_hash_cache_size);
	  printf ("Board has room for %d ports,  %d nodes/routes,  %d cache entries\n"
                  "          Port token cnt: send=%d, recv=%d\n",
		  gm_num_ports(port), max_node_id, page_hash_cache_size+1,
		  gm_num_send_tokens(port), gm_num_receive_tokens(port));

	  if (verbose)
	    verbose_details_2(port);
	  
	  printf ("Route table for this node follows:\n");
	  if (verbose)
	    printf ("The mapper 48-bit ID was:"
                    " %02x:%02x:%02x:%02x:%02x:%02x\n",
		    mapper_id[0], mapper_id[1], mapper_id[2],
		    mapper_id[3], mapper_id[4], mapper_id[5]);
	  printf
	    ("gmID MAC Address                               Hostname"
             " Route\n");
	  printf
	    ("---- ----------------- --------------------------------"
             " ---------------------\n");

	  for (node_id = 0; node_id <= max_node_id_inuse; node_id++)
	    {
	      char unique[6];
	      char *host_name;

	      /* Get and print the node IDs */
	      if (gm_node_id_to_unique_id (port, node_id, unique)
		  != GM_SUCCESS)
		{
		  printf ("  %4d ??:??:??:??:??:?? ", node_id);
		}
	      else if (unique[0] == 0 && unique[1] == 0 && unique[2] == 0 &&
		       unique[3] == 0 && unique[4] == 0 && unique[5] == 0)
		{
		  /* the whole thing is zero. */
		  continue;
		}
	      else
		{
		  unsigned int verify;
		  found_routes = 1;
		  printf ("%4d %02x:%02x:%02x:%02x:%02x:%02x ",
			  (int) node_id,
			  unique[0] & 0xff, unique[1] & 0xff,
			  unique[2] & 0xff, unique[3] & 0xff,
			  unique[4] & 0xff, unique[5] & 0xff);
		  status = gm_unique_id_to_node_id (port, unique, &verify);
		  if (status != GM_SUCCESS)
		    printf ("*** Could not verify unique mac id. ***");
		  else if (verify != node_id)
		    printf ("*** node_id->unique_id->node_id is 0x%x. ***",
			    (int) verify);
		}

#ifdef JOHN_DEBUG
	      max_node_id = 5;
#endif

	      /* print the hostname in a 32 character wide field, portably. */

	      host_name = gm_node_id_to_host_name (port, node_id);
	      {
		char *n;

		n = host_name ? host_name : "????";
		printf ("                                %s"
			+ strlen (n) % 32, n);
	      }

	      if (host_name)
		{
		  unsigned int verify;
		  verify = gm_host_name_to_node_id (port, host_name);
		  if (verify == GM_NO_SUCH_NODE_ID)
		    printf ("*** Could not verify host name id. ***");
		  else if (verify != node_id)
		    printf ("*** node_id->host_name->node_id is 0x%x. ***",
			    (int) verify);
		}

	      /* Print the route to the node, if any. */

	      {
		gm_u8_t route_buf[GM_MAX_NETWORK_DIAMETER];
		unsigned int route_len = GM_MAX_NETWORK_DIAMETER;
		int i;

		for (i = 0; i < GM_MAX_NETWORK_DIAMETER; i++)
		  {
		    route_buf[i] = -1;
		  }

		status = _gm_get_route (port, node_id, (char *) route_buf,
					&route_len);
		if (status != GM_SUCCESS)
		  {
		    printf ("*** Could not get route for node %d "
			    "(error 0x%x). ***\n",
			    (int) node_id, (int) status);
		    continue;
		  }

		if (route_len && (route_len != GM_NO_ROUTE))
		  {
		    int j = 0;
		    char bytestr[8];
		    while (route_len--)
		      {
			if (hex)
			  {
			    printf (" %02x", route_buf[j++]);
			  }
			else
			  {
			    route_byte_string (route_buf[j++], bytestr);
			    printf (" %s", bytestr);
			  }
		      }
		  }
	      }

	      if (node_id == this_node)
		printf (" (this node)");

	      if (((gm_u8_t) unique[0] == (gm_u8_t) mapper_id[0]) &&
		  ((gm_u8_t) unique[1] == (gm_u8_t) mapper_id[1]) &&
		  ((gm_u8_t) unique[2] == (gm_u8_t) mapper_id[2]) &&
		  ((gm_u8_t) unique[3] == (gm_u8_t) mapper_id[3]) &&
		  ((gm_u8_t) unique[4] == (gm_u8_t) mapper_id[4]) &&
		  ((gm_u8_t) unique[5] == (gm_u8_t) mapper_id[5]))
		{
		  printf (" (mapper)");
		}

	      printf ("\n");
	    }

	  if (!found_routes)
	    {
	      printf ("   *** No routes found *** \n");
	    }

	  /* close the port */
	  gm_close (port);
	  break;
	}
      /*
         if (port_num == 16) 
         break;
       */

      if (my_board_num != -1)
	{
	  break;
	}
    }
  if (!found)
    printf ("No boards found or all ports busy\n");

  return 0;
}

/****************************************************************
 * Entry point
 ****************************************************************/

#define GM_MAIN gm_board_info
#include "gm_main.h"

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
