/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1999 by Myricom, Inc.                                   *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* $Id: bridgeversion.c,v 1.3 2000/09/21 18:48:13 maxstern Exp $
 * author: maxstern@myri.com
 *
 * Description:
 *
 *    This program discovers and displays the Bridge ID register,
 *    along with the decoded Revision level of the Bridge chip.
 *
 * USAGE:
 *
 *    The following command would display the above information for
 *    every bridge chip in the system:
 *
 * find /hw -name controller -print | grep pci | xargs -l1 ./bridgeversion
 *
 *    In a test on an Origin at SGI, the above command produced the
 *    following output:
 *
 *       Device: /hw/module/1/slot/io8/xtalk_pci/pci/controller
 *       Bridge ID Reg == 0x4c00206d == "Rev.D"
 *
 *       Device: /hw/module/1/slot/io7/hippi_serial/pci/controller
 *       Bridge ID Reg == 0x4c00206d == "Rev.D"
 *
 *       Device: /hw/module/1/slot/io1/baseio/pci/controller
 *       Bridge ID Reg == 0x4c00206d == "Rev.D"
 *
 *       Device: /hw/module/1/slot/io4/menet/pci/controller
 *       Bridge ID Reg == 0x4c00206d == "Rev.D"
 *
 *       Device: /hw/module/1/slot/io2/pci_xio/pci/controller
 *       Bridge ID Reg == 0x3c00206d == "Rev.C"
 *
 *    In the above output, "pci_xio" is a Shoebox, and "xtalk_pci"
 *    is a Shoehorn.  Note that the Shoebox above is not useable
 *    with Myrinet, because its Revision Id is not 'D' or greater.
 *
 * NOTES:
 *
 *    Notwithstanding the Copyright and author notices above, this
 *    program actually originated at SGI in Mountain View, and was
 *    provided to us by Rob Warnock there.
 *
 *    The bridge version is found in the high-order four bits of the 
 *    "Bridge ID register."  That register, in turn, is found at
 *    offset 0 from the configuration space of the 'controller'
 *    vertex of the hwgraph.
 */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/systeminfo.h>

#define  MACHINE_COUNT 16


static void usage()
{
   fprintf(stderr,
           "usage: bridgeversion /hw/{...}/pci/controller\n");
   exit(1);
}

void main(int argc, char **argv)
{
   static char *prefix;
   static char suffix[] = "/pci/controller";
   int         arglen,
               fd;
   __uint32_t  id_register;
   void        *cfg_space;
   char        machine[MACHINE_COUNT];
   int         machine_count;

   machine_count = sysinfo(SI_MACHINE, machine, MACHINE_COUNT);

   if (machine_count > MACHINE_COUNT)
      printf ("WARNING: Machine id %s is truncated from %d characters\n",
	      machine, machine_count);

   if (strcmp(machine, "IP27") == 0)       /* Origin                */
      /* Typical path: /hw/module/1/slot/io2/pci_xio/pci/controller */
      prefix = "/hw/module/";
   else if (strcmp(machine, "IP30") == 0)  /* Octane                */
      /* Typical path: /hw/node/xtalk/15/pci/controller             */
      prefix = "/hw/node/";
   else if (strcmp(machine, "IP32") == 0)  /* O2                    */
      /* Typical path: /hw/node/ip/pci/controller                   */
      prefix = "/hw/node/";
   else
      prefix = "/hw/";
   printf("Machine: %s; path prefix: %s\n", machine, prefix);

   if (2 != argc ||
       (arglen = strlen(argv[1])) < strlen(prefix) + strlen(suffix) ||
       strncmp(argv[1], prefix, strlen(prefix)) ||
       strncmp(argv[1] + arglen - strlen(suffix), suffix, strlen(suffix)))

      usage();			/* Function will exit */

   if (0 > (fd = open(argv[1], O_RDONLY)))
   {
      fprintf(stderr, "Couldn't open %s for reading!\nReason: %s\n",
              argv[1], strerror(errno));
      exit(1);
   }

   if (MAP_FAILED ==
           (cfg_space = mmap(0, 0x100000, PROT_READ, MAP_SHARED, fd, 0)))
   {
      fprintf(stderr, "Couldn't mmap %s\nReason: %s\n",
       	      argv[1], strerror(errno));
      exit(1);
   }

   close(fd); 	/* (don't need it any more) */

   /* "Danger, Will Robinson, Danger!!" */
   id_register = *(__uint32_t *)(((char *)cfg_space) + 4);
   printf("Device: %s\nBridge ID Reg == 0x%8.8x == \"Rev.%c\"\n\n",
	   argv[1], id_register, "?ABCDEFGHIJKLMNO"[id_register >> 28]);

   exit(0);
}
