/*
  ScrolledText.java
  finucane@myri.com
  May, 1998

  Get the latest scoop package from
  http://www.myri.com/staff/finucane

*/


import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;
import java.io.*;
import java.util.*;

public class ScrolledText extends Panel implements MouseListener
{
  private static final int MARGIN = 10;
  private static final int DESCENTS_BETWEEN_LINES = 0;
  private int linePadding;
  protected TextCanvas canvas;
  private Scrollbar vertical;
  protected BigString text;
  private boolean untouched;
  private PageListener pageListener;
  protected int selectionStart;
  protected int selectionEnd;
  
  protected class TextCanvas extends Canvas implements AdjustmentListener, KeyListener, MouseMotionListener, ClipboardOwner
  {    
    public Image image;
    private int downKey = 0;
    private Dragger dragger;
    private Clipboard clipboard = getToolkit().getSystemClipboard ();
    private PersistentStringSelection stringSelection = new PersistentStringSelection ("");
    public void update (Graphics g)
    {
      paint (g);
      setHeader ();
    }
    private synchronized void paintOffscreen (Graphics g)
    {
      g.setColor (Color.black);
      Rectangle r = getBounds ();
      g.drawRect (0, 0, r.width -1, r.height -1);  

      int nLines = Math.min (getNumLines (), getNumLinesPerPage ());
      int y = getLineHeight ();
      int start = getCurrentLine ();
      
      int a = Math.min (selectionStart, selectionEnd);
      int b = Math.max (selectionStart, selectionEnd);
      
      for (int i = 0; i < nLines; i++)
      {
	String s = text.getLine (i + start);
	
	int pa = text.getPositionOfLine (i + start);
	int pb = pa + s.length ();
	
	int d1, d2, d3;
	d1 = 0;
	d2 = 0;
	d3 = 0;

	if (a >= pa && b <= pb)
	{
	  d1 = a - pa;
	  d2 = b - a;
	  d3 = pb - b;
	}
	else if (a >= pa && a < pb)
	{
	  d1 = a - pa;
	  d2 = s.length() - d1;
	}
	else if (b >= pa && b < pb)
	{
	  d3 = pb - b;
	  d2 = s.length () - d3;
	}
	else if (pa >= a && pb <= b)
	  d2 = s.length ();
	else d1 = s.length ();
	
	String s1 = s.substring (0, d1);
	String s2 = s.substring (d1, d1 + d2);
	String s3 = s.substring (d1 + d2, s.length());
	
	g.drawString (s1, MARGIN, y*(i+1));

	if (s2.length () > 0)
	{  
	  g.setColor (Color.black);
	  g.fillRect (MARGIN + text.stringWidth (s1),
		      y*(i) + y/3, text.stringWidth (s2), y);
	  g.setColor (Color.white);
	  g.drawString (s2, MARGIN + text.stringWidth (s1), y*(i+1));
	  g.setColor (Color.black);
	}
	g.drawString (s3, MARGIN + text.stringWidth (s1 + s2), y*(i+1));
      }
    }
    
    public synchronized void paint (Graphics g)
    {
      if (image == null)
	image = createImage (getBounds().width, getBounds().height);
      Graphics go = image.getGraphics ();

      go.setColor (Color.white);
      Rectangle r = getBounds ();
    
      go.fillRect (0, 0, image.getWidth (null), image.getHeight (null));
    
      paintOffscreen (go);
      g.drawImage (image, 0,0, null);
    }
    int getLineHeight ()
    {
      FontMetrics fm = getFontMetrics (getFont ());
      return fm.getDescent() * (DESCENTS_BETWEEN_LINES + 1) + fm.getMaxAscent();
    }
    public int getNumLinesPerPage ()
    {
      return getBounds().height / getLineHeight ();
    }
    public synchronized void invalidate ()
    {
      image = null;
      int pos = getCurrentChar ();
      text.setWidth (getBounds().width - 2 * MARGIN);
      setValues (getLineOfChar (pos), getNumLinesPerPage(), 0, getNumLines());
      super.invalidate ();
      setHeader ();
    }
    public void adjustmentValueChanged (AdjustmentEvent e)
    {
      untouched = false;
      repaint ();
    }
    public void keyReleased (KeyEvent e)
    {
    }
    public void keyPressed (KeyEvent e)
    {
      
      switch (e.getKeyCode ())
      {
	case KeyEvent.VK_PAGE_UP:
	  ScrolledText.this.move (getCurrentLine() - getNumLinesPerPage() + 1);
	  break;
	case KeyEvent.VK_SPACE:
	case KeyEvent.VK_PAGE_DOWN:
	  ScrolledText.this.move (getCurrentLine() + getNumLinesPerPage() - 1);
	  break;
	case KeyEvent.VK_UP:
	  ScrolledText.this.move (getCurrentLine() - 1);
	  break;
	case KeyEvent.VK_DOWN:
	  ScrolledText.this.move (getCurrentLine() + 1);
	  break;
	case KeyEvent.VK_N:
	  ScrolledText.this.move (getLineOfPage (getCurrentPage () + 2));
	  break;
	case KeyEvent.VK_P:
	  ScrolledText.this.move (getLineOfPage (getCurrentPage () - 2));
	  break;
	case KeyEvent.VK_HOME:
	  ScrolledText.this.move (0);
	  break;
	case KeyEvent.VK_END:
	  ScrolledText.this.move (getNumLines());
	  break;
	case KeyEvent.VK_C:
	  if (e.isControlDown ())
	    copy();
	  break;
	default:
	  return;
      }
    }
    private synchronized void selectWord (int pos)
    {
      int line = text.getLineAt (pos);
      String s = text.getLine (line);
      int start = text.getPositionOfLine (line);
      int offset = pos - start;
      
      if (offset >= s.length ())
	offset = s.length () - 1;
      
      if (offset < 0)
	offset = 0;
      
      if (s.length () == 0)
      {
	//1.1.6 compiler crash if on one line
	selectionEnd = 0;
	selectionStart = 0;
	repaint ();
	return;
      }
      
      int a;
      int b;

      for (a = offset; a >= 0; a--)
	if (s.charAt (a) == ' ' ||
	    s.charAt (a) == '"')
	{
	  if (offset != s.length () - 1)
	    a++;
	  break;
	}
      
      for (b = offset; b < s.length (); b++)
	if (s.charAt (b) == ' ' ||
	    s.charAt (b) == '.' ||
	    s.charAt (b) == '"' ||
	    s.charAt (b) == ',' ||
	    s.charAt (b) == ':' ||
	    s.charAt (b) == ';' ||
	    s.charAt (b) == '+' ||
	    s.charAt (b) == '?')
	  break;

      selectionStart = start + a;
      selectionEnd = start + b;
      repaint ();
    }
    private synchronized void selectLine (int pos)
    {
      int line = text.getLineAt (pos);
      String s = text.getLine (line);
      selectionStart = text.getPositionOfLine (line);
      selectionEnd = selectionStart + s.length ();
      repaint ();
    }
    private synchronized void selectPage (int pos)
    {
      int page = text.getPageAt (pos);
      int startLine = getLineOfPage (page);
      int endLine;

      if (page == getNumPages () - 1)
	endLine = getNumLines () -1;
      else
	endLine =  getLineOfPage (page + 1) - 1;
      
      String s = text.getLine (endLine);
      selectionStart = text.getPositionOfLine (startLine);
      selectionEnd = text.getPositionOfLine (endLine) + s.length ();
      repaint ();
    }
    protected String getSelectedString ()
    {
      int a = Math.min (selectionStart, selectionEnd);
      int b = Math.max (selectionStart, selectionEnd);

      return text.substring (a, b);
    }
    private void copy ()
    {
      int a = Math.min (selectionStart, selectionEnd);
      int b = Math.max (selectionStart, selectionEnd);
      
      clipboard.setContents(new StringSelection (text.getFilledString (a, b)), this);
    }
    public void lostOwnership (Clipboard clipboard, Transferable contents)
    { 
    }
    public void keyTyped (KeyEvent e)
    {
    }
    private class Dragger extends Thread
    {
      public int travel;
      public int x;
      public int y;
      
      public Dragger ()
      {
	travel = 0;
      }
      public void run ()
      {
	for (;;)
	{
	  while (travel != 0)
	  {
	    ScrolledText.this.move (getCurrentLine() + travel);
	    selectionEnd = getPositionOfPoint (x, y);
	    try
	    {
	      sleep (100);
	    }
	    catch (java.lang.InterruptedException e)
	    {
	      System.out.println (e);
	    }
	  }
	  suspend ();
	}
      }
    }
    public synchronized void mouseDragged (MouseEvent e)
    {
      Rectangle r = getBounds ();
      int travel = 0;
      int x = e.getX();
      int y = e.getY();

      if (y < 0)
	travel = y;
      else if (y > r.height)
	travel = y - r.height;

      travel /= getLineHeight();
      selectionEnd = getPositionOfPoint (x, y);

      dragger.travel = travel;
      dragger.x = x;
      dragger.y = y;
      if (travel != 0)
	dragger.resume ();
      
      repaint ();
    }      
    public void mousePressed (MouseEvent e)
    {
      /*1.1.6 compiler bug if on one line*/
      selectionEnd  = canvas.getPositionOfPoint (e.getX(), e.getY());
      selectionStart = selectionEnd;
      
      repaint ();
      requestFocus();
    }
    public void mouseReleased (MouseEvent e)
    {
      dragger.travel = 0;
    }
    public void mouseClicked (MouseEvent e)
    {
      int clicks =  e.getClickCount ();
      if (clicks == 2)
	selectWord (getPositionOfPoint (e.getX (), e.getY ()));
      else if (clicks == 3)
	selectPage (getPositionOfPoint (e.getX(), e.getY()));
    }
    public void mouseEntered (MouseEvent e)
    { 
    }
    public void mouseExited (MouseEvent e)
    { 
    }
    public void mouseMoved (MouseEvent e)
    {
    }
    protected synchronized int getPositionOfPoint (int x, int y)
    {
      int start = getCurrentLine ();
      int lineHeight = getLineHeight ();
      
      int line =  start + y / lineHeight;
      
      int numLines = getNumLines ();
      if (line >= numLines)
	line = numLines - 1;
      if (line < 0) line = 0;
      
      return text.getPositionOfLine (line) + text.getPositionInLine (line, x);      
    }   
    public TextCanvas (Font font, MouseListener mouseListener)
    {
      dragger = new Dragger ();
      dragger.start ();
      
      this.setFont (font);
      addKeyListener (this);
      addMouseListener (mouseListener);
      addMouseMotionListener (this);
    }
  } 
  public int getCurrentChar ()
  {
    return text.getPositionOfLine (getCurrentLine());
  }
  public int getLineOfChar (int position)
  {
    return text.getLineAt (position);
  }
  public int getCurrentPage ()
  {
    return text.getPageAt (1 + getCurrentChar());
  }
  public int getLineOfPage (int p)
  {
    p = Math.max (p, 0);
    p = Math.min (p, getNumPages() - 1);

    return text.getLineOfPage (p);
  }
  protected synchronized void move (int line)
  {
    untouched = false;
    
    line = Math.max (line, 0);
    line = Math.min (line, getNumLines() - 1);

    setValues (line, canvas.getNumLinesPerPage(), 0, getNumLines());
    canvas.repaint ();
  }
  
  public synchronized int findPosition (Vector v, int position)     
  {
    int a = 0;
    int b = v.size();
    while (a < b - 1)
    {
      int c = (a + b) / 2;
      
      int p =  ((Integer)v.elementAt (c)).intValue ();
      if (position < p)
	b = c;
      else if (position > p)
	a = c;
      else return c;
    }
    return a;  
  }
  public synchronized int getCurrentLine ()
  {
    return vertical.getValue ();
  }
  public synchronized int getNumLines ()
  {
    return text.getNumLines ();
  }
  public synchronized void appendText (String s)
  {
    boolean reposition = (getCurrentLine () +
			  canvas.getNumLinesPerPage () == getNumLines () ||
			  untouched && getCurrentLine () == 0);
    
    text.append (s);
    
    setValues (reposition ? getNumLines() : getCurrentLine (), canvas.getNumLinesPerPage(), 0, getNumLines());

    canvas.repaint();
    
  }
  protected synchronized void setValues (int v, int p, int min, int max)
  {
    vertical.setValues (v, p, min, max);
    vertical.setBlockIncrement (canvas.getNumLinesPerPage () - 1);
  }
  public void dump (String filename)
  {
    text.toFile (filename);
  }
  public void setText (String s)
  {
    untouched = true;
    selectionEnd = selectionStart = 0;
    
    text.empty ();
    text.append (s);
    
    setValues (0, canvas.getNumLinesPerPage(), 0, getNumLines());
    canvas.repaint();
  }
  
  public int getNumPages ()
  {
    return text.getNumPages ();
  }
  
  public void setHeader ()
  {
    pageListener.setHeader (getCurrentLine (), getNumLines (), getCurrentPage (), getNumPages());
    
  }
  public synchronized void setFont (Font font, int pad)
  {
    int pos = getCurrentChar ();
    linePadding = pad;
    setFont (font);
    canvas.setFont (font);
    text.setFont (font);
    setValues (getLineOfChar (pos), canvas.getNumLinesPerPage(), 0, getNumLines());
    canvas.invalidate ();
    canvas.repaint (); 
  }
  public void mousePressed (MouseEvent e)
  {
    canvas.mousePressed (e);
  }
  public void mouseReleased (MouseEvent e)
  {
    canvas.mouseReleased (e);
  }
  public void mouseClicked (MouseEvent e)
  {
    canvas.mouseClicked (e);
  }
  public void mouseEntered (MouseEvent e)
  { 
  }
  public void mouseExited (MouseEvent e)
  { 
  }
  public ScrolledText (Font font, PageListener pageListener)
  {
    this.pageListener = pageListener;
    
    linePadding = 2;
    text = new BigString (64, 60, font);
    setFont (font);
    
    setLayout (new BorderLayout ());
    add ("Center", canvas = new TextCanvas (font, this));
    add ("East", vertical = new Scrollbar (Scrollbar.VERTICAL));    
    vertical.addAdjustmentListener (canvas);
    vertical.setValue (0);
    
    untouched = true;
    setText ("");
    setHeader ();
  }
}



