/*
  timeout.c
  demonstrate my timer problem
  finucane@myri.com (David Finucane)
*/

#include <stdlib.h>
#include <stdio.h>
#include "gm.h"
#include "gm_internal.h"

#define die(s){printf s; printf ("\n"); exit (1);}
#define comment(e){if (trace)printf e;}
#define MAX_SENDS 40
#define MTU 4000


static void send (struct gm_port*port, char*p, int length)
{
  int routeLength = 1;
  int i;
  
  p [0] = 3;
  
  for (i = 0; i < routeLength; i++)
    p [i] = (p [i] & '\x3f') | '\x80';

  _gm_raw_send (port, p, length, routeLength);
}

static int waiting;
static gm_alarm_t the_alarm;
static int trace = 0;

static void alarm_hook (void*)
{
  waiting = 0;
}


static void set_timer (struct gm_port*port, int microseconds)
{
  waiting = 1;
  gm_set_alarm (port, &the_alarm, microseconds, alarm_hook, 0);
}

static void receive (struct gm_port*port, int*pendingSends)
{
  gm_recv_event_t*event;
  event = gm_blocking_receive_no_spin (port);

  switch (GM_RECV_EVENT_TYPE (event))
  {
    case GM_SENT_EVENT:
      comment (("GM_SENT_EVENT %d\n", GM_RECV_EVENT_TYPE (event)));
      (*pendingSends)--;
      return;
    case GM_RAW_RECV_EVENT:
      comment (("GM_RAW_RECV_EVENT %d\n", GM_RECV_EVENT_TYPE (event)));
      return;
    case GM_FAST_RECV_EVENT:
    case GM_FAST_HIGH_RECV_EVENT:
    case GM_FAST_PEER_RECV_EVENT:
    case GM_FAST_HIGH_PEER_RECV_EVENT:
      comment (("GM_RAW_RECV_EVENT %d\n", GM_RECV_EVENT_TYPE (event)));
      return;
    case GM_RECV_EVENT:
    case GM_HIGH_RECV_EVENT:
    case GM_PEER_RECV_EVENT:
    case GM_HIGH_PEER_RECV_EVENT:
      comment (("GM_RECV_EVENT %d\n", GM_RECV_EVENT_TYPE (event)));
      return;
    case GM_NO_RECV_EVENT:
      comment (("GM_NO_RECV_EVENT %d\n", GM_RECV_EVENT_TYPE (event)));
      return;
    case GM_ALARM_EVENT:
      comment (("GM_ALARM_EVENT %d\n", GM_RECV_EVENT_TYPE (event)));
      waiting = 1;
      gm_unknown (port, event);
      comment (("waiting is %d\n", waiting));
      return;
    default:
      comment (("default %d\n", GM_RECV_EVENT_TYPE (event)));
      gm_unknown (port, event);
      return;
  }
}

int run (int unit, int numSends, int count, int size, int microseconds)
{
  struct gm_port*port;
  int i;
  int pendingSends;
  char*sends [MAX_SENDS];

  if (_gm_mapper_open (&port, unit, GM_API_VERSION_1_0) != GM_SUCCESS)
    die (("couldn't open mapper port on unit %d\n", unit));

  gm_initialize_alarm (&the_alarm);
  
  for (i = 0; i < numSends; i++)
    if (!(sends [i] = (char*) gm_dma_malloc (port, MTU)))
      die (("gm_dma_malloc failed"));

  trace = 0;
  
  pendingSends = 0;
  for (i = 0; i < count; i++)
  {
    comment (("sent a message\n"));
    send (port, sends [i % numSends], size);
    pendingSends++;

    //printf ("setting timer\n");
    set_timer (port, microseconds);
     
    while (waiting && pendingSends)
    {
      receive (port, &pendingSends);
      if (waiting && !pendingSends)
      {
	comment (("cancelling alarm\n"));
	gm_cancel_alarm (&the_alarm);
      }
    }
    
    if (pendingSends == numSends)
      die (("timed out waiting for send done"));
    
  }

  gm_close (port);

  comment (("opening the port again\n"));
  
  if (_gm_mapper_open (&port, unit, GM_API_VERSION_1_0) != GM_SUCCESS)
    die (("couldn't open mapper port on unit %d\n", unit));

  comment (("initializing the timer again, this doesn't seem to matter.\n"));
  gm_initialize_alarm (&the_alarm);

  comment (("sleeping for %d microseconds \n", microseconds));
  comment (("the problem is, that this timer never goes off.\n"));
  
  set_timer (port, microseconds);
  while (waiting)
    receive (port, &pendingSends);

  comment (("timers worked\n"));
  gm_close (port);

  return 1;
}


int main (int argc, char*argv [])
{
  int unit, count, size, numSends, microseconds;

  if (argc != 6)
    die (("usage %s <unit> <buffers> <count> <size> <micros>", argv [0]));

  unit = atoi (argv [1]);
  numSends = atoi (argv [2]);
  count = atoi (argv [3]);
  size = atoi (argv [4]);
  microseconds = atoi (argv [5]);

  if (size > MTU)
    die (("size %d is greater than MTU %d\n", size, MTU));

  if (numSends <= 0 || numSends > MAX_SENDS)
    die (("buffers %d is 0 or  greater than max sends %d\n", numSends, MAX_SENDS));

  run (unit, numSends, count, size, microseconds);
  return 0;
}

