/*
  simulator.c
  map tools
  dmazzoni@myri.com (Dominic Mazzoni)
*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>

#include "mt_Args.h"
#include "mt_Simulation.h"
#include "mt_MapFile.h"
#include "sm_Simulator.h"

#define die(s){mt_Component::printFormat s; exit (1);}
#define insist(e) do { if (!(e)) {mt_Component::printFormat ("Assertion failed in file %s, line %d",__FILE__,__LINE__); goto exception;}} while (0)
#define insistp(e, s) do { if (!(e)) {mt_Component::printFormat s; goto exception;}} while (0)

static void printLine (char*s)
{
  printf ("%s\n", s);
}

sm_Simulator* simulator = 0;
  
static void handleSignal (int)
{
  if (simulator)
    simulator->abort ();
}

int main (int argc, char *argv[]) 
{
  mt_Component::initialize (printLine);

  mt_Simulation*simulation = 0;
  mt_Calculator*calculator = 0;
  mt_Args*args = 0;

  simulator = (sm_Simulator*) mt_getNetwork ();
  if (!simulator)
    die (("couldn't create simulator"));

  simulation = mt_getSimulation ();
  if (!simulation)
    die (("mt_getSimulation failed"));

  calculator = mt_getCalculator ();
  if (!calculator)
    die (("mt_getCalculator failed"));
  
  args = new mt_Args (argc, argv);
  if (argc == 1 || !args)
  {
    printf("usage: %s ",argv[0]);
    mt_Args::usage ();
    printf("\n");

    calculator->usage();
    printf("\n");
    simulation->usage();
    printf ("\n");
    
    simulator->usage();
    printf("\n");
    return 0;
  }

  signal (SIGINT, handleSignal);
  signal (SIGTERM, handleSignal);

  if (!(simulator->parseArgs (args)))
  {
    simulator->usage ();
    return 1;
  }

  if (!(simulation->parseArgs (args)))
  {
    simulation->usage ();
    return 1;
  }

  if (!(calculator->parseArgs (args)))
  {
    calculator->usage ();
    return 1;
  }

  if (!simulator->initialize (calculator, simulation))
      die (("couldn't initialize simulator"));

  int numPackets = simulator->getMaxPackets ();
  if (numPackets != -1)
    printf("%d/%d",0, numPackets);

  while (simulator->step (10000))
  {
    char s [mt_File::MAX_LINE];
    simulator->inspectQueue (s);
    
    if (numPackets >= 0)
      fprintf(stderr, "\r%d/%d U:%5.4f L:%d T:%lld Q:%d %s",
	      simulator->getNumReceived(),
	      numPackets,
	      simulator->getUtilization (),
	      simulator->getAverageLatency (),
	      simulator->getCurrentTime (),
	      simulator->getEventQueueSize (),
	      s);
    else
      fprintf (stderr, "\rU:%5.4f L:%d  T:%lld Q:%d %s",
	       simulator->getUtilization (),
	       simulator->getAverageLatency (),
	       simulator->getCurrentTime (),
	       simulator->getEventQueueSize (),
	       s);
    
    fflush(stdout);
  }

  if (numPackets >= 0)
    printf("\r%d/%d U:%5.4f L:%d   ",
	   simulator->getNumReceived(),
	   numPackets,
	   simulator->getUtilization(),
	   simulator->getAverageLatency());
  else
    printf("\rU:%5.4f L:%d   ",
	   simulator->getUtilization(),
	   simulator->getAverageLatency());
  
  printf("\n\n");

  simulation->simpleDump (simulator, stdout);
  simulation->dump (simulator);
  simulator->simpleDump (stdout);
  simulator->dump ();

  return 0;
}
