/*
  What would your gracious figure?
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>

#include "mt_Switch.h"
#include "mt_Calculator.h"
#include "mt_Follower.h"

#include "insist.h"

static void printLine (char*s)
{
  printf ("%s\n", s);
}

class si_Node : mt_Component
{
  private:
  int in [mt_Switch::NUM_PORTS];
  int out [mt_Switch::NUM_PORTS];
  mt_Node*node;

  public:
  si_Node ();
  mt_Node*getNode ();
  int setNode (mt_Node*n);
  int addIn (int port);
  int addOut (int port);
  int printResults ();
};

si_Node::si_Node ()
{
  node = 0;
  for (int i = 0; i < mt_Switch::NUM_PORTS; i++)
    in [i] = out [i] = 0;
}
int si_Node::printResults ()
{
  insist (this);
  
  for (int i = 0; i < mt_Switch::NUM_PORTS; i++)
  {
    if (in [i])
      printFormat ("%s port %d had %d, link to %s", node->getName (), i, in [i], node->getNode (i)->getName ());
  }
  return 1;
  exception: return 0;
}

mt_Node*si_Node::getNode ()
{
  insist (this);
  return node;
  exception: return 0;
}

int si_Node::setNode (mt_Node*n)
{
  insist (this);
  insist (n);
  node = n;
  return 1;
  exception: return 0;
}

int si_Node::addIn (int port)
{
  insist (this);
  insist (port >= 0 && port < mt_Switch::NUM_PORTS);
  in [port]++;
  return 1;
  exception:return 0;
}

int si_Node::addOut (int port)
{
  insist (this);
  insist (port >= 0 && port < mt_Switch::NUM_PORTS);
  out [port]++;
  return 1;
  exception:return 0;
}


class si_Sift : public mt_Component, public mt_Follower
{
  private:
  
  char mapFilename [mt_File::FILENAME_LENGTH];	   
  char routeFilename [mt_File::FILENAME_LENGTH];
  char errorFilename [mt_File::FILENAME_LENGTH];
  mt_Calculator*calculator;
  int*errors;
  int numErrors;
  si_Node*nodes;
  int numNodes;
  
  public:
  si_Sift ();
  ~si_Sift ();
  void usage ();
  int parseArgs (int argc, char*argv []);
  int parseErrorFile (mt_Graph*g, char*filename, int*errors);
  int initialize ();
  int sift ();
  si_Node*getNode (mt_Node*n);
  int printResults ();
  virtual int step (mt_Node*node, int port);
};


si_Node*si_Sift::getNode (mt_Node*n)
{
  insist (this);
  insist (nodes && numNodes > 0);
  for (int i = 0; i < numNodes; i++)
    if (nodes [i].getNode () == n)
      return &nodes [i];
  exception: return 0;
}

int si_Sift::printResults ()
{
  insist (this);
  insist (nodes && numNodes > 0);
  for (int i = 0; i < numNodes; i++)
    nodes [i].printResults ();
  exception: return 0;
}

int si_Sift::parseErrorFile (mt_Graph*g, char*filename, int*errors)
{
  insist (this);
  insist (g && filename);
  
  int numErrors, error;
  mt_Tokenizer*tokens;
  
  tokens = new mt_Tokenizer (filename, "(), \n\r\t", 255, "#;");
  insist (tokens);

  numErrors = 0;
  
  while (tokens->getNext ())
  {
    if (!tokens->isInteger ())
      break;

    if (!g->getGmId (error = tokens->getInteger ()))
    {
      printFormat ("GM ID %d is not in the network", error);
      break;
    }
    if (errors)
      errors [numErrors] = error;
    numErrors++;
  }

  if (numErrors % 2)
  {
    printFormat ("There is an odd number of GM ids (%d). This must not be.", numErrors);
    delete tokens;
    return 0;
  }
  
  if (!tokens->getNext ())
  {
    delete tokens;
    return numErrors;
  }
  
  printFormat ("line: %d. Syntax error near the text \"%s\".", tokens->getLineNumber (), tokens->getWord ());
  delete tokens;
  exception: return 0;
}


int si_Sift::initialize ()
{
  mt_Args args (mt_Args::CALCULATOR, 2);
  
  args.addArg ("-routes-file");
  args.addArg (routeFilename);
  
  calculator = mt_getCalculator ();
  insist (calculator);
  
  if (!calculator->parseArgs (&args) || !calculator->initialize (mapFilename))
    return 0;

  if (!(numErrors = parseErrorFile (calculator, errorFilename, 0)))
    return 0;
  
  insist (numErrors > 0);
  
  if (errors) free (errors);
  errors = (int*) malloc (sizeof (int) * numErrors);
  insistp (errors, ("alloc failed\n"));

  numErrors = parseErrorFile (calculator, errorFilename, errors);
  insist (numErrors);

  if (nodes) delete [] nodes;
  numNodes = calculator->getNumSwitches ();
  insist (numNodes > 0);
  
  nodes = new si_Node [numNodes];
  insistp (nodes, ("couldn't alloc %d nodes\n", nodes));

  for (int i = 0; i < numNodes; i++)
    nodes [i].setNode (calculator->getSwitch (i));
  
  return 1;
  exception: return 0;
}

int si_Sift::sift ()
{
  insist (this);
  insist (calculator);
  insist (errors && numErrors && !(numErrors % 2));
  
  for (int i = 0; i < numErrors; i+=2)
  {
    mt_Route r;
    mt_Node*n;
    int in;
    
    mt_Node*from = calculator->getGmId (errors [i]);
    mt_Node*to = calculator->getGmId (errors [i + 1]);
    
    insist (from && to);
    calculator->getRoute (from->getTypeIndex (), to->getTypeIndex (), 0, &r);
    from->follow (&r, &n, &in, this);
    insist (n == to);
  }
  return 1;
  exception: return 0;
}

si_Sift::~si_Sift ()
{
  if (calculator) delete calculator;
  if (errors) delete errors;
}

si_Sift::si_Sift ()
{
  *mapFilename = *routeFilename = *errorFilename = 0;
  calculator = 0;
  errors = 0;
  numErrors = 0;
  nodes = 0;
}

void si_Sift::usage ()
{
  printf ("usage: -map-file <mapfile> -route-file <routefile> -error-file <errorfile>\n");
}

int si_Sift::parseArgs (int argc, char*argv [])
{
  insist (this);
  insist (argv);

  for (int i = 1; i < argc; i++)
  {
    if (!strcmp (argv[i], "-map-file"))
    {
      if (++i > argc)
      {
	printf ("expected map-file\n");
	return 0;
      }
      strncpy (mapFilename, argv [i], mt_File::FILENAME_LENGTH);
    }
    else if (!strcmp (argv[i], "-route-file"))
    {
      if (++i > argc)
      {
	printf ("expected route-file\n");
	return 0;

      }
      strncpy (routeFilename, argv [i], mt_File::FILENAME_LENGTH);
    }
    else  if (!strcmp (argv[i], "-error-file"))
    {
      if (++i > argc)
      {
	printf ("expected error-file\n");
	return 0;
      }
      strncpy (errorFilename, argv [i], mt_File::FILENAME_LENGTH);
    }
    else
    {
      printf ("unknown argument %s\n", argv [i]);  
      return 0;
    }
  }
  return 1;
  exception : return 0;
}


int si_Sift::step (mt_Node*node, int port)
{
  insist (node);
  
  if (node->isSwitch ())
  {
    si_Node*n = getNode (node);
    n->addIn (port);
  }
  return 1;
  exception: return 0;
}

int main (int argc, char*argv[])
{
  mt_Component::initialize (printLine);

  si_Sift sift;
  
  if (!sift.parseArgs (argc, argv))
  {
    sift.usage ();
    return 0;
  }
  if (!sift.initialize ())
    return 0;

  if (!sift.sift ())
    return 0;
  
  sift.printResults ();
  return 0;
}


