/*
  sandia2myricom.c
  to convert sandia style routes files to myricom routes
  finucane@myri.com
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "mt_Component.h"
#include "mt_MapFile.h"
#include "mt_Graph.h"
#include "mt_Calculator.h"
#include "mt_Args.h"
#include "mt_RouteTable.h"


#define die(s){mt_Component::printFormat s; exit (1);}
#define insist(e) do { if (!(e)) {mt_Component::printFormat ("Assertion failed in file %s, line %d",__FILE__,__LINE__); goto exception;}} while (0)
#define insistp(e, s) do { if (!(e)) {mt_Component::printFormat s; goto exception;}} while (0)


static void printLine (char*s)
{
  printf ("%s\n", s);
}

static char*getSandiaHostname (char*buffer, int host, int unit)
{
  insist (buffer && host >= 0 && unit >= 0);
  
  sprintf (buffer, "c-%d.SU-%d.SM-0.alaska", host, unit);
  return buffer;
  
  exception: return 0;
}

static int getRoute (mt_Route*r, char*line)
{
  mt_Tokenizer tokens (line, " \t\n\r");
  char hops [mt_Route::MAX_ROUTE];
  int numHops;
  
  insist (r && line);
  
  for (numHops = 0; tokens.getNext (); numHops++)
    if (!(hops [numHops] = tokens.getInteger ()) && numHops)
      break;

  return  r->fromBytes (mt_Route::logical (hops, hops, numHops), numHops);
  
  exception:  
  return 0;  
}

static int getRoutes (FILE*fp, mt_Graph*g, char*filename, int numHosts)
{
  mt_FileReader reader (filename, mt_Route::MAX_HOP * 10, "");
  mt_Route r;
  int numHops = 0;
    
  insist (filename);
  insist (g);
  
  for (int i = 0; i < numHosts; i++)  
  {
    char*s = reader.readLine ();
    insistp (s, ("unexpected end of file %s which was supposed to have %d entries", filename, numHosts));  
    int length = getRoute (&r, s);
    insist (length);
    numHops += length;
    
    if (fp)
      fprintf (fp, "\"%s\" %s\n", g->getHost (i)->getName (),  r.toString ());
  }

  return numHops;

  exception: return 0;
}

int main (int argc, char*argv [])
{
  mt_Component::initialize (printLine);
  char filename [mt_File::FILENAME_LENGTH];
  
  if (argc != 4)
    die (("usage: %s <map file> <routes directory>", argv [0]));
  
  mt_MapFile mf (argv [1], mt_File::_READ);
  mt_Graph g;

  if (!mf.read (&g))
    die (("main: couldn't parse map file %s", argv [1]));

  char*routesDirectory;
  routesDirectory = argv [2];
  insistp (routesDirectory && strlen (routesDirectory) > 0, ("%s is a bad routes directory", argv [1]));
    
  int numHosts;
  numHosts = g.getNumHosts ();
  
  printf ("%d 1\n", numHosts);
  
  for (int i = 0; i < numHosts; i++)
  {
    char*hostname = g.getHost (i)->getName ();
    int numHops = getRoutes (0, &g, mt_File::combine (filename, routesDirectory, hostname), numHosts);
    insist (numHops);
    
    printf ("\"%s\" (%d %d)\n", hostname, numHosts, numHops);
    
    getRoutes (stdout, &g, filename, numHosts);
  }
   
  return 0;
  exception: return 1;
}
