/*
  examine.c
  examine routes
  finucane@myri.com (David Finucane)
*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "mt_MapFile.h"
#include "mt_RouteTable.h"
#include "mt_Follower.h"
#include "mt_Switch.h"

#define die(s){mt_Component::printFormat s; exit (1);}
#define insist(e) do { if (!(e)) {mt_Component::printFormat ("Assertion failed in file %s, line %d",__FILE__,__LINE__); goto exception;}} while (0)
#define insistp(e, s) do { if (!(e)) {mt_Component::printFormat s; goto exception;}} while (0)


static void printLine (char*s)
{
  printf ("%s\n", s);
}

class ex_Counts : public mt_Component, mt_Follower
{
  private:
  int numSwitches;
  int maxRoutes;
  int* counts;
  int pass;
  int numLinks;
  int numDuplicates;
  int mode;
  int routeIndex;
  
  int stepGraph (mt_Node*n, int in);
  int stepPair (mt_Node*n, int in);

  enum
  {
    PAIR,
    GRAPH
  };
  
  
  public:
  
  ex_Counts (int numSwitches, int maxRoutes);
  ~ex_Counts ();
  int getDuplicates (mt_Graph*graph, mt_RouteTable*routes, int from, int to, int*numLinks);
  int getDuplicates (mt_Graph*graph, mt_RouteTable*routes, int*numLinks);
  int step (mt_Node*n, int in);
};

ex_Counts::ex_Counts (int numSwitches, int maxRoutes)
{
  counts = new int [(this->maxRoutes = maxRoutes) * (this->numSwitches = numSwitches) * mt_Switch::NUM_PORTS];
  insistp (counts, ("ex_Counts::ex_Counts: alloc failed"));
  exception:;
}

ex_Counts::~ex_Counts ()
{
  if (counts) delete counts;
}

int ex_Counts::step (mt_Node*n, int in)
{
  insist (this && n);
  insist (counts);

  switch (mode)
  {
    case PAIR:
      return stepPair (n, in);
    case GRAPH:
      return stepGraph (n, in);
    default:
      insist (0);
  }
  exception: return 0;
}

int ex_Counts::stepPair (mt_Node*n, int in)
{
  insist (this && n);
  insist (counts);
  insist (mode == PAIR);
  
  if (n->isSwitch ())
  {
    int p = n->getTypeIndex ();
    insist (p >= 0 && p < numSwitches);
    insist (in >= 0 && in < mt_Switch::NUM_PORTS);
  
    switch (pass)
    {
      case 0:
	counts [p * mt_Switch::NUM_PORTS + in] = 0;
	break;
      case 1:
	counts [p * mt_Switch::NUM_PORTS + in] ++;
	break;
      case 2:
	numLinks++;
	if (counts [p * mt_Switch::NUM_PORTS + in] > 1)
	  numDuplicates++;
	break;
      default:
	insist (0);
    }
  }
  return 1;
  exception: return 0;
}

int ex_Counts::stepGraph (mt_Node*n, int in)
{
  insist (this && n);
  insist (counts);
  insist (mode == GRAPH);
  
  if (n->isSwitch ())
  {
    int p = n->getTypeIndex ();
    insist (p >= 0 && p < numSwitches);
    insist (in >= 0 && in < mt_Switch::NUM_PORTS);
  
    insist (routeIndex >= 0 && routeIndex < maxRoutes);
    
    switch (pass)
    {
      case 0:
	counts [routeIndex * (p * mt_Switch::NUM_PORTS + in)] = 0;
	break;
      case 1:
	counts [routeIndex * (p * mt_Switch::NUM_PORTS + in)] ++;      
	break;
      case 2:
	{
	  int c = 0;
	  int t = 0;
	  for (int i = 0; i < maxRoutes; i++)
	  {
	    if (counts [i * (p * mt_Switch::NUM_PORTS + in)])
	      t++;
	    
	    c += counts [i * (p * mt_Switch::NUM_PORTS + in)];
	    counts [i * (p * mt_Switch::NUM_PORTS + in)] = 0;
	  }
	  if (c)
	  {
	    numLinks += c;
	    if (t > 1)
	      numDuplicates += c;
	  }
	}
    }
  }
  return 1;
  exception: return 0;
}


int ex_Counts::getDuplicates (mt_Graph*graph, mt_RouteTable*routes, int from, int to, int*numLinks)
{
  insist (this);

  this->numLinks = 0;
  numDuplicates = 0;
  pass = 0;
  mode = PAIR;
  
  int numRoutes;
  numRoutes = routes->getNumRoutes (from, to);
  mt_Node*n;
  n =graph->getHost (from);
  
  insist (n);
  
  for (pass = 0; pass < 3; pass++)
  {
    for (int k = 0; k < numRoutes; k++)
    {
      mt_Route r;
      mt_Node*other;
      int in;
     
      if (!routes->getRoute (from, to, k, &r))
	return 0;

      if (!n->follow (&r, &other, &in, this) || other != graph->getHost (to))
	return 0;
    }
  }

  *numLinks = this->numLinks;
  return numDuplicates;
  exception: return 0;
}

int ex_Counts::getDuplicates (mt_Graph*graph, mt_RouteTable*routes, int*numLinks)
{
  insist (this);

  this->numLinks = 0;
  numDuplicates = 0;
  pass = 0;
  mode = GRAPH;

  int numHosts;
  numHosts = graph->getNumHosts ();
  insist (numHosts >= 0);  

  for (pass = 0; pass < 3; pass++)
  {
    for (int i = 0; i < numHosts; i++)
    {
      mt_Node*n = graph->getHost (i);
      insist (n);
  
      for (int j = 0; j < numHosts; j++)
      {	
	int numRoutes =  routes->getNumRoutes (i, j);
	insist (numRoutes <= routes->getMaxRoutes ());
	
	for (routeIndex = 0; routeIndex < numRoutes; routeIndex++)
	{
	  mt_Route r;
	  mt_Node*other;
	  int in;
     
	  if (!routes->getRoute (i, j, routeIndex, &r))
	    return 0;

	  if (!n->follow (&r, &other, &in, this) || other != graph->getHost (j))
	    return 0;
	}
      }
    }
  }
  *numLinks = this->numLinks;
  return numDuplicates;
  exception: return 0;
}

  
int main (int argc, char*argv[])
{  
  mt_Component::initialize (printLine);
    
  mt_RouteTable routes;
  mt_Graph graph;
  
  if (argc != 3)
    die (("usage: %s <mapfile> <routefile>\n", argv[0]));
 
  char*mapFile = argv [1];
  char*routeFile = argv[2];

  mt_MapFile mf (mapFile, mt_File::_READ);
  
  if (!mf.read (&graph))
    die (("main: couldn't parse map file %s", mapFile));
  
  if (!routes.fromFile (routeFile, &graph))
    die (("main: couldn't read %s", routeFile));

  int numHosts = graph.getNumHosts ();
  ex_Counts*counts = new ex_Counts (graph.getNumSwitches (), routes.getMaxRoutes ());
  insistp (counts, ("main: alloc failed"));
  insist (routes.check (&graph));
  
  int totalLinks, totalDuplicates;
  totalLinks = totalDuplicates = 0;
  
  for (int i = 0; i < numHosts; i++)
  {
    for (int j = 0; j < numHosts; j++)
    {

      int numLinks = 0;
      int numDuplicates = counts->getDuplicates (&graph, &routes, i, j, &numLinks);

      totalLinks += numLinks;
      totalDuplicates += numDuplicates;

#if 0      
      mt_Component::printFormat ("%s %s %d %d",
				 graph.getHost (i)->getName (),
				 graph.getHost (j)->getName (),
				 numDuplicates, numLinks);
#endif

    }
  }
  mt_Component::printFormat ("%d / %d (%5.4lf)", totalDuplicates, totalLinks, (double) (totalLinks ? (double) totalDuplicates / (double) totalLinks : 0));

  totalDuplicates = counts->getDuplicates (&graph, &routes, &totalLinks);
  
  mt_Component::printFormat ("%d / %d (%5.4lf)", totalDuplicates, totalLinks, (double) (totalLinks ? (double) totalDuplicates / (double) totalLinks : 0));

  exception:;
  return 0;
}
