/*
  sw_Test162.c
  test 16 port switch version 2.
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <string.h>
#include "sw_Test162.h"
#include "sw_MessageSender.h"
#include "sw_ImpedenceReader.h"
#include "sw_CrossPointSender.h"
#include "sw_PortSender.h"
#include "sw_RouteSender.h"
#include "sw_SpeedWriter.h"
#include "sw_DeadlockSender.h"

#include "insist.h"
#include "test.h"

#define comment_c(e) {if (verbose) printFormat e;}
#define error_c(r,e) {printFormat ("\nerror: %s:%d", __FILE__, __LINE__); printFormat e; printFormat (""); if (lazy) return 0; else r = 0;}


sw_Test162::sw_Test162 (int unit, mt_Graph*graph, mt_Network*network, mt_Node*host, sw_TestPort*port, int numSwitches, mt_Node**switches, int verbose, int lazy, int lanai) : 
  sw_Test16 (unit, graph, network, host, port, numSwitches, switches, verbose, lazy, lanai)

{
}

int sw_Test162::testId (int xbar)
{
  int r = 1;
  char low, high;

  insist (this);

  comment_c (("testing id register"));
  
  if (!testPort->getChipId (xbar, &low, &high))
    error_c (r, ("getChipId failed"));
  
  if ((low != 0x20 && low != 0x21) || high != 0x50)
    error_c (r, ("id register error. was %#x %#x, expected 0x50 0x20", high & 0xff, low & 0xff));

  return r;
  exception: return 0;
}

int sw_Test162::testInitialOutputImpedence (int xbar)
{
  int r = 1;
  
  insist (this);
  
  char low, high;
  
  comment_c (("verifying initial output impedence"));
  
  if (!testPort->getOutputImpedence (xbar, &low, &high))
    error_c (r, ("getOutputImpedence failed"));
  
  if (low != (char) 0xe1 || high != (char) 0xe1)
    error_c (r, ("output impedence error. was %#x %#x, expected e1 e1", low & 0xff, high & 0xff));
  
  return r;
  exception: return 0;
}


int sw_Test162::testOutputImpedence (int xbar, char vlow, char vhigh, char, char)
{
  int r = 1;
  char low, high;

  insist (this);
  
  comment_c (("testing output impedence with %#x, %#x", vlow & 0xff, vhigh & 0xff));

  if (!testPort->setOutputImpedence (xbar, vlow, vhigh) || !testPort->getOutputImpedence (xbar, &low, &high))
    error_c (r, ("setOutputImpedence failed"));

  if (low != vlow || high != vhigh)
    error_c (r, ("output impedence error. was %#x %#x, expected %#x %#x", high & 0xff, low & 0xff, vhigh & 0xff, vlow & 0xff));

  return r;
  exception: return 0;
}

int sw_Test162::testImpedence (int xbar)
{
  int r = 1;
  insist (this);

  comment_c (("testing scan impedence"));
  
  if (!testPort->resetXbar (xbar))
    error_c (r, ("resetXbar failed"));

  if (!testPort->setGlobalControl (xbar, 0xfd))
    error_c (r, ("setGlobalControl failed"));
  
  comment_c (("setting ztest to 0 0"));

  if (!testPort->setTestImpedence (xbar, 0, 0))
    error_c (r, ("setTestImpedence failed"));

  double impedence;
  double v;
  
  v = (double) testPort->getA2D (xbar, sw_TestPort::OAL_CHANNEL);
  if (v >= 512) v = 511;
  impedence = (50.0 * v) / (512.0 - v);
  if (impedence <= 62)
    error_c (r, ("OAL impedence is %4.2lf", impedence));

  v = (double) testPort->getA2D (xbar, sw_TestPort::OAH_CHANNEL);
  if (v < 0) v = 1;
  impedence = 50.0 * (512 - v) / v;
  if (impedence <= 62)
    error_c (r, ("OAH impedence is %4.2lf", impedence));

  comment_c (("setting ztest to ff ff"));

  if (!testPort->setTestImpedence (xbar, 0xff, 0xff))
    error_c (r, ("setTestImpedence failed"));

  v = (double) testPort->getA2D (xbar, sw_TestPort::OAL_CHANNEL);
  if (v >= 512) v = 511;
  impedence = (50.0 * v) / (512.0 - v);
  if (impedence >= 40)
    error_c (r, ("OAL impedence is %4.2lf", impedence));

  v = (double) testPort->getA2D (xbar, sw_TestPort::OAH_CHANNEL);
  if (v < 0) v = 1;
  impedence = 50.0 * (512 - v) / v;
  if (impedence >= 40)
    error_c (r, ("OAH impedence is %4.2lf", impedence));

  for (int i = 0x02; i <= 0x40; i <<=1)
  {  
    comment_c (("setting ztest to %#x %#x", i, i));
    
    if (!testPort->setTestImpedence (xbar, i, i))
      error_c (r, ("setTestImpedence failed"));
  
    int pdl, pul, pdh, puh;
    
    pdl = testPort->getA2D (xbar, sw_TestPort::OAL_CHANNEL);
    pul = testPort->getA2D (xbar, sw_TestPort::OAH_CHANNEL);
  
    if (!testPort->setTestImpedence (xbar, i - 1, i - 1))
      error_c (r, ("setTestImpedence failed"));
  
    pdh = testPort->getA2D (xbar, sw_TestPort::OAL_CHANNEL);
    puh = testPort->getA2D (xbar, sw_TestPort::OAH_CHANNEL);
    
    if (pdl >= pdh || pul <= puh)
      error_c (r, ("pdl is %d, pdh is %d, pul is %d, puh is %d", pdl, pdh, pul, puh));
  }
  return r;
  exception: return 0;
}


int sw_Test162::testGlobalControl (int xbar)
{
  int r = 1;
  char low;

  insist (this);  

  comment_c (("testing global control"));
  
  if (!testPort->getGlobalControl (xbar, &low))
    error_c (r, ("getGlobalControl failed"));
  
  if (low != (char) 0xfc)
    error_c (r, ("global control error. was %#x, expected 0xfc", low & 0xff));

  if (!sw_Test16::testGlobalControl (xbar, 0x00))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x01))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x04))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x08))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x10))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x20))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x40))
    error_c (r, ("testGlobalControl failed"));

  if (!sw_Test16::testGlobalControl (xbar, 0x80))
    error_c (r, ("testGlobalControl failed"));

  if (!testPort->setGlobalControl (xbar, 0xfc))
    error_c (r, ("testGlobalControl failed"));
  
  return r;
  exception: return 0;
}

int sw_Test162::testAuto1 (int xbar)
{
  int r = 1;
  
  insist (this);

  if (!testAuto1 (xbar, 0, 0))
    error_c (r, ("testAuto1 failed"));
  
  if (!testAuto1 (xbar, 0, 1))
    error_c (r, ("testAuto1 failed"));
  
  if (!testAuto1 (xbar, 1, 0))
    error_c (r, ("testAuto1 failed"));
  
  if (!testAuto1 (xbar, 1, 1))
    error_c (r, ("testAuto1 failed"));

  return r;
  exception: return 0;
}

int sw_Test162::testAuto1 (int xbar, int en, int ha)
{
  int r = 1;
  insist (this);

  comment_c (("test auto 1"));
  comment_c (("setting AE and AHAE to %d %d", en, ha));
  
  if (!testPort->setHa (ha) || !testPort->setAuto (en))
    error_c (r, ("setHa or setAuto failed"));

  comment_c (("verifying auto en and auto ha"));
  
  if (!testPort->resetXbar (xbar))
    error_c (r, ("resetXbar failed"));
  
  if (!verifyAuto (xbar, en, ha))
    error_c (r, ("verifyAuto failed"));

  return r;
  exception: return 0;
}

int sw_Test162::verifyAuto (int xbar, int en, int ha)
{
  int r = 1;
  
  insist (this);
  
  char control [sw_TestPort::NUM_PORTS];
  if (!testPort->getControl (xbar, control))
    error_c (r, ("getControl failed"));
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if ((control [i] & 0x08) && !en || !(control [i] & 0x08) && en)
      error_c (r, ("auto en is wrong on port %d", i));

    if ((control [i] & 0x04) && !ha || !(control [i] & 0x04) && ha)
      error_c (r, ("auto ha is wrong on port %d", i));
  }
  
  return r;
  exception : return 0;
}


int sw_Test162::testAutoImpedence (int xbar)
{
  int r = 1;
  
  insist (this);

  comment_c (("testing auto impedence"));
  
  if (!testPort->resetXbar (xbar))
    error_c (r, ("resetXbar failed"));

  if (!testPort->setGlobalControl (xbar, 0xf0))
    error_c (r, ("setGlobalControl failed"));
  
  testPort->usleep (1000 * 1000 * 2);
  
  double d;
  d = testPort->getVoltage (xbar, sw_TestPort::BIAS_CHANNEL);
  comment_c (("BIAS voltage is %3.4f", d));
  if (d < 1.0 || d > 1.4)
    error_c (r, ("out of range"));

  d = testPort->getVoltage (xbar, sw_TestPort::VTH_CHANNEL);
  comment_c (("VTH voltage is %3.4f", d));
  if (d < 0.6 || d > 0.65)
    error_c (r, ("out of range"));

  d = testPort->getVoltage (xbar, sw_TestPort::LVDD_CHANNEL);
  comment_c (("LVDD voltage is %3.4f", d));
  if (d < 1.2 || d > 1.3)
    error_c (r, ("out of range"));

  d = testPort->getVoltage (xbar, sw_TestPort::OAL_CHANNEL);
  comment_c (("OAL voltage is %3.4f", d));
  if (d < 0.55 || d > 0.7)
    error_c (r, ("out of range"));

  d = testPort->getVoltage (xbar, sw_TestPort::OAH_CHANNEL);
  comment_c (("OAH voltage is %3.4f", d));
  if (d < 0.55 || d > 0.7)
    error_c (r, ("out of range"));
  
  return r;
  
  exception: return 0;
}

int sw_Test162::testSpeed (int xbar, int port, int set)
{
  int r = 1;
  
  insist (this);

  comment_c (("testing speed on port %d with fast bits %s", port, set ? "set" : "cleared"));

  if (!testPort->setSpeed (xbar, port - getPortCorrection (xbar), set))
    error_c (r, ("setSpeed failed"));
 
  comment_c (("sending some packets"));
  if (!send (xbar, 0, 0, 1, 100))
    error_c (r, ("send failed"));

  comment_c (("getting bandwidth")); 
  
  double d;
  d = getBandwidth (xbar, port);
  comment_c (("bandwidth is %5.2f (mbps)", d));
  
  if (d > 680)
    error_c (r, ("bandwidth too high"));  

  comment_c (("clearing fast bits"));

  if (!testPort->setSpeed (xbar, set ? 0 : 0xff, set ? 0 : 0xff, set ? 0 : 0xff, set ? 0 : 0xff))
    error_c (r, ("setSpeed failed"));
  
  comment_c (("sending some packets"));
  if (!send (xbar, 0, 0, 1, 100))
    error_c (r, ("send failed"));

  comment_c (("getting bandwidth"));
  
  d = getBandwidth (xbar, port);
  comment_c (("bandwidth is %5.2f (mbps)", d));
  
  if (d < 700)
    error_c (r, ("bandwidth too low"));

  return r;
  
  exception: return 0;
}


int sw_Test162::testSpeed (int xbar)
{
  int r = 1;
  
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  
  comment_c (("testing speed"));

  comment_c (("setting CLSOW to 62.5, CFAST to 20"));
  
  if (!testPort->setIcs (0x606c81) || !testPort->setIcs (0x5581))
    error_c (r, ("setIcs failed"));
  
  comment_c (("writing speed to 0 0 0 0"));
  
  if (!testPort->setSpeed (xbar, 0, 0, 0, 0))
    error_c (r, ("setSpeed failed"));
  
  comment_c (("waiting V seconds"));
  testPort->usleep (1000 * 5000);

  comment_c (("sending some packets and ignoring them"));
  if (!send (xbar, 0, 1, 1, 16, 50))
    error_c (r, ("send failed"));
  
  comment_c (("sending some more, this time caring about them"));
  
  if (!send (xbar, 1, 1, 1, 100))
    error_c (r, ("send failed"));
  
  comment_c (("getting bandwidth"));
  
  double d;
  d = getBandwidth (xbar);
  comment_c (("bandwidth is %5.2f (mbps)", d));
  
  if (d < 700)
    error_c (r, ("bandwidth too low"));
  
  d = getBandwidth (xbar, 2);
  comment_c (("bandwidth is %5.2f (mbps)", d));
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i) || !switches [xbar]->getNode (i)->isSwitch () || i == switches [xbar]->getIn ())
    {
      comment_c (("skipping port %d for a good reason", i));
      continue;
    }
    
    if (!testSpeed (xbar, i, 1))
      error_c (r, ("testSpeed failed"));
  }
  
  comment_c (("setting CLSOW to 20, CFAST to 62.5"));
  
  if (!testPort->setIcs (0x605581) || !testPort->setIcs (0x6c81))
    error_c (r, ("setIcs failed"));

  comment_c (("writing speed to ff ff ff ff"));
  
  if (!testPort->setSpeed (xbar, 0xff, 0xff, 0xff, 0xff))
    error_c (r, ("setSpeed failed"));

  comment_c (("waiting 10 half seconds"));
  testPort->usleep (1000 * 5000);
    
  comment_c (("sending some packets and ignoring them"));
  if (!send (xbar, 0, 0, 0, -1, 1, 54))
    error_c (r, ("send failed"));
  
  comment_c (("sending some more, this time caring about them"));
  
  if (!send (xbar, 1, 1, 1, 100))
    error_c (r, ("send failed"));
  
  comment_c (("getting bandwidth"));
  
  d = getBandwidth (xbar);
  comment_c (("bandwidth is %5.2f (mbps)", d));
  
  if (d < 700)
    error_c (r, ("bandwidth too low"));
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i) || !switches [xbar]->getNode (i)->isSwitch ()  || i == switches [xbar]->getIn ())
    {
      comment_c (("skipping port %d for a good reason", i));
      continue;
    }
    
    if (!testSpeed (xbar, i, 0))
      error_c (r, ("testSpeed failed"));
  }

  if (!testPort->setIcs (0x606c81) || !testPort->setIcs (0x6c81))
    error_c (r, ("setIcs failed"));

  return r;
  exception: return 0;
}

int sw_Test162::testAuto2 (int xbar)
{
  int r = 1;
  insist (this);

  comment_c (("testing auto2"));
  
  if (!testPort->setHa (1) || !testPort->setAuto (1) || !verifyAuto (xbar, 1, 1))
    error_c (r, ("setHa, setAuto or verifyAuto failed"));

  comment_c (("sleeping 2 seconds"));
  
  testPort->usleep (1000 * 1000 * 2);
  
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  comment_c (("reading links status 3 times"));
  
  for (int i = 0; i < 3; i++)
    if (!testPort->getLinkStatus2 (xbar, receive, transmit))
      error_c (r, ("getLinkStatus failed"));
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i + getPortCorrection (xbar)))
    {
      comment_c (("skipping physical port %d because it is disconnected", i));
      continue;
    }
    
    if (testPort->getReceiveLinkStatusField (receive, i, 2))
      error_c (r, ("port %d has !BEAT bit set", i));
    
    if (testPort->getReceiveLinkStatusField (receive, i, 1))
      error_c (r, ("port %d has missed beat bit set", i));

    if (testPort->getReceiveLinkStatusField (receive, i, 0))
      error_c (r, ("port %d has illegal symbol bit set", i));

    if (testPort->getReceiveLinkStatusField (receive, i, 5))
      error_c (r, ("port %d has !gap received bit set", i));

    if (testPort->getTransmitLinkStatusField (transmit, i, 0))
      error_c (r, ("port %d has !xmit send gap bit set", i));

    if (testPort->getTransmitLinkStatusField (transmit, i, 1))
      error_c (r, ("port %d has !xmit in go bit set", i));

    if (testPort->getReceiveLinkStatusField (receive, i, 9))
      error_c (r, ("port %d has auto port off bit set", i));

    if (testPort->getReceiveLinkStatusField (receive, i, 10))
      error_c (r, ("port %d has auto rec to bit set", i));

    if (testPort->getTransmitLinkStatusField (transmit, i, 3))
      error_c (r, ("port %d has auto xmit to bit set", i));
  }
  
  return r;
  exception: return 0;
}

int sw_Test162::testMessage (int xbar)
{
  int r = 1;
  int good1 [sw_TestPort::NUM_PORTS];
  int bad1 [sw_TestPort::NUM_PORTS];
  int good2 [sw_TestPort::NUM_PORTS];
  int bad2 [sw_TestPort::NUM_PORTS];
  int counts [sw_TestPort::NUM_PORTS];
 
  insist (this);
  insist (host && graph);
  insist (xbar >= 0 && xbar < numSwitches);

  {
    int length = mt_Network::MTU;      
    sw_MessageSender ms (this, getNetwork (), graph, switches [xbar], 1, 1, length);
    sw_SpeedWriter sw (xbar, testPort);

    comment_c (("setting CLSOW to 80, CFAST to 80"));
  
    if (!testPort->setIcs (0x606c81) || !testPort->setIcs (0x6c81))
      error_c (r, ("setIcs failed"));

    comment_c (("writing speed to 0 0 0 0"));
  
    if (!testPort->setSpeed (xbar, 0, 0, 0, 0))
      error_c (r, ("setSpeed failed"));

    if (!reopen ())
      error_c (r, ("open failed"));
    
    comment_c (("waiting one second, five times in a row"));
    testPort->usleep (1000 * 5000);

    comment_c (("getting packet counters"));
    
    if (!testPort->getPacketCounters (xbar, good1, bad1))
      return -1;

    comment_c (("testing %d byte messages for 20 seconds while writing speed register...", length));

    ms.createThread ();
    sw.createThread ();

    testPort->usleep (1000 * 1000 * 20);

    comment_c (("20 seconds are up."));
    comment_c (("may have to wait %d seconds for a dropped message.", sw_Sender::TIMEOUT / (1000 * 1000)));
    
    int r1, r2;
    
    r1 = ms.stopThread ();
    r2 = sw.stopThread ();

    comment_c (("%d / %d received (%d missing)", ms.getNumReceived (), ms.getNumSent (), ms.getNumSent () - ms.getNumReceived ()));
    comment_c (("wrote speed register %d times", sw.getCount ()));

    comment_c (("sifting through packet counters, trying to see what happened."));
      
    ms.getCounts (counts);
      
    if (!testPort->getPacketCounters (xbar, good2, bad2))
      return -1;
      
    for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
    {
      if (bad1 [i] != bad2 [i])
      {
	printFormat ("%d bad packets on port %d", bad2 [i] - bad1 [i], i);
      }
    }
      
    for (int i = 1; i < sw_TestPort::NUM_PORTS; i++)
    {
      if (counts [i] != good2 [i] - good1 [i])
      {
	printFormat ("%d missing packets on port %d", counts [i] - (good2 [i] - good1 [i]), i);
      }
    }

    if (!r1)
      error_c (r, ("message test failed"));
    
    if (!r2)
      error_c (r, ("speed writing failed"));

    close ();
  
    comment_c (("setting channel speed to whatever"));
    comment_c (("actually, not doing anything at all here"));
  }
  return r;
  exception: return 0;
}

int sw_Test162::testAuto3 (int xbar)
{
  int r = 1;
  
  insist (this);
  
  char control [sw_TestPort::NUM_PORTS];
  memset (control, 0, sizeof (control));

  comment_c (("test auto3"));

  if (!testPort->setControl (xbar, control) || !testPort->getControl (xbar, control))
    error_c (r, ("setControl failed"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i + getPortCorrection (xbar)))
    {
      comment_c (("skipping physical port %d because it is disconnected", i));
      continue;
    }
    if (control [i])
      error_c (r, ("control byte for port %d nonzero", i));
  }
  return r;
  exception: return 0;
}


int sw_Test162::getIllegalSymbols (int xbar, int port)
{
  int r = 1;
  insist (this);
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus failed"));
  
  return testPort->extractReceive (port,receive) & 0x800;

  exception: return 0;
}

int sw_Test162::getStatusUpdate (int xbar, int port)
{
  int r = 1;
  insist (this);
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));
  
  return  (testPort->extractReceive (port,receive) & 0x900) == 0x800;

  exception: return 0;
}

int sw_Test162::getPortShutdown (int xbar, int port)
{
  int r = 1;
  
  insist (this);
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));

  return (testPort->extractReceive (port,receive) & 0x930) == 0x900;

  exception: return 0;
}

int sw_Test162::getInvalidHeads (int xbar, int port)
{
  int r = 1;
  insist (this);
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));

  return  testPort->extractReceive (port,receive) & 0x020 ;

  exception: return 0;
}

int sw_Test162::getTimeout (int xbar, int port, int set)
{
  int r = 1;
  insist (this);
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));

  return set ?
    (testPort->extractReceive(port,receive) & 0x0c0) == 0x0c0 && 
    (testPort->extractTransmit(port,transmit) & 0xc) == 0xc :
    (testPort->extractReceive(port,receive) & 0x0c0) == 0x0 &&
    (testPort->extractTransmit(port,transmit) & 0x0c) == 0x0;
  
  exception: return 0;
}


int sw_Test162::testAuto4 (int xbar)
{
  int r = 1;
  insist (this);

  comment_c (("test auto 4"));

  if (!testPort->setHa (0) || !testPort->setAuto (1) || ! testPort->resetXbar (xbar))
    error_c (r, ("setHa, setAuto or resetXbar failed"));

  comment_c (("writing 0xf0 to global control"));

  if (!testPort->setGlobalControl (xbar, 0xf0))
    error_c (r, ("setGlobalControl failed"));

  comment_c (("sleeping for 2 seconds"));
  
  testPort->usleep (1000 * 1000 * 2);

  comment_c (("reading status register twice and ignoring it"));
  
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit) || !testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));

  comment_c (("sending 1000 packets through all ports"));  

  if (!send (xbar, 0, 1, 1, 1000))
    error_c (r, ("send failed"));

  comment_c (("checking data in and data out bits are set for each port"));
  
  if (!testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i + getPortCorrection (xbar)))
    {
      comment_c (("skipping physical port %d because it is disconnected", i));
      continue;
    }
   
    if (!testPort->getReceiveLinkStatusField (receive, i, 4))
      error_c (r, ("data in not set on port %d", i));
    
    if (!testPort->getTransmitLinkStatusField (transmit, i, 1))
      error_c (r, ("data out not set on port %d", i));
  }

  for (int i = 0; i < 3; i++)
    if (!testPort->getLinkStatus2 (xbar, receive, transmit))
      error_c (r, ("getLinkStatus2 failed"));
      
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i) || i == switches [xbar]->getIn ())
      continue;
    
    int pport;
    pport  = i - getPortCorrection (xbar);
    insist (pport >= 0 && pport < sw_TestPort::NUM_PORTS);  
  
    comment_c (("sending a packet to deadlock on port %d", pport));

    if (!deadlock (xbar, i))
      error_c (r, ("failed to deadlock on port %d", pport));
    
    comment_c (("waiting one thousand five hundred milliseconds"));
    testPort->usleep (1000 * 1500);

    if (!testPort->getLinkStatus2 (xbar, receive, transmit))
      error_c (r, ("getLinkStatus2 failed"));

    if (!testPort->getReceiveLinkStatusField (receive, pport, 2))
      error_c (r, ("port %d has auto rec to bit clear", pport));

    if (!testPort->getTransmitLinkStatusField (transmit, pport, 0))
      error_c (r, ("port %d has auto xmit to bit clear", pport));
    
    if (testPort->getReceiveLinkStatusField (receive, pport, 3))
      error_c (r, ("port %d has auto port off bit set", pport));    

    comment_c (("sending a regular packet through port"));

    if (!send (xbar, 1, 1, i, 1, 5))
      error_c (r, ("failed to send to the port"));
  }
  
  return r; 
  exception: return 0;
}

int sw_Test162::testAuto5 (int xbar)
{
  int r = 1;
  insist (this);

  comment_c (("test auto 5"));

  if (!testPort->setHa (1) || !testPort->setAuto (1) || ! testPort->resetXbar (xbar))
    error_c (r, ("setHa, setAuto, or resetXbar failed"));

  if (!testPort->setGlobalControl (xbar, 0xf0))
    error_c (r, ("setGlobalControl failed"));

  if (!testPort->setRelays (xbar, 0xffff))
    error_c (r, ("setRelays failed"));
    
  comment_c (("waiting 2 seconds"));
  testPort->usleep (1000 * 1000 * 2);
  comment_c (("waiting 3 seconds"));
  testPort->usleep (1000 * 1000 * 3);

  comment_c (("reading status register twice and ignoring it"));
  
  char receive [12 * 16 / 8];
  char transmit [4 * 16 / 8];

  if (!testPort->getLinkStatus2 (xbar, receive, transmit) || !testPort->getLinkStatus2 (xbar, receive, transmit))
    error_c (r, ("getLinkStatus2 failed"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i + getPortCorrection (xbar)))
    {
      comment_c (("skipping physical port %d because it is disconnected", i));
      continue;
    }
    
    int pport;
    pport  = i - getPortCorrection (xbar);
    insist (pport >= 0 && pport < sw_TestPort::NUM_PORTS);

    comment_c (("sending 100 packets through port %d", pport));  

    if (!send (xbar, 1, 1, i, 1, 100))
      error_c (r, ("failed to send packets through port"));
    
    comment_c (("turning relay on"));  

    if (!testPort->setRelays (xbar, 0xffff ^ (1 << pport)))
      error_c (r, ("setRelays failed"));
    
    comment_c (("sending 100 packets through port %d", pport));  
  
    if (!send (xbar, 0, 1, i, 1, 100))
      error_c (r, ("failed to send packets through port"));
    
    comment_c (("reading status register"));
    
    if (!testPort->getLinkStatus2 (xbar, receive, transmit))
      error_c (r, ("getLinkStatus2 failed"));

    if (!testPort->getReceiveLinkStatusField (receive, pport, 3))
      error_c (r, ("port %d has auto port off bit cleared", pport));    

    if (!testPort->getReceiveLinkStatusField (receive, pport, 11))
      error_c (r, ("port %d has ill received clear", pport));    
    
    comment_c (("turning relay off"));

    if (!testPort->setRelays (xbar, 0xffff))
      error_c (r, ("setRelays failed"));
    
    comment_c (("sleeping for 5 seconds"));
    testPort->usleep (1000 * 5000);
    
    comment_c (("sending 100 packets through port %d", pport));  

    if (!send (xbar, 1, 1, i, 1, 100))
      error_c (r, ("failed to send packets through port"));

    comment_c (("reading status twice"));

    if (!testPort->getLinkStatus2 (xbar, receive, transmit) || !testPort->getLinkStatus2 (xbar, receive, transmit))
      error_c (r, ("getLinkStatus2 failed"));
    
    if (testPort->getReceiveLinkStatusField (receive, pport, 9))
      error_c (r, ("port %d has auto port off bit set", pport));    

    if (testPort->getReceiveLinkStatusField (receive, pport, 0))
      error_c (r, ("port %d has ill received set", pport));
  }
  
  comment_c (("writing 0xf0 to global control"));

  
  return r;
  exception: return 0;
}


int sw_Test162::testTimeout (int xbar, int port, int globalControl, double min, double max)
{
  int r = 1;
  double d;
  sw_DeadlockSender ds (this, getNetwork (), graph, switches [xbar], port);
  
  insist (this);

  comment_c (("test timeouts with global control %x", globalControl));

  if (!testPort->setGlobalControl (xbar, globalControl))
    error_c (r, ("setGlobalControl failed"));
  
  comment_c (("sleeping for 8 seconds"));
  testPort->usleep (1000 * 1000 * 8);

  if (!open ())
    error_c (r, ("open failed"));

  if (!ds.startThread ())
    error_c (r, ("deadlock sender failed"));
  
  close ();

  d = ((double) ds.getDuration ()) / (1000 * 1000);
  
  comment_c (("timeout measured as %6.4f seconds", d));
  
  if (d <= min || d > max)
    error_c (r, ("measured timeout %6.4f not between %6.4f and %6.4f seconds", d, min, max));
  
  return r;
  exception: return 0;
}


int sw_Test162::testAuto6 (int xbar)
{
  int r = 1;
  int port;
  insist (this);

  comment_c (("test auto 6"));

  int maxNodes;
  maxNodes = switches [xbar]->getMaxNodes ();
  
  computeRoutes ();
  
  for (port = 0; port < maxNodes; port++)
    if (port != switches [xbar]->getIn () && switches [xbar]->getNode (port) && switches [xbar]->getNode (port)->isSwitch ())
      break;
    
  if (port == maxNodes)
  {
    printFormat ("couldn't find decent port to deadlock on");
    return 0;
  }
  
  comment_c (("deadlocking on port %d", port));
  

  if (!testTimeout (xbar, port, 0xf0, 1.0/20.0, 1.0/6.0))
    error_c (r, ("testTimeout failed"));
  if (! testTimeout (xbar, port, 0xf4, 1.0/5.0, 2.0/3.0))
    error_c (r, ("testTimeout failed"));
  if (!testTimeout (xbar, port, 0xf8, 0.8, 2.5))
    error_c (r, ("testTimeout failed"));
  if (!testTimeout (xbar, port, 0xfc, 3.0, 10.0))
    error_c (r, ("testTimeout failed"));

  return r;
  exception: return 0;
}

int sw_Test162::initialize (int xbar)
{
  insist (this);

  return sw_Test16::initialize (xbar) && testPort->setIcs (0x61E803) && testPort->setIcs (0x6D01) && testPort->disableImpedence (xbar);
  
  return 1;
  exception: return 0;
}
