/*
  sw_Test16.c
  class to test a 16 port switch
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <string.h>
#include "sw_Test16.h"
#include "sw_MessageSender.h"
#include "sw_ImpedenceReader.h"
#include "sw_CrossPointSender.h"
#include "sw_PortSender.h"
#include "sw_RouteSender.h"
#include "sw_BandwidthSender.h"
#include "sw_SnakeySender.h"

#include "insist.h"
#include "test.h"

/* Bits in the MYRINET register */

#define MYRINET_REGISTER 0x130

#define		NRES_ENABLE_BIT		(1<<0)
#define		CRC32_ENABLE_BIT	(1<<1)
#define		TX_CRC8_ENABLE_BIT	(1<<2)
#define		RX_CRC8_ENABLE_BIT	(1<<3)
#define		ILLEGAL_ENABLE		(1<<4)
#define		BEAT_ENABLE		(1<<5)
#define		TIMEOUT0		(1<<6)
#define		TIMEOUT1		(1<<7)
#define		WINDOW0			(1<<8)
#define		WINDOW1			(1<<9)

#define CRC32_ENABLED (TIMEOUT0 | TIMEOUT1 | WINDOW0  | WINDOW1  | RX_CRC8_ENABLE_BIT | TX_CRC8_ENABLE_BIT)
#define CRC32_DISABLED (TIMEOUT0 | TIMEOUT1 | WINDOW0  | WINDOW1)

sw_Test16::sw_Test16 (int unit, mt_Graph*graph, mt_Network*network, mt_Node*host, sw_TestPort*port, int numSwitches, mt_Node**switches, int verbose, int lazy, int lanai) : mt_Job (host, network)
{
  insist (this);
  insist (graph && host && switches);
  insist (numSwitches > 0 && numSwitches <= MAX_SWITCHES);

  routesComputed = 0;
  this->portCorrectionXbar = -1;
  this->unit = unit;
  this->verbose = verbose;
  this->lazy = lazy;
  this->graph = graph;
  this->host = host;
  this->numSwitches = numSwitches;
  this->lanai = lanai;
  testPort = port;
  opened = 0;
  memcpy (this->switches, switches, sizeof (mt_Node*) * numSwitches);
  
  exception:;
}

sw_Test16::~sw_Test16 ()
{
  insist (this);
  exception:;
}

int sw_Test16::reopen ()
{
  if (opened)
  {
    getNetwork ()->close (this);
    opened = 0;
  }
  return open ();
}

int sw_Test16::open ()
{
  mt_Address address;
  char hostname [mt_Node::NAME_LENGTH];
  int type;

  insist (this);

  return opened || (opened = getNetwork()->open (this, &address, hostname, &type, unit));
  
  exception: return 0;
}

int sw_Test16::close ()
{
  insist (this);

  /*  opened = 0;
      getNetwork()->close (this);
  */
  exception: return 1;
}

#define comment_c(e) {if (verbose) printFormat e;}
#define error_c(r,e) {printFormat ("\nerror: %s:%d", __FILE__, __LINE__); printFormat e; printFormat (""); if (lazy) return 0; else r = 0;}

int sw_Test16::testId (int xbar)
{
  int r = 1;
  char low, high;

  insist (this);

  comment_c (("testing id register"));
  
  if (!testPort->getChipId (xbar, &low, &high))
    error_c (r, ("getChipId failed"));
  
  if (low != 0x00 || high != 0x50)
    error_c (r, ("id register error. was %#x %#x, expected 0x50 0x00", high & 0xff, low & 0xff));

  return r;
  exception: return 0;
}

int sw_Test16::testGlobalControl (int xbar, char vlow)
{
  int r = 1;
  char low;
  
  insist (this);
  
  comment_c (("testing global control with %#x", vlow & 0xff));
  
  if (!testPort->setGlobalControl (xbar, vlow) || !testPort->getGlobalControl (xbar, &low))
    error_c (r, ("setGlobalControl  failed"));
  
  if (low != vlow)
    error_c (r, ("global control error. was %#x, expected %#x", low & 0xff, vlow & 0xff));

  return r;
  exception: return 0;
}

int sw_Test16::testGlobalControl (int xbar)
{
  int r = 1;
  char low;

  insist (this);  

  comment_c (("testing global control"));
  
  if (!testPort->getGlobalControl (xbar, &low))
    error_c (r, ("sw_Test16 failed"));
  
  if (low != (char) 0xc0)
    error_c (r, ("global control error. was %#x, expected 0xc0", low & 0xff));

  if (!testGlobalControl (xbar, 0x00))
    error_c (r, ("testGlobalControl failed"));
  
  if (!testGlobalControl (xbar, 0x40))
    error_c (r, ("testGlobalControl failed"));
  
  if (!testGlobalControl (xbar, 0x80))
    error_c (r, ("testGlobalControl failed"));
  
  if (!testGlobalControl (xbar, 0xc0))
    error_c (r, ("testGlobalControl failed"));
  
  return r;
  exception: return 0;
}

int sw_Test16::testTestImpedence (int xbar, char vlow, char vhigh)
{
  int r = 1;
  char low, high;

  insist (this);
  
  comment_c (("testing test impedence with %#x, %#x", vlow & 0xff, vhigh & 0xff));

  if (!testPort->setTestImpedence (xbar, vlow, vhigh) || !testPort->getTestImpedence (xbar, &low, &high))
    error_c (r, ("setTestImpedence failed"));

  if (low != vlow || high != vhigh)
    error_c (r, ("test impedence error. was %#x %#x, expected %#x %#x", high & 0xff, low & 0xff, vhigh & 0xff, vlow & 0xff));

  return r;
  exception: return 0;
}

int sw_Test16::testOutputImpedence (int xbar, char vlow1, char vhigh1, char vlow2, char vhigh2)
{
  int r = 1;
  char low1, high1, low2, high2;

  insist (this);
  
  comment_c (("testing output impedence with %#x, %#x, %#x, %#x", vlow1 & 0xff, vhigh1 & 0xff, vlow2 & 0xff, vhigh2 & 0xff));

  if (!testPort->setOutputImpedence (xbar, vlow1, vhigh1, vlow2, vhigh2) || !testPort->getOutputImpedence (xbar, &low1, &high1, &low2, &high2))
    error_c (r, ("setOutputImpedence failed"));

  if (low1 != vlow1 || high1 != vhigh1 || low2 != vlow2 || high2 != vhigh2)
    error_c (r, ("output impedence error. was %#x %#x %#x %#x, expected %#x %#x %#x %#x",
		 high1 & 0xff, low1 & 0xff, high2 & 0xff, low2 & 0xff, vhigh1 & 0xff, vlow1 & 0xff, vhigh2 & 0xff, vlow2 & 0xff));

  return r;
  exception: return 0;
}

int sw_Test16::testInitialOutputImpedence (int xbar)
{
  int r = 1;
  
  insist (this);
  
  char low1, low2, high1, high2;  
  
  comment_c (("verifying initial output impedence"));
  
  if (!testPort->getOutputImpedence (xbar, &low1, &high1, &low2, &high2))
    error_c (r, ("getOutputImpedence failed"));
  
  if (low1 != (char) 0xe1 || high1 != (char) 0xe1 || low2 != (char) 0xe1 || high2 != (char) 0xe1)
    error_c (r, ("output impedence error. was %#x %#x %#x %#x, expected e1 e1 e1 e1",
		 low1 & 0xff, high1 & 0xff, low2 & 0xff, high2 & 0xff));
  
  return r;
  exception: return 0;
}

int sw_Test16::testImpedence (int xbar)
{
  int r = 1;
  char low1, high1;
  int lowi, highi;
  double lowd, highd;
  
  insist (this);
  
  comment_c (("verifying initial test impedence"));
  
  if (!testPort->getTestImpedence (xbar, &low1, &high1))
    error_c (r, ("getTestImpedence failed"));
  
  if (low1 != (char) 0xe1 || high1 != (char) 0xe1)
    error_c (r, ("test impedence error. was %#x %#x, expected 0xe1 0xe1", high1 & 0xff, low1 & 0xff));
  
  comment_c (("verifying initial output impedence"));
  
  if (!testInitialOutputImpedence (xbar))
    error_c (r, ("testInitialOutputImpedence failed"));

  if (!testTestImpedence (xbar, 0x0, 0x0) || !testOutputImpedence (xbar, 0, 0, 0, 0))
    error_c (r, ("testOutputImpedence failed"));

  comment_c (("verifying that output impedence is within range."));
  
  if (!testPort->getVoltages (xbar, &lowi, &highi))
    error_c (r, ("getVoltages failed"));
  
  highd = (double) lowi * sw_TestPort::VSCALE10;
  lowd = (double) highi * sw_TestPort::VSCALE10;

  if (lowd > 0.01)
    error_c (r, ("low value should be <= 0.01V, was %4.2f", lowd));
  
  if (highd < 1.2)
    error_c (r, ("high value should be >= 1.2V, was %4.2f", highd));
  
  comment_c (("verifying that output impedence is within range."));

  if (!testTestImpedence (xbar, 0xff, 0xff) || !testOutputImpedence (xbar, 0xff, 0xff, 0xff, 0xff))
    error_c (r, ("testOutputImpedence failed"));

  if (!testPort->getVoltages (xbar, &lowi, &highi, 512, 1))
    error_c (r, ("getVoltages failed"));
  
  lowd = (double) lowi;
  highd = (double) highi;
  
  lowd = (50.0 * lowd) / (512 - lowd);
  highd = (50.0 * (512 - highd)) / highd;
  
  if (lowd >= 33.0)
    error_c (r, ("impedence is too high, should be <= 33, is %4.2f", lowd));
  
  if (highd >= 33.0)
    error_c (r, ("impedence is too high, should be <= 33, is %4.2f", highd));
  
  if (!testMonotonicity (xbar))
    error_c (r, ("testMonotonicity failed"));
  
  return r;
  exception: return 0;
}

int sw_Test16::testMonotonicity (int xbar)
{
  int r = 1;
  insist (this);
  int lowi, highi;
  double pul, puh, pdl, pdh;
  
  comment_c (("testing monotonicity"));
  
  for (int i = 0x02; i <= 0x40; i= i << 1)
  {
    if (!testTestImpedence (xbar, i, i) || !testOutputImpedence (xbar, i, i, i, i))
      error_c (r, ("testOutputImpedence failed"));

    if (!testPort->getVoltages (xbar, &lowi, &highi))
      error_c (r, ("getVoltages failed"));
    
    pdh = (double) lowi;
    puh = (double) highi;

    if (!testTestImpedence (xbar, i - 1, i - 1))
      error_c (r, ("testTestImpedence failed"));
    
    if (!testPort->getVoltages (xbar, &lowi, &highi))
      error_c (r, ("getVoltages failed"));
    
    pdl = (double) lowi;
    pul = (double) highi;

    if (pdh >= 512) pdh = 511;
    if (puh == 0) puh = 1;
    if (pdl >= 512) pdl = 511;
    if (pul == 0) pul = 1;

    if (pdh > pdl * 1.2)
      error_c (r, ("pulldown monotonicity error at value = 0x%04x\nv = %04x ==> z= %4.1f, v = %04x ==> z = %4.1f", i + 256 * i, i+256*i,50.0*pdh/(512-pdh),(i+256*i)-0x0101, 50.0*pdl/(512-pdl)));
    
    if (puh * 1.2 < pul)
      error_c (r, ("pullup monotonicity error at value = 0x%04x=n\nv = %04x ==> z= %4.1f, v = %04x ==> z = %4.1f",i+256*i,i+256*i,50.0*(512-puh)/puh, (i+256*i)-0x0101, 50.0*(512-pul)/pul));

    comment_c (("v=%04x: pdZ=%6.1f puZ=%6.1f, v=%04x: pdZ=%6.1f puZ=%6.1f",i+256*i, 50.0*pdh/(512-pdh),50.0*(512-puh)/puh,(i+256*i)-0x0101,50.0*pdl/(512-pdl),50.0*(512-pul)/pul));
  }
  return r;
  exception: return 0;
}

int sw_Test16::testPorts (int xbar)
{
  int r = 1;
  insist (this);

  insist (xbar >= 0 && xbar < numSwitches);

  char receive [sw_TestPort::NUM_PORTS];
  char transmit [sw_TestPort::NUM_PORTS];

  comment_c (("testing ports"));

  int correction;
  correction = getPortCorrection (xbar);
  
  for (int i = 0; i < 3; i++)
  {
    if (!testPort->getLinkStatus (xbar, receive, transmit))
      error_c (r, ("getLinkStatus failed"));
  }
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i))
      continue;
    
    if (HW_RECEIVE_NO_BEAT (receive [i - correction]))
      error_c (r, ("receive no beat set on port %d", i - correction));
    
    if (HW_RECEIVE_MISSED_3_BEATS (receive [i - correction]))
      error_c (r, ("receive missed beat set on port %d", i - correction));
    
    if (HW_RECEIVE_ILLEGAL_SYMBOL (receive [i - correction]))
      error_c (r, ("receive illegal symbol set on port %d", i - correction));
    
    if (HW_TRANSMIT_NOT_GAP (transmit [i - correction]))
      error_c (r, ("send not gap set on port %d", i - correction));

    if (HW_TRANSMIT_NOT_IN_GO (transmit [i - correction]))
      error_c (r, ("send not in go set on port %d", i - correction));

    if (HW_TRANSMIT_DATA_OUT (transmit [i - correction]))
      error_c (r, ("send data out set on port %d", i - correction));
    
    if (HW_RECEIVE_DATA_IN (receive [i - correction]))
      error_c (r, ("receive data in set on port %d", i - correction));
  }
  
  return r;
  exception: return 0;
}


int sw_Test16::testMessage (int xbar)
{
  int r = 1;

  insist (this);
  insist (host && getNetwork() && graph);
  insist (xbar >= 0 && xbar < numSwitches);
  
  {
    int length = mt_Network::MTU;  

    sw_MessageSender ms (this, getNetwork(), graph, switches [xbar], 1, 1, length);
    sw_ImpedenceReader ir (xbar, testPort);

    comment_c (("setting channel speed to 200."));

    if (!testPort->setChannel200 ())
      error_c (r, ("setChannel200 failed"));
  
    comment_c (("testing %d byte messages with impedence writing for 10 seconds...", length));

    if (!open ())
      error_c (r, ("open failed"));
  
    ms.createThread ();
    ir.createThread ();
  
    testPort->usleep (1000 * 1000 * 10);
  
    comment_c (("10 seconds is up."));
    
    int r1, r2;
  
    r1 = ms.stopThread ();
    r2 = ir.stopThread ();
  
    comment_c (("%d / %d received (%d missing)", ms.getNumReceived (), ms.getNumSent (), ms.getNumSent () - ms.getNumReceived ()));
    comment_c (("wrote impedence %d times", ir.getCount ()));
  
    if (!r1)
      error_c (r, ("message test failed"));
  
    if (!r2)
      error_c (r, ("impedence reading failed"));

    close ();
  
    comment_c (("setting channel speed to 160."));

    if (!testPort->setChannel160 ())
      error_c (r, ("setChannel160 failed"));
  
  }
  return r;
  exception: return 0;
}

int sw_Test16::testCrosspoints (int xbar)
{
  int r = 1;
  
  insist (this);
  insist (host && getNetwork() && graph);
  insist (xbar >= 0 && xbar < numSwitches);
  
  {
    comment_c (("testing crosspoints"));

    sw_CrossPointSender cs (this, getNetwork (), graph, switches [xbar], 10, mt_Network::MTU, 512, 1);

    if (!open ())
      error_c (r, ("open failed"));
    
    r = cs.startThread ();
    comment_c (("%d / %d received (%d missing)", cs.getNumReceived (), cs.getNumSent (), cs.getNumSent () - cs.getNumReceived ()));
  
    if (verbose)
      cs.printGrid ();

    if (!r)
      error_c (r, ("crosspoint test failed")); 
    
    close ();
  }
  return r;
  exception: return 0;
}

int sw_Test16::computeRoutes ()
{
  insist (this);
  return routesComputed || (routesComputed = graph->computeRoutes (host));
  exception: return 0;
}

int sw_Test16::sortSwitches ()
{
  int r = 1;
  insist (this);
  insist (graph);

  if (!computeRoutes ())
    error_c (r, ("computeRoutes failed"));

  for (int i = 0; i < numSwitches; i++)
    xbars [i] = i;
  
  for (int i = 0; i < numSwitches; i++)
  {
    for (int j = i; j < numSwitches; j++)
    {
      if (switches [j]->getHeight () > switches [i]->getHeight ())
      {
	int t = xbars [j];
	xbars [j] = i;
	i = t;
      }
    }
  }
  return 1;
  exception: return 0;
}

int sw_Test16::testCounters (int xbar, int waitForReceive, sw_Test16GetCounts getCounts)
{
  int r = 1;
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);

  int counts [sw_TestPort::NUM_PORTS];
  
  int correction;
  correction = getPortCorrection (xbar);
  if (correction < 0)
    return 0;
  
  int maxNodes;
  maxNodes = switches [xbar]->getMaxNodes ();

  if (!(this->*getCounts) (xbar, counts))
    return 0;
  
  for (int i = 0; i < maxNodes + 1; i++)
  {
    int p = i;
    
    if (i == maxNodes)
    {
      if (!(this->*getCounts) (xbar, counts))
	return 0;
      p = switches [xbar]->getIn ();
    }
    else
      if (p == switches [xbar]->getIn () || !switches [xbar]->getNode (p) || !switches [xbar]->getNode (p)->isSwitch ())
	continue;

    int count;
    count = counts [p - correction];

    comment_c (("clearing counter (%d) on xbar %d port %d by sending %d messages", count, xbar, p - correction, 0x0000ffff - count + 1));
    insist (count >=0 && count <= 0xffff);

    if (!send (xbar, waitForReceive, 0, p, 1, 0x0000ffff - count + 1))
      error_c (r, ("send failed"));
  }
  
  if (!(this->*getCounts) (xbar, counts))
    return 0;
  
  for (int i = 0; i < maxNodes; i++)
  {
    if (counts [i])
      error_c (r, ("failed to clear counters on on xbar %d port %d. count is %d", xbar, i - correction, counts [i]));
  }
  return r;
  exception: return 0;
}


int sw_Test16::testCounters (int xbar, int waitForReceive, sw_Test16GetCounts getCounts, int count)
{
  int r = 1;
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (count >=0 && count <= 0x0000ffff);

  int c1s [sw_TestPort::NUM_PORTS];
  int c2s [sw_TestPort::NUM_PORTS];
  
  int correction;
  correction = getPortCorrection (xbar);
  if (correction < 0)
    return 0;
  
  int maxNodes;
  maxNodes = switches [xbar]->getMaxNodes ();

  if (!(this->*getCounts) (xbar, c1s))
    return 0;
  
  for (int i = 0; i < maxNodes; i++)
  {
    int p = i;
    
    if (p == switches [xbar]->getIn () || !switches [xbar]->getNode (p) || !switches [xbar]->getNode (p)->isSwitch ())
      continue;

    int c1 = c1s [ p - correction];

    comment_c (("incrementing counter (%d) on xbar %d port %d", c1, xbar, p - correction));

    if (!send (xbar, waitForReceive, 0, p, 1, count))
      error_c (r, ("send failed"));
  }

  if (!(this->*getCounts) (xbar, c2s))
    return 0;
  
  for (int i = 0; i < maxNodes; i++)
  {
    int p = i;
    int c1 = c1s [p - correction];
    int c2 = c2s [p - correction];

    if (p == switches [xbar]->getIn ())
      continue;
    
    if (c2 != count + c1 && !(count == 1 && c1 == 65535  && c2 == 0))
      error_c (r, ("failed counter increment by %d on xbar %d port %d. count is %d", count, xbar, p - correction, c2));
  }
  
  {
    int p = switches [xbar]->getIn ();
    comment_c (("incrementing counter (%d) on xbar %d port %d", c2s [p], xbar, p - correction));

    if (!send (xbar, waitForReceive, 0, p, 1, count))
      error_c (r, ("send failed"));

    if (!(this->*getCounts) (xbar, c1s))
      return 0;

    int c1 = c2s [p - correction];
    int c2 = c1s [p - correction];

    if (c2 != count + c1 && !(count == 1 && c1 == 65535  && c2 == 0))
      error_c (r, ("failed counter increment by %d on xbar %d port %d. count is %d", count, xbar, p - correction, c2));
  }
  
  return r;
  exception: return 0;
}

int sw_Test16::testOuterCounters (int xbar, sw_Test16GetCount getCount, int count)
{
  int r = 1;
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);

  int correction;
  correction = getPortCorrection (xbar);
  if (correction < 0)
    return 0;
  
  int maxNodes;
  maxNodes = switches [xbar]->getMaxNodes ();
  for (int i = maxNodes - 1; i < maxNodes + 1; i++)
  {
    int p = i;
    
    if (i == maxNodes)
      p = switches [xbar]->getIn ();
    else
      if (p == switches [xbar]->getIn () || !switches [xbar]->getNode (p) || !switches [xbar]->getNode (p)->isSwitch ())
	continue;

    int c1 = (this->*getCount) (xbar, p - correction);

    comment_c (("incrementing counter (%d) on xbar %d port %d", c1, xbar, p - correction));
    insist (count >=0 && count <= 0x0000ffff);

    if (!send (xbar, 1, 0, p, 1, count))
      error_c (r, ("send failed"));
    
    int c2 = (this->*getCount) (xbar, p - correction);

    if (c2 != count + c1 && !(count == 1 && c1 == 65535  && c2 == 0))
      error_c (r, ("failed counter increment by %d on xbar %d port %d. count is %d", count, xbar, p - correction, c2));
  }
  return r;
  exception: return 0;
}

int sw_Test16::testSnakeyCounters (int xbar, int count)
{
  int r = 1;
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);

  int correction;
  correction = getPortCorrection (xbar);
  if (correction != 0)
    return 0;
  
  int maxNodes;
  maxNodes = switches [xbar]->getMaxNodes ();

  if (correction != 0 || switches [xbar]->getIn () != 0 || maxNodes != 16)
  {
    printFormat ("xbar %d is not connected right for a snakey send", xbar);
    return 0;
  }

  int c1;
  c1 = getGoodCount (xbar, 1);
  insist (count >=0 && count <= 0x0000ffff);
  
  comment_c (("incrementing counter (%d) on xbar %d ports %d through %d", c1, xbar, 1, maxNodes -1));

  int numReceived;
  if (!snakeySend (xbar, 1, 0, 1, count, &numReceived))
    error_c (r, ("send failed"));

  int badCount, badPort;
  int c2;
  c2 = this->getSomeGoodCounts (xbar, &badCount, &badPort);

  if (c2 != count + c1 && !(count == 1 && c1 == 65535  && c2 == 0))
    error_c (r, ("failed counter increment by %d on xbar %d port %d. count is %d", count, xbar, badPort, badCount));
  
  return r;
  exception: return 0;
}

int sw_Test16::getGoodCount (int xbar, int port)
{
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (port >= 0 && port < sw_TestPort::NUM_PORTS);
  
  int good [sw_TestPort::NUM_PORTS];
  int bad [sw_TestPort::NUM_PORTS];
  
  if (!testPort->getPacketCounters (xbar, good, bad))
    return -1;
  
  return good [port];
  exception: return -1;
}

int sw_Test16::getGoodCounts (int xbar, int*good)
{
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);

  int bad [sw_TestPort::NUM_PORTS];
  
  return testPort->getPacketCounters (xbar, good, bad);
  
  exception: return 0;
}

int sw_Test16::getBadCounts (int xbar, int*bad)
{
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);

  int good [sw_TestPort::NUM_PORTS];
  
  return testPort->getPacketCounters (xbar, good, bad);
  
  exception: return 0;
}

int sw_Test16::getSomeGoodCounts (int xbar, int*badCount, int*badPort)
{
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (badCount && badPort);
  
  int good [sw_TestPort::NUM_PORTS];
  int bad [sw_TestPort::NUM_PORTS];
  
  if (!testPort->getPacketCounters (xbar, good, bad))
    return -1;
  
  int count;
  count = good [1];
  
  for (*badPort = 1; *badPort < sw_TestPort::NUM_PORTS; (*badPort)++)
    if ((*badCount = good [*badPort]) != count)
      return -1;
  
  return count;
  exception: return -1;
}

int sw_Test16::getBadCount (int xbar, int port)
{
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (port >= 0 && port < sw_TestPort::NUM_PORTS);
  
  int good [sw_TestPort::NUM_PORTS];
  int bad [sw_TestPort::NUM_PORTS];
  
  if (!testPort->getPacketCounters (xbar, good, bad))
    return -1;
  
  return bad [port];
  exception: return -1;
}

int sw_Test16::enableCrc ()
{
  int r;
  
  insist (this);
  
  if (!open ())
    return 0;


  r = lanai >= 8 ? getNetwork ()->writeRegister (MYRINET_REGISTER, CRC32_ENABLED) : getNetwork ()->enableCrc ();
  close ();
  return r;
  exception: return 0;
}

int sw_Test16::disableCrc ()
{
  int r;
  
  insist (this);
  
  if (!open ())
    error_c (r, ("open failed"));

  r = lanai >= 8 ? getNetwork ()->writeRegister (MYRINET_REGISTER, CRC32_DISABLED) : getNetwork ()->enableCrc ();
  close ();
  return r;
  exception: return 0;
}

int sw_Test16::testCounters ()
{
  int r = 1;
  insist (this);

  if (!sortSwitches ())
    error_c (r, ("sortSwitches failed"));

  comment_c (("testing wrapping of good counters"));

  reopen ();
  
  if (!enableCrc ())
    error_c (r, ("enableCrc failed"));
  
  for (int i = 0; i < numSwitches; i++)
    if (!testCounters (xbars [i], 1, mt_fp (sw_Test16::getGoodCounts)))
      error_c (r, ("test of good counters failed on xbar %d", xbars [i]));

  comment_c (("testing wrapping of bad counters"));

  if (!disableCrc ())
    error_c (r, ("disableCrc failed"));

  for (int i = 0; i < numSwitches; i++)
    if (!testCounters (xbars [i], 0, mt_fp (sw_Test16::getBadCounts)))
      error_c (r, ("test of bad counters failed on xbar %d", xbars [i]));

  comment_c (("testing good counting in powers of 2"));

  if (!enableCrc ())
    error_c (r, ("enableCrc failed"));
  
#define SNAKEY 1
#ifdef SNAKEY
  for (int i = 0; i < numSwitches; i++)
  {
    int size = 1;
    for (int j = 0; j < 16; j++, size *= 2)
      if (!testSnakeyCounters (xbars [i], size))
	error_c (r, ("test of good counters failed on xbar %d", xbars [i]));

    if (!testSnakeyCounters (xbars [i], 1))
      error_c (r, ("test of good counters failed on xbar %d", xbars [i]));

    size = 1;
    for (int j = 0; j < 16; j++, size *= 2)
      if (!testOuterCounters (xbars [i],  mt_fp (sw_Test16::getGoodCount), size))
	error_c (r, ("test of good counters failed on xbar %d", xbars [i]));

    if (!testOuterCounters (xbars [i], mt_fp (sw_Test16::getGoodCount), 1))
      error_c (r, ("test of good counters failed on xbar %d", xbars [i]));
  }
#else
  for (int i = 0; i < numSwitches; i++)
  {
    int size = 1;
    for (int j = 0; j < 16; j++, size *= 2)
      if (!testCounters (xbars [i], 1, mt_fp (sw_Test16::getGoodCount), size))
	error_c (r, ("test of good counters failed on xbar %d", xbars [i]));

    if (!testCounters (xbars [i], 1, mt_fp (sw_Test16::getGoodCount), 1))
      error_c (r, ("test of good counters failed on xbar %d", xbars [i]));
  }
#endif


  comment_c (("testing bad counting in powers of 2"));

  if (!disableCrc ())
    error_c (r, ("disableCrc failed"));

  for (int i = 0; i < numSwitches; i++)
  {
    int size = 1;
    for (int j = 0; j < 16; j++, size *= 2)
      if (!testCounters (xbars [i], 0, mt_fp (sw_Test16::getBadCounts), size))
	error_c (r, ("test of bad counters failed on xbar %d", xbars [i]));

    if (!testCounters (xbars [i], 0, mt_fp (sw_Test16::getBadCounts), 1))
      error_c (r, ("test of bad counters failed on xbar %d", xbars [i]));
  }
  
  if (!enableCrc ())
    error_c (r, ("enableCrc failed"));

  return r;
  exception: return 0;
}


int sw_Test16::getPortCorrection (int xbar)
{
  int r = 1;
  
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);

  if (portCorrectionXbar == xbar)
    return portCorrection;
  
  if (!computeRoutes ())
    return -1;

  //return portCorrection = 0;
  
  int port;
  port = switches [xbar]->getIn ();  

  if (!send (xbar, 1, 1, port, 1, 5))
    error_c (r, ("send failed"));

  int good1 [sw_TestPort::NUM_PORTS];
  int good2 [sw_TestPort::NUM_PORTS];
  int bad [sw_TestPort::NUM_PORTS];

  if (!testPort->getPacketCounters (xbar, good1, bad))
    return -1;

  if (!send (xbar, 1, 1, port, 1, 1))
    error_c (r, ("send failed"));
  
  if (!testPort->getPacketCounters (xbar, good2, bad))
    return -1;
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (good1 [i] != good2 [i])
    {
      insist (port >= i);
      for (int j = i + 1; j < sw_TestPort::NUM_PORTS; j++)
      {
	insist (good1 [j] == good2 [j]);
      }
      portCorrectionXbar = xbar;
      portCorrection = port - i;
      comment_c (("port correction is %d", portCorrection));
      return portCorrection;
    }
  }  
  exception: return -1;
}


int sw_Test16::send (int xbar, int receiveWait, int sendWait, int port, int hop, int size, int count, int*numReceived)
{
  int r = 1;
  
  insist (this);
  insist (hop == mt_Route::EVIL_HOP || (hop >= - sw_TestPort::NUM_PORTS && hop <= sw_TestPort::NUM_PORTS));
  
  {
  sw_RouteSender rs (this, getNetwork (), graph, receiveWait, sendWait, size, count);

  mt_Route r1;
  if (!rs.getPortRoute (&r1, switches [xbar], port))
    error_c (r, ("getPortRoute failed"));
  
  mt_Route r2 (&r1, hop);
  
  if (!rs.setRoute (&r2))
    error_c (r, ("setRoute failed"));
  
  //printFormat ("using route %s", r2.toString ());
  
  if (!open ())
    error_c (r, ("open failed"));
  
  r = rs.startThread ();
  close ();
  
  if (numReceived) *numReceived = rs.getNumReceived ();
  
  return r;
  }
  exception: return 0;
}


int sw_Test16::snakeySend (int xbar, int receiveWait, int sendWait, int size, int count, int*numReceived)
{
  int r = 1;
  
  insist (this);
  
  {
    sw_SnakeySender ss (this, getNetwork (), graph, switches [xbar], receiveWait, sendWait, size, count);

    if (!open ())
      error_c (r, ("open failed"));
    
    r = ss.startThread ();
    close ();
    
    if (numReceived) *numReceived = ss.getNumReceived ();
    
    return r;
  }
  exception: return 0;
}


int sw_Test16::deadlock (int xbar, int port)
{
  int r = 1;
  mt_Route r1;
  sw_RouteSender rs (this, getNetwork (), graph, 0, 0, mt_Network::MTU, 1);

  insist (this);  

  if (!rs.getDeadlockRoute (&r1, switches [xbar], port))
    error_c (r, ("getDeadlockRoute failed"));
  
  if (!rs.setRoute (&r1))
    error_c (r, ("setRoute failed"));
  
  if (!open ())
    error_c (r, ("open failed"));  

  r = rs.startThread ();
  close ();
  
  return r && rs.getNumReceived () == 0;
  
  exception: return 0;
}



int sw_Test16::send (int xbar, int receiveWait, int sendWait, int port, int size, int count, int*numReceived)
{
  int r = 1;
  insist (this);
  {
  sw_PortSender ps (this, getNetwork (), graph, switches [xbar], receiveWait, sendWait, port, size, count);

  if (!open ())
    error_c (r, ("open failed"));
  
  r = ps.startThread ();
  close ();
  
  if (numReceived) *numReceived = ps.getNumReceived ();
  }
  return r;
  exception: return 0;
}


int sw_Test16::send (int xbar, int receiveWait, int sendWait, int size, int count, int*numReceived)
{
  int r = 1;
  sw_MessageSender ms (this, getNetwork (), graph, switches [xbar], receiveWait, sendWait, 1, count);
  insist (this);

  if (!open ())
    error_c (r, ("open failed"));  

  r = ms.startThread ();
  close ();
  
  if (numReceived) *numReceived = ms.getNumReceived ();
  
  printFormat ("%d / %d", ms.getNumSent (), ms.getNumReceived ());
  
  return r;
  exception: return 0;
}


int sw_Test16::testIllegal (int xbar, int port)
{
  int r = 1;
  
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (port >= 0 && port < sw_TestPort::NUM_PORTS);  

  int pport;
  pport  = port - getPortCorrection (xbar);

  insist (pport >= 0 && pport < sw_TestPort::NUM_PORTS);  
  
  comment_c (("testing illegal symbols on xbar %d port %d", xbar, pport));
  
  char control [sw_TestPort::NUM_PORTS];
  memset (control, 0, sizeof (control));

  if (!testPort->setControl (xbar, control) || !testPort->setRelays (xbar, 0xffff))
    error_c (r, ("setControl or setRelays failed"));

  getIllegalSymbols (xbar, 0);
  getIllegalSymbols (xbar, 0);
  
  comment_c (("testing ilegal symbol not being set"));

  if (!send (xbar, 0, 1, port, 1, 5))
    error_c (r, ("send failed"));

  if (getIllegalSymbols (xbar, pport))
    error_c (r, ("illegal symbol detected on port %d", pport));

  comment_c (("testing ilegal symbol being set"));

  if (!testPort->setRelays (xbar, 0xffff ^ (1 << pport)))
    error_c (r, ("setRelays failed"));
  
  if (!send (xbar, 0, 1, port, 1, 5))
    error_c (r, ("send failed"));

  if (!getIllegalSymbols (xbar, pport))
    error_c (r, ("illegal symbols not detected on port %d", pport));

  comment_c (("testing clearing ilegal symbols"));

  getIllegalSymbols (xbar, pport);
  
  if (getIllegalSymbols (xbar, pport))
    error_c (r, ("illegal symbols not detected on port %d", pport));

  comment_c (("testing responding to ilegal symbols"));

  control [pport] = 0x10;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));
  
  if (!send (xbar, 0, 1, port, 1, 5))
    error_c (r, ("send failed"));

  if (!getPortShutdown (xbar, pport))
    error_c (r, ("not reponding correctly to illegal chars on port %d", pport));

  if (!testPort->setRelays (xbar, 0xffff))
    error_c (r, ("setRelays failed"));

  int numReceived;

  comment_c (("testing dropping packets after illegal symbols"));

  if (!send (xbar, 0, 1, port, 1, 5, &numReceived))
    error_c (r, ("send failed"));

  if (numReceived)
    error_c (r, ("not dropping packets after illegal symbol"));

  comment_c (("testing port status updated"));

  control [pport] = 0xc0;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  if (!getStatusUpdate (xbar, pport))
    error_c (r, ("status not updated on port %d", pport));

  comment_c (("testing dropping packets again"));

  if (!send (xbar, 0, 1, port, 1, 5, &numReceived))
    error_c (r, ("send failed"));

  if (numReceived)
    error_c (r, ("not dropping packets again"));

  comment_c (("testing not dropping packets after turned back on"));

  control [pport] = 0x00;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  if (!send (xbar, 1, 1, port, 1, 100))
    error_c (r, ("dropping packets after turned back on"));

  return r;
  exception: return 0;
}

int sw_Test16::testIllegal (int xbar)
{
  int r = 1;
  
  insist (this);

  comment_c (("testing ilegal symbols"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (i != switches [xbar]->getIn () &&
	(!switches [xbar]->getNode (i) || 
	 !switches [xbar]->getNode (i)->isSwitch ()))
      continue;
    
    if (!testIllegal (xbar, i))
      error_c (r, ("testIllegal failed"));
  }
  
  return r;  
  exception: return 0;
}

int sw_Test16::testInvalidHeads (int xbar, int port)
{
  int r = 1;
  
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (port >= 0 && port < sw_TestPort::NUM_PORTS);  

  int pport;
  pport  = port - getPortCorrection (xbar);
  insist (pport >= 0 && pport < sw_TestPort::NUM_PORTS);  

  char control [sw_TestPort::NUM_PORTS];
  memset (control, 0, sizeof (control));
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));
  
  comment_c (("testing invalid heads on xbar %d port %d", xbar, pport));

  for (int i = - sw_TestPort::NUM_PORTS; i < sw_TestPort::NUM_PORTS; i++)
  {
    int p = pport + i;
    int lp = p + getPortCorrection (xbar);
    
    if (p < 0 || p >= sw_TestPort::NUM_PORTS || lp != switches [xbar]->getIn () && i == 0)
      continue;

    insist (lp >= 0 && lp < sw_TestPort::NUM_PORTS);

    if (!switches [xbar]->getNode (lp))
    {
      comment_c (("skipping physical port %d because it is not connected", p));
      continue;
    }
    
    comment_c (("testing valid hop %d", i));

    if (!send (xbar, 0, 0, port, i, 1, 5))
      error_c (r, ("send to valid hop failed"));
  }
  
  if (getInvalidHeads (xbar, pport))
    error_c (r, ("incorrectly detected an invalid routing byte on port %d", pport));

  comment_c (("testing clearing invalid heads"));

  getInvalidHeads (xbar, pport);
  
  if (getInvalidHeads (xbar, pport))
    error_c (r, ("invalid heads not cleared on port %d", pport));

  comment_c (("testing invalid head too small (to physical port %d)", port - pport - 1));

  if (!send (xbar, 0, 0, port, - pport - 1, 1, 5))
    error_c (r, ("send failed"));

  if (!getInvalidHeads (xbar, pport))
    error_c (r, ("didn't detect an invalid routing byte on port %d", pport));

  comment_c (("testing clearing invalid heads again"));

  getInvalidHeads (xbar, pport);
  
  if (getInvalidHeads (xbar, pport))
    error_c (r, ("invalid heads not cleared on port %d", pport));

  comment_c (("testing invalid head too big (to physical port %d)", port + sw_TestPort::NUM_PORTS - pport));

  if (!send (xbar, 0, 0, port, + sw_TestPort::NUM_PORTS - pport, 1, 5))
    error_c (r, ("send failed"));

  if (!getInvalidHeads (xbar, pport))
    error_c (r, ("didn't detect an invalid routing byte on port %d", pport));

  comment_c (("testing clearing invalid heads yet again"));

  getInvalidHeads (xbar, pport);
  
  if (getInvalidHeads (xbar, pport))
    error_c (r, ("invalid heads not cleared on port %d", pport));

  comment_c (("testing invalid head 0xa0"));
  
  reopen ();
  
  if (!send (xbar, 0, 0, port, 0xa0, 1, 5))
    error_c (r, ("send failed"));

  if (!getInvalidHeads (xbar, pport))
    error_c (r, ("didn't detect an invalid routing byte on port %d", pport));

  return r;
  exception: return 0;
}

int sw_Test16::testInvalidHeads (int xbar)
{
  int r = 1;
  
  insist (this);

  comment_c (("testing invalid heads"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (i != switches [xbar]->getIn () &&
	(!switches [xbar]->getNode (i) || 
	 !switches [xbar]->getNode (i)->isSwitch ()))
    {
      comment_c (("skipping port %d because it is disconnected", i));
      continue;
    }

    if (!testInvalidHeads (xbar, i))
      error_c (r, ("testInvalidHeads failed"));
  }
  
  return r;  
  exception: return 0;
}

int sw_Test16::testTimeouts (int xbar, int port)
{
  int r = 1;
  
  insist (this);
  insist (xbar >= 0 && xbar < numSwitches);
  insist (port >= 0 && port < sw_TestPort::NUM_PORTS);  

  int pport;
  pport  = port - getPortCorrection (xbar);
  insist (pport >= 0 && pport < sw_TestPort::NUM_PORTS);  

  char control [sw_TestPort::NUM_PORTS];
  memset (control, 0, sizeof (control));
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));
  
  comment_c (("testing timeouts on xbar %d port %d", xbar, pport));

  if (!deadlock (xbar, port))
    error_c (r, ("failed to deadlock on port %d", pport));

  comment_c (("testing status"));
  
  getTimeout (xbar, pport, 1);
  getTimeout (xbar, pport, 1);

  if (!getTimeout (xbar, pport, 1))
    error_c (r, ("didn't get expected status thing on %d", pport));

  control [pport] = 0xc0;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  comment_c (("sleeping for 1/10th second"));

  testPort->usleep (1000 * 100);

  comment_c (("testing status again"));

  if (!getTimeout (xbar, pport, 0))
    error_c (r, ("didn't get expected status thing on %d", pport));

  return r;
  exception: return 0;
}


int sw_Test16::testTimeouts (int xbar)
{
  int r = 1;
  
  insist (this);

  comment_c (("testing timeouts"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (!switches [xbar]->getNode (i) || !switches [xbar]->getNode (i)->isSwitch () || switches [xbar]->getIn () == i)
      continue;
    
    if (!testTimeouts (xbar, i))
      error_c (r, ("testTimeouts failed"));
  }
  
  return r;  
  exception: return 0;
}


int sw_Test16::testPortOnOff (int xbar, int port)
{
  int r = 1;
  
  insist (this);
  insist (port >= 0 && port < sw_TestPort::NUM_PORTS);  

  int pport;
  pport  = port - getPortCorrection (xbar);
  insist (pport >= 0 && pport < sw_TestPort::NUM_PORTS);

  char control [sw_TestPort::NUM_PORTS];
  memset (control, 0, sizeof (control));

  control [pport] = 0x00;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  comment_c (("sending 5 packets and ignoring the result"));
  
  if (!send (xbar, 0, 1, port, 1, 5))
    error_c (r, ("send failed"));  

  comment_c (("testing sending of 100 packets"));
  
  if (!send (xbar, 1, 0, port, 1, 100))
    error_c (r, ("sending of 100 packets failed"));

  comment_c (("turning transmit off"));
  
  control [pport] = 0x80;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));  

  comment_c (("reading packet counters"));
  
  int good1 [sw_TestPort::NUM_PORTS];
  int good2 [sw_TestPort::NUM_PORTS];
  int bad [sw_TestPort::NUM_PORTS];

  if (!testPort->getPacketCounters (xbar, good1, bad))
    return -1;
  
  comment_c (("packet counter is %d", good1 [pport]));
  
  comment_c (("testing dropping of 100 packets"));
  
  int numReceived;
  
  if (!send (xbar, 0, 0, port, 1, 100, &numReceived) || numReceived)
    error_c (r, ("dropping of 100 packets failed"));

  comment_c (("reading packet counters again"));

  if (!testPort->getPacketCounters (xbar, good2, bad))
    return -1;

  if (port != switches [xbar]->getIn () && good1 [pport] != good2 [pport])
    error_c (r, ("packet counter changed (to %d)", good2 [pport]));

  comment_c (("turning port back on"));
  
  control [pport] = 0x00;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  comment_c (("testing sending of 100 packets"));
  
  if (!send (xbar, 1, 0, port, 1, 100))
    error_c (r, ("sending of 100 packets failed"));

  comment_c (("turning receive off"));
  
  control [pport] = 0x40;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  if (!testPort->getPacketCounters (xbar, good1, bad))
    return -1;

  comment_c (("testing dropping of 100 packets"));

  if (!send (xbar, 0, 0, port, 1, 100, &numReceived) || numReceived)
    error_c (r, ("dropping of 100 packets failed"));

  if (!testPort->getPacketCounters (xbar, good2, bad))
    return -1;

  if ((good2 [pport] & 0xffff) != (good1 [pport] + 100) & 0xffff)
    error_c (r, ("packet counter didn't increase by 100 (changed by %d)", good2 [pport] - good1 [pport]));

  comment_c (("turning port back on"));
  
  control [pport] = 0x00;
  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  comment_c (("testing sending of 100 packets"));
  
  if (!send (xbar, 1, 0, port, 1, 100))
    error_c (r, ("sending of 100 packets failed"));
  
  return r;
  
  exception: return 0;
}

int sw_Test16::testPortOnOff (int xbar)
{
  int r = 1;
  
  insist (this);

  comment_c (("testing port on/off"));

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (i != switches [xbar]->getIn () &&
	(!switches [xbar]->getNode (i) || 
	 !switches [xbar]->getNode (i)->isSwitch ()))
      continue;
    
    if (!testPortOnOff (xbar, i))
      error_c (r, ("testPortOnOff failed"));
  }
  
  return r;  
  exception: return 0;
}


int sw_Test16::testBandwidth (int xbar)
{
  insist (this);
  
  comment_c (("testing bandwidth for no reason"));
  comment_c (("%5.2f (mbps)", getBandwidth (xbar)));  

  return 1;
  exception: return 0;
}

double sw_Test16::getBandwidth (int xbar, int port)
{
  int r = 1;
  sw_BandwidthSender bs (this, getNetwork (), graph, switches [xbar], 1000, port);

  insist (this);

  comment_c (("sending anywhere from 1000 to 16,000 messages to test bandwidth"));
  
  if (!open ())
    error_c (r, ("open failed"));

  r = bs.startThread ();
  
  close ();
  
  return bs.getBandwidth ();
  
  exception: return 0;
}

int sw_Test16::getIllegalSymbols (int xbar, int port)
{
  int r = 1;
  insist (this);

  char receive [sw_TestPort::NUM_PORTS];
  char transmit [sw_TestPort::NUM_PORTS];

  if (!testPort->getLinkStatus (xbar, receive, transmit))
    error_c (r, ("getLinkStatus failed"));

  return receive [port] & 0x80;
  
  exception: return 0;
}

int sw_Test16::getPortShutdown (int xbar, int port)
{
  int r = 1;
  insist (this);

  char receive [sw_TestPort::NUM_PORTS];
  char transmit [sw_TestPort::NUM_PORTS];

  if (!testPort->getLinkStatus (xbar, receive, transmit))
    error_c (r, ("getLinkStatus failed"));

  return receive [port] & 0x93 == 0x90;
  
  exception: return 0;
}

int sw_Test16::getStatusUpdate (int xbar, int port)
{
  int r = 1;
  insist (this);

  char receive [sw_TestPort::NUM_PORTS];
  char transmit [sw_TestPort::NUM_PORTS];

  if (!testPort->getLinkStatus (xbar, receive, transmit))
    error_c (r, ("getLinkStatus failed"));

  return receive [port] & 0x90 == 0x80;
  exception: return 0;
}

int sw_Test16::getInvalidHeads (int xbar, int port)
{
  int r = 1;
  insist (this);

  char receive [sw_TestPort::NUM_PORTS];
  char transmit [sw_TestPort::NUM_PORTS];

  if (!testPort->getLinkStatus (xbar, receive, transmit))
    error_c (r, ("getLinkStatus failed"));

  return receive [port] & 0x02;

  exception: return 0;
}

int sw_Test16::getTimeout (int xbar, int port, int set)
{
  int r = 1;
  insist (this);

  char receive [sw_TestPort::NUM_PORTS];
  char transmit [sw_TestPort::NUM_PORTS];

  if (!testPort->getLinkStatus (xbar, receive, transmit))
    error_c (r, ("getLinkStatus failed"));

  return set ?
    (receive [port] & 0x0c) == 0x0c && (transmit [port] & 0xc0) == 0xc0 :
    (receive [port] & 0x0c) == 0x0 && (transmit [port] & 0xc0) == 0x00;
  
  exception: return 0;
}

int sw_Test16::initialize (int xbar)
{
  int r = 1;
  insist (this);

  comment_c (("initializing switch"));
  
  char control [sw_TestPort::NUM_PORTS];

  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
    control [i] = 0xc0;

  if (!testPort->setControl (xbar, control))
    error_c (r, ("setControl failed"));

  enableCrc ();
  testPort->enableImpedence (xbar);
  testPort->enableUpdate (xbar);

  open ();
  
  comment_c (("sleeping for 2 seconds"));
  
  testPort->usleep (1000 * 1000 * 2);

  exception: return 0;
}

int sw_Test16::printBadCounters (int xbar)
{
  insist (this);
  insist (testPort);
  int good;
  good = 1;
  
  sw_TestPortState states [sw_TestPort::NUM_PORTS];
  
  if (!testPort->getPortStates (xbar, states))
    return 0;
  
  for (int i = 0; i < sw_TestPort::NUM_PORTS; i++)
  {
    if (states [i].dead_routes)
    {
      printFormat ("port %d dead_routes: %d", i, states [i].dead_routes);
      good = 0;
    }
    
    
    if (states [i].bad_low)
    {
      printFormat ("port %d bad_low: %d", i, states [i].bad_low);
      good = 0;
    }
    
    if (states [i].bad_high)
    {
      printFormat ("port %d bad_high: %d", i, states [i].bad_high);
      good = 0;
    }

    if (states [i].bad_routes)
    {
      printFormat ("port %d bad_routes: %d", i, states [i].bad_routes);
      good = 0;
    }

    if (states [i].illegal_symbols)
    {
      printFormat ("port %d illegal_symbols: %d", i, states [i].illegal_symbols);
      good = 0;
    }
  }
  return good;
  exception: return 0;  
}

int sw_Test16::printResult (int result)
{
  insist (this);

  if (result)
  {

    printFormat ("test passed.");
    printFormat (" #####     ##     ####    ####");
    printFormat (" #    #   #  #   #       #");
    printFormat (" #    #  #    #   ####    ####");
    printFormat (" #####   ######       #       #");
    printFormat (" #       #    #  #    #  #    #");
    printFormat (" #       #    #   ####    ####");
  }
  else
  {
    printFormat ("test failed.");
    printFormat (" ######    ##       #    #");
    printFormat (" #        #  #      #    #");
    printFormat (" #####   #    #     #    #");
    printFormat (" #       ######     #    #");
    printFormat (" #       #    #     #    #");
    printFormat (" #       #    #     #    ######");
  }
  
  return 1;
  exception: return 0;
}

void sw_Test16::receive (int, char*, int)
{
  insist (this);
  insist (0);
  
  exception:;
}


void sw_Test16::dump (FILE*fp)
{
  insist (this);
  insist (fp);
  insist (0);
  
  exception:;
}

int sw_Test16::willWait ()
{
  insist (this);
  return 1;
  exception: return 0;
}

int sw_Test16::start ()
{
  insist (this);
  insist (0);
  
  exception: return 0;
}
