/*
  mm_Mapper.c
  myricom mapper
  finucane@myri.com (David Finucane)
*/

#include <time.h>
#include <string.h>
#include <stdlib.h>

#include "insist.h"
#include "mt_Network.h"
#include "mm_Mapper.h"
#include "mm_Host.h"
#include "mm_Cloud.h"

mm_Mapper::mm_Mapper (mt_Node*node, mt_Network*network, mt_Calculator*calculator,
		      mt_MapperOptions&options)
  : mt_Mapper (node, network, calculator, options)
{
  for (int i = 0; i < mt_Message::NUM_TYPES; i++)
    clearHandler (i);
  
  currentGraph = oldGraph = 0;
  phase = 1;
}

mm_Mapper::~mm_Mapper ()
{  
}

void mm_Mapper::dump (FILE*)
{
}

void mm_Mapper::setHandler (int type, mm_MessageHandler handler)
{
  insist (this);
  insist (handler != 0);
  insist (type >= 0 && type < mt_Message::NUM_TYPES);
  
  handlers [type] = handler;
  
  exception: return;
}

void mm_Mapper::clearHandler (int type)
{
  insist (this);
  insist (type >= 0 && type < mt_Message::NUM_TYPES);
  
  handlers [type] = mt_fp (mm_Mapper::nullHandler);

  exception: return;
}


void mm_Mapper::callHandler (int type, mt_Message*m, int length)
{
  (this->*handlers [type])(m, length);
}

void mm_Mapper::nullHandler (mt_Message*, int)
{
}

void mm_Mapper::handleScout (mt_Message*m, int length)
{
  mt_Address a;

  insist (this);
  insist (m);

  mt_ScoutMessage*sm;
  sm = (mt_ScoutMessage*) m;
  
  mt_mapper_c (("got a scout message"));
  
  if (length < (int) sizeof (mt_ScoutMessage) - mt_Route::MAX_ROUTE)
  {
    counters.badLengths++;
    mt_mapper_w (("bad length"));
    if (options.verbose)
      sm->dump (length);
    return;
  }
  sm->swap ();
  insist (sm->subtype == mt_Message::SCOUT);
  
  a.fromBytes (sm->address);

  mt_mapper_c (("from %s", a.toString ()));
  
  if (higherPriority (&a, sm->level, &address, options.level))
  {
    mt_mapper_c (("%s is higher than me. stopping.", a.toString ()));
    done = 1;
    active = 0;
  }
  
  exception: return;
}


void mm_Mapper::handleReply (mt_Message*m, int length)
{
  mt_ReplyMessage*rm = (mt_ReplyMessage*) m;
  mt_Address a;
  char buffer [30];
  
  insist (this);
  insist (m && length);
  insist (currentGraph);

  mt_mapper_c (("got a reply message"));

  if (length < (int) sizeof (mt_ReplyMessage) - mt_Message::HOSTNAME_LENGTH)
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      rm->dump (length);
    
    counters.badLengths++;
    return;
  }
  
  rm->swap ();

  char*hn;
  hn = rm->getHostname ();
  
  insist (rm->subtype == mt_Message::REPLY);
  if (rm->phase != phase)
  {
    mt_mapper_w (("late message"));
    counters.badPhases++;
    increaseTimeouts (rm->subtype, rm->phase);
    return;
  }
  
  a.fromBytes (rm->address);
  if (a.isZero ())
  {
    mt_mapper_w (("empty myrinet address in packet"));
    counters.badContents++;
    return;
  }
  
  
  mt_mapper_c (("got a reply on port %d from address %s", rm->port, a.toString (buffer)));
  mt_mapper_c (("reply option is %d", rm->option));

  if (rm->gmId == 0)
  {
    mt_mapper_c (("GM ID in reply packet was 0. This node (%s) hasn't been configured yet.", rm->getHostname ()));
    numUnconfiguredHosts++;
  }

  if (higherPriority (&a, rm->level, &address, options.level))
  {
    mt_mapper_c (("replying host has a higher priority than me."));
    if (rm->mapperRunning)
    {
      mt_mapper_c (("other node has a mapper running. I'm going passive."));
      done = 1;
      active = 0;
    }
    else 
    {  
      mt_mapper_c (("and I don't care"));
    }
  }

  if (duplicate)
    return;
  
  insist (!duplicate && currentSwitch);
  int p;
  p = rm->port;
  if (p < mm_Switch::MIN_PORT || p >= mm_Switch::MAX_PORT)
  {
    mt_mapper_w (("bad port number in packet."));
    counters.badContents++;
    return;
  }

  mm_Node*n;
  if ((n = (mm_Node*) currentGraph->getHost (&a)) || (n = (mm_Node*) currentGraph->getNode (mt_Node::CLOUD, &a)))
  {
    /*duplicate switch*/

    insist (previousSwitch);
    
    mm_Switch*s = (mm_Switch*) n->mGetNode (0);
    insist (s && s->isSwitch ());

    mt_mapper_c (("I have seen node %s before (on switch %s)", n->getName (), s->getName()));
    mt_mapper_c (("the hostname in the reply packet was %s", rm->getHostname ()));
    
    int sp = n->mGetOpposite (0) - p;
    int pp = previousSwitch->getPort (currentSwitch);

    insist (sp >= mm_Switch::MIN_PORT && sp < mm_Switch::MAX_PORT);
    insist (pp >= mm_Switch::MIN_PORT && pp < mm_Switch::MAX_PORT);

    mt_mapper_c (("switch %s has been seen before", currentSwitch->getName ()));
    
    if (s != currentSwitch)
    {
      int r = merge (currentGraph, -sp, s, currentSwitch);
      insist (r);

    }
    else
    {
      insist (!s->mGetNode (sp));
      n->setMapVersion (rm->mapVersion);
      
      s->mConnect (sp, pp, previousSwitch);
      previousSwitch->mConnect (pp, sp, s);
    
      mt_mapper_c (("deleting %s.", s->getName ()));
      insist (s->getNumNodes () == 1);
      insist (!queue.isQueued (s));
      currentGraph->remove (s);
      currentSwitch = 0;

    }
    duplicate = 1;
    clearHandler (mt_Message::REPLY);
    return;
  }  

  if (rm->option == mt_Message::CLOUD_OPTION)
  {
    
    mt_mapper_c (("it is a cloud node"));
    mt_CloudReplyMessage*crm = (mt_CloudReplyMessage*) rm;

    rm->swap ();
    crm->swap ();

    if (length < (int) sizeof (mt_CloudReplyMessage))
    {
      mt_mapper_w (("bad length"));
      if (options.verbose)
	crm->dump (length);
    
      counters.badLengths++;
      return;
    }
  
    if (crm->numHosts < 0)
    {
      mt_mapper_w (("bad contents"));
      counters.badContents++;
      return;
    }
      
    mt_Route r (currentSwitch->getRoute (0), p);
    mm_Cloud*c = new mm_Cloud (&a, &r, crm->numHosts + 1);

    insist (currentSwitch && !currentSwitch->mGetNode (p));
    c->mConnect (0, p, currentSwitch);
    currentSwitch->mConnect (p, 0, c);
    currentGraph->add (c);
    addCloudHosts (currentGraph, c, crm);

    return;
  }   

  if (!hostTable->resolve (rm->gmId, rm->hostType, &a))
  {
    mt_mapper_w (("id conflict gmId %d, address %s", rm->gmId, a.toString ()));
    mt_Node*other = currentGraph->getGmId (rm->gmId);

    if (other)
    {
      mt_mapper_w (("this id has been seen before on host %s", other->getName ()));
    }
    else
    {
      mt_mapper_w (("this may be just a conflict with the hosts file."));
    }
    
    counters.gmIdConflicts++;
  }
  
  formatHostname (hn, 0);
  if (*hn)
    hostTable->setHostname (&a, hn);
  else
    hn = hostTable->getHostname (&a);
  mm_Host*h;

  insist (hn && *hn);
  insist (!a.isZero ());

  mt_mapper_c (("new host %s type %d", hn, rm->hostType));
  mt_mapper_c (("its current gm id is %d", rm->gmId));
  
  hostTable->setOption (&a, rm->option);

  if (!(rm->option & mt_Message::PACKED_ROUTES_OPTION))
  {
    mt_mapper_c (("this host is expecting obsolete configuration messages"));
  }

  if (!options.makeHosts && !rm->gmId)
  {
    mt_mapper_c (("makeHosts option disabled. host %s cannot be configured.", hn));
  }


  h = new mm_Host (hn , rm->hostType);
  insistp (h, ("mm_Mapper::handleReply: alloc failed"));
  h->setAddress (&a);
  h->setGmId (rm->gmId);
  h->setMapVersion (rm->mapVersion);
  
  mt_mapper_c (("setting %s mapVersion to %d", h->getName (), rm->mapVersion)); 

  h->setRoute (0, mt_Route (currentSwitch->getRoute (0), p));
  insist (currentSwitch && !currentSwitch->mGetNode (p));
  h->mConnect (0, p, currentSwitch);
  currentSwitch->mConnect (p, 0, h);
  currentGraph->add (h);
  exception: return;
}

int mm_Mapper::firstHost (mm_Graph*graph)
{
  mt_Address a;
  char*p = 0;
  int length = 0;
  int size;
  mt_Route r;
  mt_ScoutMessage m (0, ++phase, &r, &address, options.level);
  mt_Network*network = getNetwork ();
  
  //m.physical ();
  
  size = m.getSize ();
  m.swap ();

  mt_mapper_c (("looking for directly connected host"));

  int i;
  for (i = 0; i < options.scoutTries; i++)
  {
    if (!send (&r, (char*) &m, size))
    {
      mt_mapper_w (("send failed. aborting"));
      network->abort ();
    }
    
    counters.sends [mt_Message::SCOUT]++;
    network->setTimer (this, options.scoutMicroseconds);
    if (wait (mt_Network::RECEIVE, mt_Message::REPLY, phase, &p, &length))
      break;
  }
  if (i == options.scoutTries)
  {
    mt_mapper_c (("didn't find directly connected host"));
    return 0;
  }
  network->clearTimer (this);
  insist (p);
  if (length < (int) sizeof (mt_ReplyMessage) - mt_Message::HOSTNAME_LENGTH)
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      ((mt_Message*)p)->dump (length);
    
    counters.badLengths++;
    network->freePacket (p);
    return 0;
  }

  mt_ReplyMessage*t;
  t = (mt_ReplyMessage*) p;

  t->swap ();

  char*hn;
  hn = t->getHostname ();

  mm_Host*root;
  root = (mm_Host*) graph->getRoot ();
  insist (root);
  insist (!root->mGetNode (0));

  a.fromBytes (t->address);
  insist (!a.isZero ());

  if (t->option == mt_Message::CLOUD_OPTION)
  {
    mt_mapper_c (("it is a cloud node"));
    mt_CloudReplyMessage*crm = (mt_CloudReplyMessage*) t;

    t->swap ();
    crm->swap ();
    
    if (length < (int) sizeof (mt_CloudReplyMessage))
    {
      mt_mapper_w (("bad length"));
      if (options.verbose)
	crm->dump (length);
    
      counters.badLengths++;
      return 0;
    }
  
    if (crm->numHosts < 0)
    {
      mt_mapper_w (("bad contents"));
      counters.badContents++;
      return 0;
    }
      
    mt_Route r;
    mm_Cloud*c = new mm_Cloud (&a, &r, crm->numHosts + 1);

    root->mConnect (0, 0, c);
    c->mConnect (0, 0, root);
    currentGraph->add (c);

    if (!addCloudHosts (currentGraph, c, crm))
      return 0;

    network->freePacket (p);
    done = 1;
    return 1;
  }   

  /*if the address is our own, we are connected to a loopback cable.*/
  
  
  if (!hostTable->resolve (t->gmId, t->hostType, &a))
  {
    mt_mapper_w (("id conflict gmId %d, address %s", t->gmId, a.toString ()));

    mt_Node*other = currentGraph->getGmId (t->gmId);

    if (other)
    {
      mt_mapper_w (("this id has been seen before on host %s", other->getName ()));
    }
    else
    {
      mt_mapper_w (("this may be just a conflict with the hosts file."));
    }
    
    counters.gmIdConflicts++;
  }  
  
  formatHostname (hn, 0);
  
  if (*hn)
    hostTable->setHostname (&a, hn);
  else
    hn = hostTable->getHostname (&a);
  insist (hn && *hn);
  
  hostTable->setOption (&a, t->option);

  if (!(t->option & mt_Message::PACKED_ROUTES_OPTION))
  {
    mt_mapper_c (("this host is expecting obsolete configuration messages"));
  }

  mm_Host*h;
  if (!address.compare (&a))
  {
    mt_mapper_c (("looped back"));   
    h = root;
    //h->setMapVersion (t->mapVersion);
  }
  else
  {
    mt_mapper_c (("new host %s found", hn));
    mt_mapper_c (("its current gm id is %d", t->gmId));

    if (!options.makeHosts && !t->gmId)
    {
      mt_mapper_c (("makeHosts option disabled. host %s cannot be configured.", hn));
    }
    
    h = new mm_Host (hn, t->hostType);
    insistp (h, ("mm_Mapper::firstHost: alloc failed"));
    h->setAddress (&a);
    h->setGmId (t->gmId);
    h->setMapVersion (t->mapVersion);
    graph->add (h);
  }

  root->mConnect (0, 0, h);
  h->mConnect (0, 0, root);
  network->freePacket (p);
  done = 1;
  return 1;
  exception: return 0;
}

static mt_Route*e;

mm_Switch* mm_Mapper::firstSwitch (mm_Graph*graph)
{
  mt_Route to;
  mt_Route r (&to, 0);
  int length;
  char*p;
  mt_Network*network = getNetwork ();
  mt_ProbeMessage m (0, ++phase);
  
  {
    e = new mt_Route (&to, 0);
    delete e;
  }
    
  insist (this);
  insist (graph);
  insist (network);

  mt_mapper_c (("looking for first switch"));
  
  mm_Host*root;
  root = (mm_Host*) graph->getRoot ();
  insist (root && !address.compare (root->getAddress ()));
  insist (!root->mGetNode (0));  

  m.swap ();
  int i;

  for (i = 0; i < options.probeTries; i++)
  {
    if (!send (&r, (char*) &m, sizeof (mt_ProbeMessage)))
    {
      mt_mapper_w (("send failed. aborting"));
      network->abort ();
    }
    
    counters.sends [mt_Message::PROBE]++;
    network->setTimer (this, options.probeMicroseconds);
    if (wait (mt_Network::RECEIVE, mt_Message::PROBE, phase, &p, &length))
      break;
  }

  if (i == options.probeTries)
  {
    mt_mapper_c (("didn't find first switch"));
    root->disconnect (0);
    done = 1;
    return 0;
  }
  network->clearTimer (this);
  network->freePacket (p);
  mm_Switch*s;
  s = new mm_Switch (currentGraph->makeSwitchName(), &to);
  insist (s);

  mt_mapper_c (("found first switch %s", s->getName ()));

  root->mConnect (0, 0, s);
  s->mConnect (0, 0, root);
  return s;
  exception: return 0;
}


mm_Switch* mm_Mapper::firstIdProbe (mm_Graph*graph)
{
  mt_Route to;
  mt_Route r (&to, mt_Route::SWITCH_ID);
  int length;
  char*p;
  mt_Network*network = getNetwork ();
  mt_IdProbeMessage m (0, ++phase);
  
  insist (this);
  insist (graph);
  insist (network);

  mt_mapper_c (("looking for first switch (with an id probe message)"));
  
  mm_Host*root;
  root = (mm_Host*) graph->getRoot ();
  insist (root && !address.compare (root->getAddress ()));
  insist (!root->mGetNode (0));  

  m.swap ();
  int i;

  for (i = 0; i < options.probeTries; i++)
  {
    if (!send (&r, (char*) &m, sizeof (mt_IdProbeMessage)))
    {
      mt_mapper_w (("send failed. aborting"));
      network->abort ();
    }
    
    counters.sends [mt_Message::ID_PROBE]++;
    network->setTimer (this, options.probeMicroseconds);
    if (wait (mt_Network::RECEIVE, mt_Message::ID_PROBE, phase, &p, &length))
      break;
  }

  if (i == options.probeTries)
  {
    mt_mapper_c (("didn't find first switch"));
    root->disconnect (0);
    done = 1;
    return 0;
  }
  network->clearTimer (this);

  if (length < (int) sizeof (mt_IdProbeMessage))
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      ((mt_Message*)p)->dump (length);

    root->disconnect (0);
    done = 1;
    counters.badLengths++;
    network->freePacket (p);
    return 0;
  }
  
  mt_IdProbeMessage*sm;
  sm = (mt_IdProbeMessage*) p;
  sm->swap ();

  mt_mapper_c (("got id probe. id is %d, in port is %d and num ports is %d",
		sm->id, sm->inPort, sm->numPorts));

  if (!sm->id || sm->numPorts <= 0 && sm->numPorts > mm_Switch::MAX_SWITCH)
  {
    mt_mapper_w (("bad contents"));
    root->disconnect (0);
    done = 1;
    counters.badLengths++;
    network->freePacket (p);
    return 0;
  }
  
  mm_Switch*s;
  s = new mm_Switch (currentGraph->makeSwitchName(), &to, sm->id, sm->inPort, sm->numPorts);
  insist (s);

  mt_mapper_c (("found first switch %s", s->getName ()));

  root->mConnect (0, 0, s);
  s->mConnect (0, 0, root);
  network->freePacket (p);
  return s;
  exception: return 0;
}

void mm_Mapper::handleIdProbe (mt_Message*m, int length)
{
  insist (this);
  insist (m && length);
  insist (currentGraph);

  mt_mapper_c (("got an id probe"));
  
  if (length < (int) sizeof (mt_IdProbeMessage))
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      m->dump (length);

    counters.badLengths++;
    return;
  }

  mt_IdProbeMessage*sm;
  sm = (mt_IdProbeMessage*) m;

  sm->swap ();
  int p;
  p = sm->port;

  insist (sm->subtype == mt_Message::ID_PROBE);
  insist (p >= mm_Switch::MIN_PORT && p < mm_Switch::MAX_PORT);

  if (sm->phase != phase)
  {
    mt_mapper_w (("late message"));
    counters.badPhases++;
    increaseTimeouts (sm->subtype, sm->phase);
    return;
  }

  mt_mapper_c (("got id probe. id is %d, in port is %d and num ports is %d",
		sm->id, sm->inPort, sm->numPorts));
  
  insist (!currentSwitch->mGetNode (p));

  mm_Switch*d;
  d = currentGraph->getSwitchFromId (sm->id);
  if (d)
  {
    mt_mapper_c (("found duplicate switch %s", d->getName ()));

    int pp = sm->inPort - d->getInPort ();
    
    d->mConnect (pp, p, currentSwitch);
    currentSwitch->mConnect (p, pp, d);
    return;
  }
  
  mt_Route*r;
  r = new mt_Route (currentSwitch->getRoute (0), p);
  insist (r);
  mm_Switch*s;
  s = new mm_Switch (currentGraph->makeSwitchName (), r, sm->id, sm->inPort, sm->numPorts);
  delete r;
  insist (s);

  mt_mapper_c (("found new switch %s on port %d of switch %s",
		s->getName (), p, currentSwitch->getName ()));

  insist (!currentSwitch->mGetNode (p));
  
  currentSwitch->mConnect (p, 0, s);
  s->mConnect (0, p, currentSwitch);
  currentGraph->add (s);
  exception: return;
}


void mm_Mapper::findSwitchesIdProbe (mm_Switch*s)
{
  insist (this);
  insist (s);

  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  phase++;

  mt_mapper_c (("looking for switches on switch %s (using id probes)", s->getName ()));
  
  currentSwitch = s;
  setHandler (mt_Message::ID_PROBE, mt_fp (mm_Mapper::handleIdProbe));

  int inPort;
  inPort = s->getInPort ();
  int numPorts;
  numPorts = s->getNumPorts ();
  
  insist (inPort >= 0 && inPort <= mm_Switch::MAX_PORT);
  insist (numPorts >= 0 && numPorts <= mm_Switch::MAX_PORT);
  
  for (int j = 0; j < options.probeTries; j++, phase++)
  {
    for (int i = -inPort; i < numPorts - inPort && !network->isAborted (); i++)
    {
      if (s->mGetNode (i))
	continue;

      mt_Route a (s->getRoute (0), i, mt_Route::SWITCH_ID, -i);
      mt_Route b (*s->getRoute (0));
      b.invert ();

      if (a.getLength () + b.getLength () > mt_Route::MAX_ROUTE)
      {
	mt_mapper_w (("route too long. not sending message"));
	continue;
      }
      
      mt_Route r (&a, &b);

      insist (i + inPort >= 0 && i + inPort < numPorts);
      
      mt_IdProbeMessage m (i, phase);
      m.swap ();

      if (!send (&r, (char*) &m, sizeof (mt_IdProbeMessage)))
      {
	mt_mapper_w (("send failed. aborting"));
	network->abort ();
      }
      counters.sends [mt_Message::ID_PROBE]++;

      if (!options.sendManyAtOnce)
	wait (options.probeMicroseconds);
    }
    if (options.sendManyAtOnce)
      wait (options.probeMicroseconds);
  }
  clearHandler (mt_Message::ID_PROBE);
  currentSwitch = 0;
  exception: return;
}

int mm_Mapper::findHosts (mm_Switch*s)
{
  insist (this);
  insist (s);

  mt_Network*network;
  network = getNetwork ();
  insist (network);

  mt_mapper_c (("looking for hosts on switch %s", s->getName ()));
  
  currentSwitch = s;
  duplicate = 0;

  int start, stop;
  
  if (options.probeSwitches)
  { 
    start = - s->getInPort ();
    stop = s->getNumPorts () - s->getInPort ();
  }
  else
  {
    start = mm_Switch::MIN_PORT;
    stop = mm_Switch::MAX_PORT;
  }

  setHandler (mt_Message::REPLY, mt_fp (mm_Mapper::handleReply));
  int somethingSent;
  
  /*first blast to ports that remembered as unconnected*/
  for (int j = 0; j < options.scoutTries; j++, phase++)
  {  
    somethingSent = 0;
    for (int i = start; i < stop && !duplicate && !done && !network->isAborted (); i++)
    {
      if (!s->mGetNode (i))
      {
	mt_Route r (s->getRoute (0), i);

	mt_Node*remembered;
	
	if (options.rememberNodes && (remembered = getRememberedNode (&r)) &&
	    remembered->isHost ())
	{
	  mt_mapper_v (("skipping port %d (remembered)", i));
	  continue;
	}
	
	mt_Route rr (r);
	rr.invert ();
	mt_ScoutMessage m (i, phase, &rr, &address, options.level);
	int size = m.getSize ();
	m.swap ();

	if (!send (&r, (char*) &m, size))
	{
	  mt_mapper_w (("send failed. aborting"));
	  network->abort ();
	}
	somethingSent = 1;
	
	if (!options.sendManyAtOnce)
	{
	  mt_mapper_v (("sent message on port %d", i));
	  wait (options.scoutMicroseconds);
	}
	
	counters.sends [mt_Message::SCOUT]++;
      }
    }
    if (somethingSent && options.sendManyAtOnce)
      wait (options.scoutMicroseconds);
  }

  /*now blast to ports that were remembered as connected*/
  if (options.rememberNodes)
  {
    for (int j = 0; j < options.verifyScoutTries; j++, phase++)
    {  
      somethingSent = 0;
      for (int i = start; i < stop && !duplicate && !done && !network->isAborted (); i++)
      {
	if (!s->mGetNode (i))
	{
	  mt_Route r (s->getRoute (0), i);

	  mt_Node*remembered;
	  
	  if (!(remembered = getRememberedNode (&r)) || !remembered->isHost ())
	  {
	    mt_mapper_v (("skipping port %d (not remembered)", i));
	    continue;
	  }
	  
	  mt_Route rr (r);
	  rr.invert ();
	  mt_ScoutMessage m (i, phase, &rr, &address, options.level);
	  int size = m.getSize ();
	  m.swap ();
	  
	  if (!send (&r, (char*) &m, size))
	  {
	    mt_mapper_w (("send failed. aborting"));
	    network->abort ();
	  }
	  somethingSent = 1;
	  
	  if (!options.sendManyAtOnce)
	  {
	    mt_mapper_v (("sent message on port %d", i));
	    wait (options.scoutMicroseconds);
	  }
	  counters.sends [mt_Message::SCOUT]++;
	}
      }
      if (somethingSent && options.sendManyAtOnce)
	wait (options.scoutMicroseconds);
    }
  }
  if (duplicate)
  {
      /*once I deleted s here*/
  }
  
  clearHandler (mt_Message::REPLY);
  return duplicate;
  exception: return 0;
}


void mm_Mapper::handleLoopProbe (mt_Message*m, int length)
{
  insist (this);
  insist (m && length);

  mt_mapper_c (("got a loopback probe"));

  if (length < (int) sizeof (mt_ProbeMessage))
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      m->dump (length);
    
    counters.badLengths++;
    return;
  }

  mt_ProbeMessage*n;
  n = (mt_ProbeMessage*) m;
  n->swap ();
  int p;
  p = n->port;

  insist (n->subtype == mt_Message::PROBE);
  insist (p >= mm_Switch::MIN_PORT && p < mm_Switch::MAX_PORT);

  mt_mapper_c (("evidentally %s is loopbacked on relative port %d", currentSwitch->getName (), p));
  
  if (n->phase != phase)
    return;

  insist (!currentSwitch->mGetNode (p) || currentSwitch->mGetNode (p) == currentSwitch);

  currentSwitch->mConnect (p, p, currentSwitch);
  exception: return;
}


void mm_Mapper::findLoops (mm_Switch*s)
{
  insist (this);
  insist (s);

  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  phase++;
  currentSwitch = s;

  setHandler (mt_Message::PROBE, mt_fp (mm_Mapper::handleLoopProbe));

  for (int j = 0; j < options.probeTries; j++)
  {
    for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT; i++)
    {
      if (s->mGetNode (i))
	continue;

      mt_Route ra (s->getRoute (0), i, -i);
      mt_Route rb (*s->getRoute (0));
      rb.invert ();
      mt_Route r (&ra, &rb);
      
      mt_ProbeMessage m (i, phase);
      m.swap ();

      if (!send (&r, (char*) &m, sizeof (mt_ProbeMessage)))
      {
	mt_mapper_w (("send failed. aborting"));
	network->abort ();
      }
      counters.sends [mt_Message::PROBE]++;
    }
    network->setTimer (this, options.probeMicroseconds);
    wait ();
  }
  clearHandler (mt_Message::PROBE);
  currentSwitch = 0;
  exception: return;
}

void mm_Mapper::handleSwitchProbe (mt_Message*m, int length)
{
  insist (this);
  insist (m && length);
  insist (currentGraph);

  mt_mapper_c (("got a switch probe"));
  
  if (length < (int) sizeof (mt_ProbeMessage))
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      m->dump (length);

    counters.badLengths++;
    return;
  }

  mt_ProbeMessage*n;
  n = (mt_ProbeMessage*) m;
  n->swap ();
  int p;
  p = n->port;

  insist (n->subtype == mt_Message::PROBE);
  insist (p >= mm_Switch::MIN_PORT && p < mm_Switch::MAX_PORT);

  if (n->phase != phase)
  {
    mt_mapper_w (("late message got %d, expected %d", n->phase, phase));
    counters.badPhases++;
    increaseTimeouts (n->subtype, n->phase);
    return;
  }
  insist (!currentSwitch->mGetNode (p));
  mt_Route*r;
  r = new mt_Route (currentSwitch->getRoute (0), p);
  insist (r);
  mm_Switch*s;
  s = new mm_Switch (currentGraph->makeSwitchName (), r);
  delete r;
  insist (s);

  mt_mapper_c (("found new switch %s on port %d of switch %s",
		s->getName (), p, currentSwitch->getName ()));

  insist (!currentSwitch->mGetNode (p));
  
  currentSwitch->mConnect (p, 0, s);
  s->mConnect (0, p, currentSwitch);
  currentGraph->add (s);
  exception: return;
}
  

void mm_Mapper::findSwitches (mm_Switch*s)
{
  insist (this);
  insist (s);

  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  phase++;

  mt_mapper_c (("looking for switches on switch %s", s->getName ()));
  
  currentSwitch = s;
  setHandler (mt_Message::PROBE, mt_fp (mm_Mapper::handleSwitchProbe));

  int somethingSent;

  /*blast to unremembered ports*/
  for (int j = 0; j < options.probeTries; j++, phase++)
  {
    somethingSent = 0;
    for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT && !network->isAborted (); i++)
    {
      if (s->mGetNode (i))
	continue;
      
      mt_Route a (s->getRoute (0), i, 0, -i);
      mt_Route b (*s->getRoute (0));
      b.invert ();

      if (a.getLength () + b.getLength () > mt_Route::MAX_ROUTE)
      {
	mt_mapper_w (("route too long. not sending message"));
	continue;
      }
      
      mt_Route r (&a, &b);

      insist (currentGraph && currentGraph->getRoot ());
      
      if (options.rememberNodes && getRememberedNode (&r) == currentGraph->getRoot ())
      {
	mt_mapper_v (("skipping port %d (remembered)", i));
	continue;
      }
      
      mt_ProbeMessage m (i, phase);
      m.swap ();

      if (!send (&r, (char*) &m, sizeof (mt_ProbeMessage)))
      {
	mt_mapper_w (("send failed. aborting"));
	network->abort ();
      }
      somethingSent = 1;
      counters.sends [mt_Message::PROBE]++;

      if (!options.sendManyAtOnce)
      {
	mt_mapper_v (("sent message on port %d", i));
	wait (options.probeMicroseconds);
      }
      
    }
    if (somethingSent && options.sendManyAtOnce)
      wait (options.probeMicroseconds);
  }

  if (options.rememberNodes)
  {
    /*blast to remembered ports*/
    for (int j = 0; j < options.verifyProbeTries; j++, phase++)
    {
      somethingSent = 0;
      for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT && !network->isAborted (); i++)
      {
	if (s->mGetNode (i))
	  continue;
	
	mt_Route a (s->getRoute (0), i, 0, -i);
	mt_Route b (*s->getRoute (0));
	b.invert ();

	if (a.getLength () + b.getLength () > mt_Route::MAX_ROUTE)
	{
	  mt_mapper_w (("route too long. not sending message"));
	  continue;
	}
	mt_Route r (&a, &b);
	
	insist (currentGraph && currentGraph->getRoot ());
	
	if (getRememberedNode (&r) != currentGraph->getRoot ())
	{
	  mt_mapper_v (("skipping port %d (not remembered)", i));
	  continue;
	}
	
	mt_ProbeMessage m (i, phase);
	m.swap ();
	
	if (!send (&r, (char*) &m, sizeof (mt_ProbeMessage)))
	{
	  mt_mapper_w (("send failed. aborting"));
	  network->abort ();
	}
	somethingSent = 1;
	counters.sends [mt_Message::PROBE]++;
	
	if (!options.sendManyAtOnce)
	{
	  mt_mapper_v (("sent message on port %d", i));
	  wait (options.probeMicroseconds);
	}
	
      }
      if (somethingSent && options.sendManyAtOnce)
	wait (options.probeMicroseconds);
    } 
  }
  
  clearHandler (mt_Message::PROBE);
  currentSwitch = 0;
  exception: return;
}


void mm_Mapper::handleCompare (mt_Message*m, int length)
{
  insist (this);
  insist (m && length);
  
  insist (currentSwitch);
  insist (options.sendManyCompares || candidateSwitch);
  insist (previousSwitch);

  if (length < (int) sizeof (mt_CompareMessage))
  {
    mt_mapper_w (("bad length"));
    if (options.verbose)
      m->dump (length);
    
    counters.badLengths++;
    return;
  }

  mt_CompareMessage*n;
  n = (mt_CompareMessage*)m;
  n->swap ();
  int p;
  p = n->getPort ();
  int ti;
  ti = n->getIndex ();
  insist (ti >= 0 && ti < currentGraph->getNumSwitches ());
  
  insist (n->subtype == mt_Message::COMPARE);
  insist (p >= mm_Switch::MIN_PORT && p < mm_Switch::MAX_PORT);

  if (n->phase != phase)
  {
    counters.badPhases++;
    increaseTimeouts (n->subtype, n->phase);
    return;
  }

  mm_Switch*cs;
  cs =  (mm_Switch*) currentGraph->getSwitch (ti);
  insist (cs);
  insist (options.sendManyCompares || candidateSwitch == cs);

  insist (!duplicate);
  insist (!currentSwitch->mGetNode (p));

  int pp;
  pp = previousSwitch->getPort (currentSwitch);
  int cp;
  cp = -p;

  insist (pp >= mm_Switch::MIN_PORT && pp < mm_Switch::MAX_PORT);
  insist (cp >= mm_Switch::MIN_PORT && cp < mm_Switch::MAX_PORT);

  mt_mapper_c (("got a compare message back through switch %s", cs->getName ()));
  
  if (cs == previousSwitch)
  {
    mm_Switch*s = (mm_Switch*) cs->mGetNode (cp);
    insist (s && s->isSwitch ());
    insist (!s->getNumNodes ());
    currentGraph->remove (s);
    cs->disconnect (cp);
  }

  insist (!cs->mGetNode (cp));
  cs->mConnect (cp, pp, previousSwitch);
  previousSwitch->mConnect (pp, cp, cs);
  duplicate = 1;

  clearHandler (mt_Message::COMPARE);
  exception: return;
}

  
  
int mm_Mapper::notUnique (mm_Switch*a, mm_Switch*b)
{
  insist (this);
  insist (a && b);

  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  currentSwitch = a;
  candidateSwitch = b;
  duplicate = 0;
  
  /* we are receiving the replies asynchronously. if a reply
     determines that the switch is a duplicate, the global will become
     set */

  setHandler (mt_Message::COMPARE, mt_fp (mm_Mapper::handleCompare));

  mt_mapper_c (("switch comparing %s and %s", a->getName (), b->getName ()));
  
  for (int j = 0; j < options.probeTries && !duplicate; j++)
  {
    if (!options.sendManyCompares)
      phase++;

    for (int i =  mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT &&
	 !duplicate && !network->isAborted (); i++)
    {
      if (a->mGetNode (i))
	continue;

      if (b == previousSwitch)
      {
	if (b->mGetNode (-i) == b)
	  continue;
      }
      else if (b->mGetNode (-i))
	continue;

      mt_Route ra (a->getRoute (0), i);
      mt_Route rb (*b->getRoute (0));
      rb.invert ();
      mt_Route r (&ra, &rb);
      
      mt_CompareMessage m (i, phase, b->getTypeIndex ());
      insist (currentGraph->getSwitch (b->getTypeIndex ()) == b);
      
      m.swap ();
      if (!send (&r, (char*) &m, sizeof (mt_CompareMessage)))
      {
	mt_mapper_w (("send failed. aborting"));
	network->abort ();
      }
    
      counters.sends [mt_Message::COMPARE]++;
      if (!options.sendManyCompares)
	wait (options.probeMicroseconds);   
    }
  }
  
  currentSwitch = 0;
  candidateSwitch = 0;
  clearHandler (mt_Message::PROBE);

  return duplicate;
  exception: return 0;
}

/* compare switch with all other switches, possibly using compare messages */
int mm_Mapper::unique (mm_Switch*s)
{
  insist (this);
  insist (s);

  mt_mapper_c (("determining if switch %s is unique", s->getName ()));

  phase++;
  
  if (!options.compareSwitches || s->getNumHosts ())
  {
    mt_mapper_c (("not comparing switches on %s. returning 1.",
		  s->getName ()));
    return 1;
  }
  
  mm_Switch*t;
  for (t = (mm_Switch*)queue.peek (); t; t = (mm_Switch*) t->getNext ())
  {
    insist (t != s);
  
    if (t->getNumHosts ())
      continue;

    if (t == previousSwitch)
      continue;

    if (notUnique (s, t))
    {
      mt_mapper_c (("discovered that %s is a duplicate. Deleting it.", s->getName ()));
      currentGraph->remove (s);
      return 0;
    } 
  }
  /*
  if (previousSwitch && notUnique (s, previousSwitch))
  {
    mt_mapper_c (("discovered that %s is a duplicate. Deleting it.", s->getName ()));
    currentGraph->remove (s);
    return 0;
  }
  */
  if (options.sendManyCompares)
  {
    duplicate = 0;
    currentSwitch = s;
    
    setHandler (mt_Message::COMPARE, mt_fp (mm_Mapper::handleCompare));
    wait (options.probeMicroseconds);
    currentSwitch = 0;
    clearHandler (mt_Message::COMPARE);
    if (duplicate)
    {
      mt_mapper_c (("discovered that %s is a duplicate. Deleting it.", s->getName ()));
      currentGraph->remove (s);
      return 0;
    }
  }
 
  return 1;
  exception: return 0;
}

  
void mm_Mapper::explore (int depth)
{
  mm_Switch*s = 0;

  insist (this);

  mt_Network*network;
  network = getNetwork ();
  insist (network);
  

  mt_mapper_c (("exploring %s %d", depth ? "to a depth of" :
		"to unknown depths greater than", depth));
  
  while (!done && !network->isAborted () && (s = (mm_Switch*) queue.get ()))
  {
    mt_mapper_c (("thinking about exploring %s", s->getName ()));

    if (depth > 0 && s->getRoute (0)->getLength () >= depth)
    {
      mt_mapper_c (("distance to %s exceeds max depth %d", s->getName (), depth));
      mt_mapper_c (("not exploring %s", s->getName ()));
      continue;
    }
    mt_mapper_c (("exploring %s", s->getName ()));

    s->setExplored (1);
    
    if (options.findLoops)
      findLoops (s);   

    if (options.probeSwitches)
      findSwitchesIdProbe (s);
    else
      findSwitches (s);
    
    previousSwitch = s;

    for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT && previousSwitch && !network->isAborted () ; i++)
    {
      mm_Switch*n = (mm_Switch*)s->mGetNode (i);
      
      if (n == s)
	continue;
      
      insist (n != s);

      if (n && n->isConnected () && n->isSwitch () && n != s && !n->getExplored ())
      {
	if (!queue.isQueued (n) && !findHosts (n) && (options.probeSwitches || unique (n)))
	{
	  mt_mapper_c (("queueing %s", n->getName ()));
	  queue.put (n);
  
	}
      }
    }
  }
  exception: return;
}

/*second switch is removed as a duplicate*/
int mm_Mapper::merge (int alignment, mm_Switch*s1, mm_Switch*s2)
{
  insist (this);
  insist (s1 && s2);
  insist (s1->isConnected () && s2->isConnected ());
  insist (s1->isSwitch ());
  insist (s2->isSwitch ());
  insist (!s1->isMarked () && !s2->isMarked ());
  insist (s1 != s2);

  
  mt_mapper_c (("merging %s and %s", s1->getName (), s2->getName ()));  
  s1->setDuplicate (s2, alignment);

  mt_mapper_c (("adding %s to the merge queue", s1->getName ()));
  insist (!s1->getNextNode () && s1 != mergeList);
  insist (!s2->getNextNode () && s2 != mergeList);  

  if (mergeList)
    mergeList->setNextNode (s1);
  mergeList = s1;
  
  mt_mapper_c (("marking %s and %s as seen", s1->getName (), s2->getName ()));
  s1->setMark (mt_Node::UP);
  s2->setMark (mt_Node::DOWN);

  if (s2 == previousSwitch)
    previousSwitch = 0;
  
  for (int i = mm_Switch::MIN_PORT; i < mm_Switch::MAX_PORT; i++)
  {
    int a = i + alignment;

    if (a < mm_Switch::MIN_PORT || a >= mm_Switch::MAX_PORT)
    {
      insist (!s1->mGetNode (i) || !s1->mGetNode (i)->isConnected ());
      continue;
    }
    
    mm_Node*n1 = s1->mGetNode (i);
    mm_Node*n2 = s2->mGetNode (a);
    
    if (n1 && n1->isConnected () && n2 && n2->isConnected ())
    {
      if (n1 == n2)
      {
	mt_mapper_c (("found a shared pointer to %s", n1->getName ()));
	mt_mapper_c (("severing %s from %s",  n1->getName (), s2->getName ()));
	
	((mm_Switch*)n1)->sever (alignment, s1, s2);
	continue;
      }
      
      insist (n1 != n2);

      insist (n1->mGetNode (s1->mGetOpposite (i)) == s1);
      insist (n2->mGetNode (s2->mGetOpposite (a)) == s2);
      
      int a2 = s2->mGetOpposite (a) - s1->mGetOpposite (i);

      if (n1->isMarked () || n2->isMarked ())
	continue;
      
      if (!merge (a2, (mm_Switch*) n1, (mm_Switch*) n2))
	return 0;

    }
  }
  return 1;
  exception: return 0;
}

int mm_Mapper::merge (mt_Graph*graph, int alignment, mm_Switch*s1, mm_Switch*s2)
{
  insist (this);
  insist (graph);
  insist (s1 && s1->isSwitch ());
  insist (s2 && s2->isSwitch ());
  
  graph->clearMarks ();
  mergeList = 0;
  
  if (!merge (alignment, s1, s2))
    return 0;

  mergeList = s1;
  
  while (mergeList)
  {
    mm_Switch*s = (mm_Switch*) mergeList;
    mm_Switch*d = s->merge (graph, &queue);
    insist (d);

    mt_mapper_c (("correcting links on %s merged with %s", s->getName (), d->getName ()));
    if (queue.isQueued (d))
    {
      mt_mapper_c (("removing %s from queue.", d->getName ()));
      queue.remove (d);
      
      if (!queue.isQueued (s))
      {
	mt_mapper_c (("queueing its duplicate %s.", s->getName ()));
	queue.put (s);
      }
    }
    mt_mapper_c (("removing %s from graph", d->getName ()));
    graph->remove (d);
    mergeList = mergeList->getNextNode ();
    s->setNextNode (0);
  }
  return 1;
  exception: return 0;
}

mm_Graph*mm_Mapper::map (int mapVersion, int depth)
{
  time_t t;
  char*s, *ss;

  insist (this);

  mt_Network*network;
  network = getNetwork ();
  insist (network);
  
  currentGraph = 0;
  numUnconfiguredHosts = network->getGmId (this) == 0 ? 1 : 0;
    
  time (&t);
  s = ctime (&t);
  insist (s);

  ss = strchr (s, '\n');
  insist (ss);

  *ss = 0;
  
  mt_mapper_c (("---------------------------"));
  mt_mapper_w (("starting to map. My clock says %s", s));

  currentGraph = new mm_Graph ();
  insistp (currentGraph, ("mm_Mapper::start: alloc failed"));
  int r;
  r = currentGraph->setSize (mt_Node::HOST, 100);
  insist (r);
  r = currentGraph->setSize (mt_Node::SWITCH, 100);
  insist (r);

  mt_mapper_c (("my host type is %d", hostType));
  
  mm_Host*root;
  root = new mm_Host (hostname, hostType);
  insistp (root, ("mm_Mapper::map: alloc failed"));
  root->setAddress (&address);
  root->setGmId (gmId);
  root->setMapVersion (mapVersion);
  currentGraph->add (root);
  currentGraph->setRoot (root);
  currentGraph->clearMarks ();
  previousSwitch = 0;
  currentSwitch = 0;
  
  while (!done)
  {
    firstHost (currentGraph);

    if (done)
      break;
    
    mm_Switch*s = options.probeSwitches ? firstIdProbe (currentGraph) : firstSwitch (currentGraph);
    if (done)
      break;
    
    insist (s);
    
    findHosts (s);
    if (done)
    {
      root->disconnect (0);
      break;
    }
    
    previousSwitch = 0;

    insist (queue.isEmpty ());
    queue.put (s);

    currentGraph->add (s);
    
    explore (depth);
    done = 1;
  }

  if (network->isAborted() || !active)
  {
    delete currentGraph;
    return 0;
  }
  
  if (currentGraph->getRoot()->getNode (0) == 0)
  {
    mt_mapper_w (("mapper node is disconnected. I don't know what to do."));
    
    if (options.mapOnce || options.configureOnce)
    {
      mt_mapper_w (("I'm just going to exit then."));
      exit (0);
    }
    
    delete currentGraph;
    return 0;
  }
  
  return currentGraph;
  exception:
  if (currentGraph) delete currentGraph;
  return 0;
}

mt_Node*mm_Mapper::getRememberedNode (mt_Route*r)
{
  insist (this);
  insist (r);

  if (oldGraph)
    return ((mt_Node*) oldGraph->getRoot())->getNode (r);
  exception: return 0;
}

int mm_Mapper::measureLatencies (mt_Graph*g, int*hostLatency, int*switchLatency)
{
  insist (this);
  insist (g);
  insist (hostLatency && switchLatency);
  
  *hostLatency = *switchLatency = 0;
  
  for (int i = 0; i < g->getNumHosts (); i++)
  {
    mm_Host*h = (mm_Host*) g->getHost (i);
  
    int n = getHostLatency (h->getRoute (0), &address, options.level, 10, &phase);
  
    if (n > *hostLatency)
      *hostLatency = n;
  
    mt_mapper_c (("round trip latency to host %s is %d", h->getName (), n));
  }

  for (int i = 0; i < g->getNumSwitches (); i++)
  {
    mm_Switch*s = (mm_Switch*) g->getSwitch (i);
  
    int n = getSwitchLatency (s->getRoute (0), 10, &phase);

    if (n > *switchLatency)
      *switchLatency = n;
  
    mt_mapper_c (("round trip latency to switch %s is %d", s->getName (), n));
  }

  mt_Int64 start;
  start = getNetwork()->getCurrentTime ();

  wait (1000);
  mt_mapper_c (("waiting 1000 microseconds took %d lanai ticks", (int) (getNetwork()->getCurrentTime () - start)));

  return 1;
  exception: return 0;
}

int mm_Mapper::start ()
{
  int numUnchangedMaps = 0;
  int numChangedMaps = 0;
  mm_Graph*lastConfiguredGraph = 0;
  

  mm_Node::setDisconnected ();
  mm_Graph::clearVersion ();
  
  mt_mapper_c (("starting mapper"));
  mt_mapper_c (("I was compiled on " __DATE__ " at " __TIME__));
  
  mt_Network*network = 0;
  insist (this);
  network = getNetwork ();
  insist (network);

  mt_mapper_c (("opening network"));
  if (!network->open (this, &address, hostname, &hostType, options.unit, options.numReceiveBuffers))
    return 0;

  insist (!address.isZero ());

  if (!formatHostname (hostname, options.unit))
    return 0;
    
  if (options.uniqueHostFile)
  {
    insist (*hostname);
    char t [mt_File::FILENAME_LENGTH];
    
    sprintf (t, HOST_FILE_FORMAT, hostname);
    mt_File::replaceFilePart (options.hostFile, t);
    
    mt_mapper_c (("changed host file name to %s", options.hostFile));
  }
  
  mt_mapper_c (("getting my gm id"));
  gmId = network->getGmId (this);

  mt_mapper_c (("my node type is %d", hostType));
  makeHostTable (gmId, hostType, &address, hostname, options.makeHosts, options.hostFile);

  if (options.neverSetHostname)
  {
    mt_mapper_c (("never-set-hostname enabled. not setting hostname."));
  }
  else
  {
    mt_mapper_c (("setting hostname to %s", hostname));
    network->setHostname (this, gmId, hostname);
  }
  
  if (options.justSetHostname)
  {
    mt_mapper_c (("just setting hostname."));

    if (options.neverEnd)
    {
      while (!network->isAborted ())
      {
	network->setTimer (this, 1000*1000);
	wait ();
      }
    }
    goto exception;
  }
  
  mt_mapper_c (("setting mapper level"));
  network->setLevel (this, options.level);
  
  setHandler (mt_Message::SCOUT, mt_fp (mm_Mapper::handleScout));

  active = 1;
  done = 0;
  writeCounters (options.countersFile);
  int breakAfterConfigure;
  breakAfterConfigure = 0;
  int forceRemap;
  forceRemap= 0;

  while (!network->isAborted())
  {
    if (!done && !network->isAborted ())
    {
      if (!gmId && !(gmId = network->getGmId (this)) &&
	  hostTable->resolve (gmId, hostType, &address))
	gmId = hostTable->getGmId (&address);
	
      mt_Route change;
      int isHost;
      char hostname [mt_Network::HOSTNAME_LENGTH + 1];
      int verified;
      verified = 0;

      if (!oldGraph || !options.verify && !options.verifyHosts || forceRemap ||
	  !(verified = verify (oldGraph, oldGraph->getRoot (),&phase, oldGraph->getMapVersion(),
		   &address, options.level, &change, &isHost, hostname)) || options.thrash)
      {
	int mapVersion = oldGraph ? oldGraph->getMapVersion() : 0;

	if ((currentGraph = map (mapVersion, options.maxDepth)))
	{
	  queryClouds (currentGraph, network, &phase);
	  currentGraph->finishSwitches ();
	  
	  int hostLatency;
	  int switchLatency;
	  
	  if (options.measureLatency && measureLatencies (currentGraph, &hostLatency, &switchLatency))
	  {
	    mt_mapper_c (("resetting timeouts using latency measurements"));
	    options.setTimeouts (hostLatency * 2, switchLatency * 2, options.configureMicroseconds);
	    mt_mapper_c (("probe timeout is %d microseconds, scout %d.", 
			  options.probeMicroseconds, options.scoutMicroseconds));
	    
	    if (hostLatency && switchLatency)
	      options.measureLatency = 0;
	  }
	}
      }
      
      if (!active)
      {
	mt_mapper_w (("becoming passive"));

	queue.empty ();
	
	if (currentGraph)
	  delete currentGraph;
	currentGraph = 0;

	mt_mapper_c (("marking all hosts as first seen"));
	
	hostTable->setStates (mt_HostTable::FIRST_SEEN);

	if (options.soreLoser)
	{
	  mt_mapper_c (("sore-loser enabled. done mapping."));
	  break;
	}
	goto sleep;
      }
      
      int unchanged;
      unchanged = 0;
      if (verified)
      {
	numUnchangedMaps++;
	unchanged = 1;
      }

      if (currentGraph)
      {
	mt_mapper_c (("trying to decrease timeouts"));
	decreaseTimeouts ();
	
	if (!oldGraph || options.thrash || !equalGraphs (oldGraph, currentGraph))
	{
	  mt_mapper_c (("map changed. %s", oldGraph ? "" : "no previous map remembered"));
	  
	  if (options.numChangedMaps)
	  {
	    if (lastConfiguredGraph)
	    {
	      if (!equalGraphs (lastConfiguredGraph, currentGraph))
	      {
		mt_mapper_c (("map is different from the last configured map"));
		numChangedMaps++;
	      }
	    }
	  }
	    
	  if (oldGraph && options.addMissingHosts)
	  {
	    mt_mapper_c (("adding missing hosts to network graph"));
	    addMissingHosts (currentGraph, oldGraph);
	  }
	  
	  if (oldGraph != lastConfiguredGraph)
	    delete oldGraph;
	  oldGraph = currentGraph;
	  numUnchangedMaps = 0;
	}
	else
	{
	  mt_mapper_w (("discarding map because it didn't change."));
	  if ((!options.expectedNodes || options.expectedNodes == oldGraph->getNumNodes ()) &&
	      (!options.expectedHosts || options.expectedHosts == oldGraph->getNumHosts ()))
	    numUnchangedMaps++;
	  else if (options.expectedNodes)
	  {
	    mt_mapper_c (("expected %d nodes, only found %d", options.expectedNodes, oldGraph->getNumNodes ()));
	  }
	  else if (options.expectedHosts)
	  {
	    mt_mapper_c (("expected %d hosts, only found %d", options.expectedHosts, oldGraph->getNumHosts ()));
	  }
	 
	  if (lastConfiguredGraph != currentGraph)
	    delete currentGraph;
	  unchanged = 1;
	}
	currentGraph = 0;
      }

      if (unchanged)
      {
	if (options.numUnchangedMaps && numUnchangedMaps >= options.numUnchangedMaps)
	{
	  mt_mapper_c (("saw %d unchanged %s", numUnchangedMaps, numUnchangedMaps == 1 ? "map" : "maps"));
	  if (*options.bootHost)
	  {
	    mt_mapper_c (("disabling bootHost option, and reconfiguring one more time"));
	    *options.bootHost = 0;
	    numUnchangedMaps = 0;

	    if (!options.fishGuts)
	    {
	      options.numUnchangedMaps = 1;
	      breakAfterConfigure = 1;
	    }
	    hostTable->setStates (mt_HostTable::NOT_CONFIGURED);
	  }
	  else
	  {
	    mt_mapper_c (("unchanged maps option enabled. done mapping"));
	    break;
	  }
	}
      }
      
      if (!oldGraph)
	continue;
      
      counters.mapVersion = oldGraph->getMapVersion ();
      counters.hostCount = oldGraph->getNumHosts ();
      counters.switchCount = oldGraph->getNumSwitches ();
      counters.activeMapper = active;

      mt_mapper_w (("found %d hosts and %d switches (%d nodes)",
		    oldGraph->getNumHosts (), oldGraph->getNumSwitches (),
		    oldGraph->getNumNodes ()));
      
      int configured = 0;

      if (options.makeHosts)
      {
	mt_mapper_c (("assigning gm ids"));
	options.fillGmIdGaps ? hostTable->assignGmIdsByFillingInGaps (oldGraph) : hostTable->assignGmIds (oldGraph);  
	writeHostTable (options.makeHosts, options.hostFile);
      }
      
      writeMap (oldGraph, options.mapFile);
      writeCounters (options.countersFile);

      if (options.numChangedMaps)
      {
	mt_mapper_c (("-changed-maps option enabled. Map has changed %d time%s.", numChangedMaps, numChangedMaps != 1 ? "s" : ""));
	
	if (lastConfiguredGraph && numChangedMaps < options.numChangedMaps)
	{
	  mt_mapper_c (("not going to configure network."));
	  continue;
	}
      }
      
      if (options.configureOnlyWhenThereAreUnconfiguredHosts)
      {
	mt_mapper_c (("configure-only-when-there-are-unconfigured-hosts enabled."));
	mt_mapper_c (("the number of unconfigured hosts was %d,", numUnconfiguredHosts));
	if (numUnconfiguredHosts == 0)
	{
	  mt_mapper_c (("not going to configure network."));
	  continue;
	}
	if (options.expectedHosts && options.expectedHosts != oldGraph->getNumHosts () ||
	    (options.expectedNodes && options.expectedNodes != oldGraph->getNumNodes ()))
	{
	  mt_mapper_c (("expected host or node count mismatch."));
	  mt_mapper_c (("not going to configure network."));
	  continue;
	}
      }
      
      configured = isConfigured (oldGraph, oldGraph->getMapVersion ());
      
      insist (!configured || !breakAfterConfigure);

      if (!configured || options.thrash)
      {
	int mapVersion = oldGraph->getMapVersion ();
	int maxVersion = oldGraph->getMaxMapVersion () + 1;
	if (maxVersion > mapVersion)
	{
	  mt_mapper_c (("increasing mapVersion from %d to %d", mapVersion, maxVersion));
	  mapVersion = maxVersion;
	  oldGraph->setMapVersion (mapVersion);
	}
	
	configured = configure (oldGraph, mapVersion, 0, &phase) && !options.thrash;
	if (configured)
	{
	  mt_mapper_c (("all nodes have been configured"));
	  
	  if (lastConfiguredGraph)
	  {
	    insist (lastConfiguredGraph != currentGraph);
	    delete lastConfiguredGraph;
	  }
	  lastConfiguredGraph = currentGraph;
	  numChangedMaps = 0;
	}
	else
	{
	  mt_mapper_w (("\nnot all nodes have been configured\n:-(\n"));
	}
	if (!configured)
	{
	  numUnchangedMaps = 0;
	  breakAfterConfigure = 0;
	}
	else if (breakAfterConfigure)
	{
	  mt_mapper_c (("done configuring for the last time"));
	  break;
	}
	oldGraph->getRoot ()->setMapVersion (mapVersion);
      }
      writeCounters (options.countersFile);

      if ((options.mapOnce || options.configureOnce) &&
	  (configured || !options.configureOnce) &&
	  !options.thrash &&
	  (!options.expectedNodes || options.expectedNodes == oldGraph->getNumNodes ()) &&
	  (!options.expectedHosts || options.expectedHosts == oldGraph->getNumHosts ()))
	{
	  mt_mapper_w (("map-once or configure-once enabled. Done mapping."));
	  break;
	}
      forceRemap = !configured;
    }
    
sleep: 
    int milliseconds = active ? options.activeMilliseconds : options.passiveMilliseconds;
    
    if (options.thrash)
      milliseconds = 30;
    
    done = 0;
    active = 1;

    mt_mapper_w (("sleeping for %d milliseconds", milliseconds));

    clearHandler (mt_Message::SCOUT);
  
    int step = 100;
    for (;milliseconds > 0; milliseconds -= step)
    {
      char*p;
      int length;
      
      if (network->isAborted ())
	goto exception;
      
      network->setTimer (this, (milliseconds > step ? step : milliseconds )* 1000);
      
      if (wait (mt_Network::RECEIVE, mt_Message::SCOUT, 0, &p, &length))
      {
	network->clearTimer (this);
	handleScout ((mt_Message*)p, length);
	break;
      }
    }
    setHandler (mt_Message::SCOUT, mt_fp (mm_Mapper::handleScout));
  }

  exception:
  if (network) network->close (this);
  return 1;
}

mt_Node*mm_Mapper::newNode (int nodeType, char*name, char*type)
{
  insist (this);
  switch (nodeType)
  {
    case mt_Node::HOST:
      return new mm_Host (name, type);
    default:
      insist (0);
  }
  exception: return 0;
}
