/*
  EditableLabel.java
  Merlyn
  (c) 1998 Myricom, Inc.
  dmazzoni@myri.com (Dominic Mazzoni)
*/

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;

class EditableLabel
{
  private String text;
  private int height;
  private int cx;
  private int top;
  private int descent;
  private Rectangle bounds;
  private int caretStart;
  private int caretEnd;
  private FontMetrics fm;
  private boolean clicked;
  
  public EditableLabel(Rectangle rect, String text)
    {
      fm = null;
      this.text = new String(text);
      bounds = rect;
      height = rect.height;
      cx = rect.x + rect.width / 2;
      top = rect.y;
      caretStart = 0;
      caretEnd = this.text.length();
      clicked = false;
    }

  public void paint(Graphics g)
    {
      bounds = new Rectangle();
      fm = g.getFontMetrics();
      descent = fm.getDescent();
    
      bounds.width = fm.stringWidth (text) + descent + 2;
      bounds.height = height;
      bounds.x = cx - (bounds.width/2);
      bounds.y = top;
    
      g.setColor (Color.white);
      g.fillRect (bounds.x, bounds.y, bounds.width, bounds.height);
    
      g.setColor (Color.black);
      g.drawString (text,
		    bounds.x + descent,
		    bounds.y + bounds.height - descent);
      g.drawRect (bounds.x, bounds.y, bounds.width, bounds.height);
    
      char chars[] = new char[text.length()];
      text.getChars(0,text.length(),chars,0);
    
      Rectangle caretBox = new Rectangle();
      caretBox.x = bounds.x + descent + fm.charsWidth(chars,0,caretStart);
      if (caretStart == caretEnd)
	caretBox.width = 1;
      else
	caretBox.width = 1+fm.charsWidth(chars,caretStart,caretEnd-caretStart);
      if (caretStart == 0 && caretEnd>0)
      {
	caretBox.x += 1 - descent;
	caretBox.width += descent - 1;
      }
      if (caretEnd == text.length() && caretStart<caretEnd)
	caretBox.width ++;
      caretBox.y = bounds.y + 1;
      caretBox.height = bounds.height - 1;

      g.setColor (Color.white);
      g.setXORMode(Color.black);
      g.fillRect(caretBox.x, caretBox.y, caretBox.width, caretBox.height);
      g.setPaintMode();
    }

  public String getText()
    {
      return text;
    }

  public boolean mouseDrag(int x, int y)
    {
      if (!clicked)
	return false;

      if (bounds==null || fm==null)
	return false;

      if (bounds.contains(x,y))
      {
	int rt = bounds.x + descent;
	int last = rt;
	int c=0;
	int newCaret;
	
	while(rt<x && c<text.length())
	{
	  last = rt;
	  rt += fm.charWidth(text.charAt(c));
	  c++;
	}
	if (rt-x < x-last || c==0)
	  newCaret = c;
	else
	  newCaret = c-1;
	
	if (newCaret<0)
	  newCaret = 0;
	if (newCaret > text.length())
	  newCaret = text.length();
	
	if (newCaret < caretStart)
	  caretStart = newCaret;
	
	if (newCaret > caretEnd)
	  caretEnd = newCaret;
	
	return true;
      }
      return false;
    }

  public boolean mouseClick(int x, int y)
    {
      clicked = false;

      if (bounds==null || fm==null)
	return false;

      if (bounds.contains(x,y))
      {
	clicked = true;

	int rt = bounds.x + descent;
	int last = rt;
	int c=0;
	
	while(rt<x && c<text.length())
	{
	  last = rt;
	  rt += fm.charWidth(text.charAt(c));
	  c++;
	}
	if (rt-x < x-last || c==0)
	  caretStart = caretEnd = c;
	else
	  caretStart = caretEnd = c-1;

	return true;
      }
      else
	return false;
    }

  public void handleKeyPressed(KeyEvent e)
    {
      int code = e.getKeyCode ();

      switch(code)
      {
	case KeyEvent.VK_UP:
	  if (e.isShiftDown())
	  {
	    if (caretStart>0)
	      caretStart = 0;
	    return;
	  }
	  caretStart = caretEnd = 0;
	  return;
	case KeyEvent.VK_DOWN:
	  if (e.isShiftDown())
	  {
	    if (caretEnd<text.length())
	      caretEnd = text.length();
	    return;
	  }
	  caretStart = caretEnd = text.length();
	  return;
	case KeyEvent.VK_LEFT:
	  if (e.isShiftDown())
	  {
	    if (caretStart>0)
	      caretStart--;
	    return;
	  }
	  if (caretStart == caretEnd)
	  {
	    if (caretStart>0)
	    {
	      caretStart--;
	      caretEnd--;
	    }
	  }
	  else
	    caretEnd = caretStart;
	  return;
	case KeyEvent.VK_RIGHT:
	  if (e.isShiftDown())
	  {
	    if (caretEnd<text.length())
	      caretEnd++;
	    return;
	  }
	  if (caretStart == caretEnd)
	  {
	    if (caretEnd<text.length())
	    {
	      caretStart++;
	      caretEnd++;
	    }
	  }
	  else
	    caretStart = caretEnd;
	  return;
      }
    }

  public void handleKeyTyped(KeyEvent e)
    {
      char c = e.getKeyChar ();

      switch(c)
      {
	case KeyEvent.VK_BACK_SPACE:
	case KeyEvent.VK_DELETE:
	  if (caretStart==caretEnd)
	  {
	    if (caretStart != 0)
	      caretStart--;
	    text = text.substring(0,caretStart) +
	      text.substring(caretEnd,text.length());
	  }
	  else
	  {
	    text = text.substring(0,caretStart) +
	      text.substring(caretEnd,text.length());
	  }
	  caretEnd = caretStart;
	  return;

	default:
	  text = text.substring(0,caretStart) + c +
	    text.substring(caretEnd,text.length());
	  caretStart++;
	  caretEnd = caretStart;
      }
    }
} // class EditableLabel



