/*
  sm_Simulator.c
  map tools
  finucane@myri.com (David Finucane)
*/


#include <time.h>
#include <string.h>
#include <stdlib.h>

#include "insist.h"
#include "mt_MapFile.h"
#include "mt_Message.h"
#include "sm_Simulator.h"
#include "sm_Host.h"
#include "sm_Switch.h"
#include "sm_Null.h"
#include "sm_Packet.h"

sm_Simulator::sm_Simulator ()
{
  deadlockedPacket = 0;
  graph = 0;
  routes = 0;
  this->list = new sm_EventList (this);
  insistp (list, ("sm_Simulator::sm_Simulator: alloc failed"));
  latencyTotal = 0;
  latencyNumPackets = 0;
  
  exception: return;
}

sm_Simulator::~sm_Simulator ()
{
  cleanup ();
}


void sm_Simulator::usage ()
{
  printFormat
  (
   "-simulator-args:\n"
   "-map-file <map file>\n"
   "-route-file <route file>\n"
   "-queue-latency <ticks>\n"
   "-queue-size <bytes>\n"
   "-switch-in-size <bytes>\n"
   "-switch-out-size <bytes>\n"
   "-switch-time-for-byte <ticks>\n"
   "-num-packets <count>\n"
   "-out-file <filename>\n"
   "-random-seed <seed>\n"
   "-let-packets-die\n"
   "-host-drop-period <integer>\n"
   "-host-drop-count <integer>\n"
   "-host-drop-type <integer>\n"
   "-host-delay <microseconds>\n"
   "-eschew-routes\n"
   "-check-for-deadlocks\n"
   "-short-output\n"
   );
}

int sm_Simulator::parseArgs (mt_Args*args)
{
  int argc;  
  char**argv = args->getArgs (mt_Args::SIMULATOR, &argc);
  *mapFile = 0;
  *routeFile = 0;
  *outFile = 0;
  queueLatency = 0;
  queueSize = 4;
  maxPackets = -1;
  letPacketsDie = 0;
  randomSeed = time (0) % 10000;
  hostDropPeriod = 0;
  eschewRoutes = 0;
  hostDelay = 0;
  switchInSize = 100;
  switchOutSize = 10;
  switchTimeForByte = 7;
  checkForDeadlocks = 0;
  shortOutput = 0;
  hostDropCount = 0;
  hostDropType = -1;
  
  for (int i = 0; i < argc; i++)
  {
    insist (argv[i]);
    
    if (!strcmp (argv[i], "-map-file"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing map filename");
	return 0;
      }
      strncpy (mapFile, argv[i], mt_File::FILENAME_LENGTH);
    }
    else if (!strcmp (argv[i], "-route-file"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing route filename");
	return 0;
      }
      strncpy (routeFile, argv[i], mt_File::FILENAME_LENGTH);
    }
    else if (!strcmp (argv[i], "-queue-latency"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing queue latency");
	return 0;
      }
      queueLatency =  atoi (argv[i]);
      if (queueLatency <= 0)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad queue-latency", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-random-seed"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing random seed");
	return 0;
      }
      randomSeed =  atoi (argv[i]);
    }
    else if (!strcmp (argv[i], "-queue-size"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing queue size");
	return 0;
      }
      queueSize =  atoi (argv[i]);
      if (queueSize < 1 || queueSize > 64)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad queue size", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-switch-in-size"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing switch in size");
	return 0;
      }
      switchInSize =  atoi (argv[i]);
      if (switchInSize < 2)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad switch in size", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-switch-out-size"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing switch out size");
	return 0;
      }
      switchOutSize =  atoi (argv[i]);
      if (switchOutSize < 2)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad switch outsize", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-switch-time-for-byte"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing queue size");
	return 0;
      }
      switchTimeForByte =  atoi (argv[i]);
      if (switchTimeForByte < 1)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad switch time for byte", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-num-packets"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing num packets");
	return 0;
      }
      maxPackets =  atoi (argv[i]);
      if (maxPackets < 1)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad packet count", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-out-file"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing out filename");
	return 0;
      }
      strncpy (outFile, argv[i], mt_File::FILENAME_LENGTH);
    }
    else if (!strcmp (argv[i], "-let-packets-die"))
      letPacketsDie = 1;
    else if (!strcmp (argv[i], "-eschew-routes"))
      eschewRoutes = 1;
    else if (!strcmp (argv[i], "-check-for-deadlocks"))
      checkForDeadlocks = 1;
    else if (!strcmp (argv[i], "-short-output"))
      shortOutput = 1;
    else if (!strcmp (argv[i], "-host-drop-period"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing host drop period");
	return 0;
      }
      hostDropPeriod =  atoi (argv[i]);
      if (hostDropPeriod < 1)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad drop period", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-host-delay"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing host delay");
	return 0;
      }
      hostDelay =  atoi (argv[i]);
      if (hostDelay < 0)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad delay", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-host-drop-count"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing host drop count");
	return 0;
      }
      hostDropCount =  atoi (argv[i]);
      if (hostDropCount < 0)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad drop count", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-host-drop-type"))
    {
      if (++i == argc)
      {
	printFormat ("sm_Simulator::parseArgs: missing host drop count");
	return 0;
      }
      hostDropType =  atoi (argv[i]);
      if (hostDropType < 0)
      {
	printFormat ("sm_Simulator::parseArgs: %s is a bad drop type", argv[i]);
	return 0;
      }
    }
    else 
    {
      printFormat ("sm_Simulator::parseArgs: bad option \"%s\"", argv[i]);
      return 0;
    }
  }
  if (!*mapFile)
  {
    printFormat ("sm_Simulator::sm_Simulator:no map file specified");
    return 0;
  }
  return 1;
  exception: return 0;
}

int sm_Simulator::initialize (mt_Calculator*calculator,mt_Simulation*simulation)
{
  insist (this);
  insist (!graph && !routes);
  insist (simulation);
  this->simulation = simulation;
  this->calculator = calculator;
  
  graph = new sm_Graph (queueSize, queueLatency, switchInSize, switchOutSize, switchTimeForByte);
  insist (graph);

  mt_Component::srand (randomSeed);
  
  mt_MapFile*mf;
  mf = new mt_MapFile (mapFile, mt_File::_READ);
  if (!mf->read (graph))
  {
    delete mf;
    return 0;
  }
  delete mf;
  
  int numHosts;
  numHosts = graph->getNumHosts ();
  int i;
  for (i = 0; i < numHosts && !graph->getHost (i)->getAddress ()->isZero (); i++);
  if (i != numHosts)
  {
    for (i = 0; i < numHosts; i++)
      graph->getHost (i)->getAddress()->set (i + 1);
  }

  if (!eschewRoutes)
  {
    routes = new mt_RouteTable();
    if (!routes)
      return 0;
    
    if (*routeFile)
    {
      if (!routes->fromFile(routeFile, graph))
	return 0;
    }
    else
    {
      insistp (calculator,
	       ("sm_Simulator::initialize: no route file or calculator given"));
      calculator->initialize (graph, 0);
      int r = routes->fromCalculator (calculator, graph);
      calculator->cleanup ();
      if (!r)
	return 0;
    } 
  }
  
  if (!addNulls (graph))
    return 0;
  
  sm_Packet::clearCounters();
  insist (sm_Packet::getNumSent() == 0);
  insist (sm_Packet::getNumReceived() == 0);

  startTime = time (0);

  if (!startJobs ())
    return 0;
  
  return 1;
  exception: return 0;
}

int sm_Simulator::startJobs ()
{
  insist (this);

  int numNodes;
  numNodes = graph->getNumNodes ();

  waitJob = 0;
  waiting = 0;
  
  for (int i = 0; i < numNodes; i++)
  {
    sm_Node*n = (sm_Node*) graph->getNode (i);
    insist (n);

    if (n->isSwitch ())
      continue;
    
    mt_Job*j = simulation->newJob (n, this, calculator);
    insist (j);
    n->setJob (j);
    n->setSimulator (this);
    
    if (j->willWait ())
    {
      insist (!waitJob);
      waitJob = j;
    }
    else if (!j->start ())
      return 0;
  }
  return 1;
  exception: return 0;
}


int sm_Simulator::step (int numEvents)
{
  insist (numEvents > 0);
  insist (maxPackets == -1 || sm_Packet::getNumReceived () < maxPackets);

  for (int i=0; i < numEvents; i++)
  {
    if (isAborted () || !list->processEvent () ||
	(maxPackets != -1 && sm_Packet::getNumReceived () >= maxPackets))
      return 0;
  }

  return 1;

  exception: return 0;
}

void sm_Simulator::run ()
{
  if (waitJob && !waitJob->start ())
    return;
  
  while (step (1000));
}

void sm_Simulator:: freePacket (char*p)
{
  insist (p);
  delete [] p;
  exception: return;
}

int sm_Simulator::waitNotify (mt_Job*job, int event, char*p, int length)
{
  insist (job && (event != mt_Network::RECEIVE || p && length));
  
  if (waiting && job == waitJob)
  {
    if (length)
    {  
      waitData = new char [length];      
      memcpy (waitData, p, length);
      insistp (waitData, ("sm_Simulator::isWaiting: alloc failed"));
    }
    
    waitEvent = event;
    waitLength = length;
    
    waiting = 0;
    return 1;
  }
  exception: return 0;
}

int sm_Simulator::wait (char**p, int*length)
{
  insist (this);
  insist (waitJob);
  
  waitData = 0;
  waitLength = 0;
  waiting = 1;
  waitEvent = 0;
  
  while (waiting && step(1));

  *length = waitLength;
  *p = waitData;
  
  return waitEvent;
  exception: return 0;
}


int sm_Simulator::getMaxPackets()
{
  return maxPackets;
}

int sm_Simulator::getNumReceived()
{
  return sm_Packet::getNumReceived ();
}

void sm_Simulator::cleanup ()
{
  if (graph) delete graph;
  if (routes) delete routes;

  sm_Null*n;
  while ((n = (sm_Null*) nullList.get()))
    delete n;

  graph = 0;
  routes = 0;
}

mt_Graph*sm_Simulator::getGraph ()
{
  insist (this);
  return graph;
  exception: return 0;
}

int sm_Simulator::getNumRoutes (int from, int to)
{
  insist (this);
  insist (routes);
  insist (from >= 0 && from < graph->getNumHosts ());
  insist (to >= 0 && to < graph->getNumHosts ());

  return routes->getNumRoutes (from, to);

  exception: return 0;
}

int sm_Simulator::getRoute (int from, int to, int index, mt_Route*route)
{

  insist (this);
  insist (routes);
  insist (from >= 0 && from < graph->getNumHosts ());
  insist (to >= 0 && to < graph->getNumHosts ());
  insist (index >= 0 && route);

  return routes->getRoute (from, to, index, route);

  exception: return 0;
}

int sm_Simulator::open (mt_Job*job, mt_Address*address, char*name, int*type, int unit, int)
{
  insist (this);
  insist (job && address && name && type && unit >= 0);
  sm_Node*n;
  n = (sm_Node*)job->getNode ();
  insist (n && (n->isHost () || n->getNodeType () == mt_Node::CLOUD));
  
  *address = *n->getAddress ();
  insist (!address->isZero ());

  strcpy (name, n->getName ());

  if (n->getNodeType () == mt_Node::CLOUD)
    *type = mt_Message::CLOUD_OPTION;
  else
    *type = n->getHostType ();

  insist (!job->getOpened ());
  job->setOpened (1);
  
  return 1;
  exception: return 0;
}

void sm_Simulator::close (mt_Job*job)
{
  insist (this);
  insist (job);

  insist (job->getOpened ());
  job->setOpened (0);
  
  exception: return;
}

int sm_Simulator::sendWithPadding (mt_Job*job, mt_Route*route, char*p, int length, int totalLength)
{
  insist (this);
  insist (job && route && p && job->getOpened ());
  insist (length>=0 && !(length % 4));
  insist (totalLength >= 0 && length <= totalLength);
  insist (totalLength < mt_Network::MTU);
  
  sm_Host*n;
  n = (sm_Host*) job->getNode ();
  insist (n->isHost () || n->getNodeType () == mt_Node::CLOUD);
  return n->send (list, route, p, length, totalLength);
  
  exception: return 0;
}

int sm_Simulator::send (mt_Job*job, mt_Route*route, char*p, int length)
{
  insist (this);
  insist (job && route && p && job->getOpened ());
  insist (length && !(length % 4));
  insist (length<mt_Network::MTU);
  
  sm_Host*n;
  n = (sm_Host*) job->getNode ();
  insist (n->isHost () || n->getNodeType () == mt_Node::CLOUD);
  return n->send (list, route, p, length, length);
  
  exception: return 0;
}

void sm_Simulator::setTimer (mt_Job*job, int microseconds)
{
  insist (this);
  insist (job && microseconds > 0);
  sm_Host*n;
  n = (sm_Host*) job->getNode ();
  insist (n && n->isHost ());
  sm_Time delay;

  delay = (microseconds * sm_EventList::MICROS_PER_TICK) +
  (::rand() % sm_EventList::MICROS_PER_TICK);

  insist (!n->hasEvent (list, sm_EventList::TIMEOUT));
  n->addEvent (list, delay, sm_EventList::TIMEOUT, 0);
  n->addSleepTime (delay);
  
  exception: return;
}

void sm_Simulator::clearTimer (mt_Job*job)
{
  insist (this);
  insist (job);
  sm_Node*n;
  n = (sm_Node*) job->getNode ();
  insist (n);

  insist (n->hasEvent (list, sm_EventList::TIMEOUT));
  
  n->removeEventsOfType (list, sm_EventList::TIMEOUT);
  exception: return;
}

void sm_Simulator::setRoute (mt_Job*, int,  mt_Route*)
{
}

void sm_Simulator::setAddress (mt_Job*, int, mt_Address*)
{
}

void sm_Simulator::setHostname (mt_Job*, int, char*)
{
}

void sm_Simulator::setGmId (mt_Job*, int)
{
}

int sm_Simulator::getGmId (mt_Job*)
{
  return 0;
}

void sm_Simulator::setLevel (mt_Job*, int)
{
}  

int sm_Simulator::getNumReceiveBuffers ()
{
  return queueSize;
}

int sm_Simulator::getReceiveMtu ()
{
  return mt_Network::MTU;
}

mt_Int64 sm_Simulator::getCurrentTime ()
{
  insist (this);
  return list->getCurrentTime ();
  exception: return 0;
}

void sm_Simulator::abortWithDeadlock(sm_Packet *deadlockedPacket)
{
  this->deadlockedPacket = deadlockedPacket;
  abort();
}

void sm_Simulator::simpleDump (FILE *fp)
{
  int td = (int) difftime (time (0), startTime);

  double st = (double) (list->getCurrentTime ()) / sm_EventList::MICROS_PER_TICK / 1000;
  
  fprintf (fp, "map file: %s\n", mapFile);
  fprintf (fp, "completion time: %d %s\n", td, td != 1 ? "seconds" : "second" );
  fprintf (fp, "simulated time: %.4f milliseconds \n", st);
  fprintf (fp, "hosts: %d switches: %d\n", graph->getNumHosts(), graph->getNumSwitches());
  fprintf (fp, "utilization: %5.4f\n", getUtilization());
  fprintf (fp, "average latency: %d\n", getAverageLatency());
  fprintf (fp, "packets received: %d\n\n", getNumReceived());

  fprintf(fp, "\n");

  if (isAborted())
  {
    if (deadlockedPacket)
    {
      fprintf (fp, "deadlock detected.\n");
      deadlockedPacket->dumpChain (fp);
    }
    else
    {
      fprintf (fp, "simulation aborted. dumping packets.\n");
      graph->dumpPackets (fp);
    }
  }
  else
  {
    if (getMaxPackets () >= 0 && getNumReceived() != getMaxPackets())
      fprintf(fp, "deadlock inferred.\n\n");
  }
}

void sm_Simulator::dump ()
{
  insist (this);
  insist (simulation);
  insist (graph);
  
  int numNodes;
  numNodes = graph->getNumNodes ();
  int numHosts;
  numHosts = graph->getNumHosts ();
  
  FILE*fp;
  FILE*myFp;
  myFp = 0;
  if (*outFile)
  { 
    myFp= fp = fopen (outFile, "w");
    insistp (fp, ("sm_Simulator::dump: couldn't open %s", outFile));
  }
  else fp = stdout;
  
  fprintf (fp, "%d/%d\n", sm_Packet::getNumReceived(), maxPackets);

  if (!shortOutput)
  {
    for (int i = 0; i < numNodes; i++)
      ((sm_Node*) graph->getNode (i))->simpleDump (fp, list->getCurrentTime());
    
    for (int i = 0; i < numHosts; i++)
      ((sm_Host*)graph->getHost (i))->getJob ()->dump (fp);
  }
  
  if (myFp) fclose (myFp);
  
  exception: return;
}

void sm_Simulator::addLatency (int latency)
{
  insist (this);
  
  latencyTotal += latency;
  latencyNumPackets++;
  
  exception: return;
}

int sm_Simulator::getAverageLatency ()
{
  insist (this);
  
  return latencyNumPackets ? latencyTotal / latencyNumPackets : 0;
  
  exception: return 0;
}

int sm_Simulator::inspectQueue (char*s)
{
  insist (this);
  insist (list);
  insist (s);

  return list->inspect (s);
  
  exception: return 0;
}

int sm_Simulator::getEventQueueSize ()
{
  insist (this);
  insist (list);
  return list->getQueueSize ();
  
  exception: return 0;  
}

double sm_Simulator::getUtilization()
{
  int numHosts;
  
  insist (this);
  insist (simulation);
  insist (graph);
  insist (list);
  
  if (list->getCurrentTime() <= 0)
    return 0.0;

  numHosts = graph->getNumHosts ();
  insist(numHosts > 0);

  sm_Time totalTime;
  totalTime = list->getCurrentTime ();
  double messageTimes, expectedTimes;
  expectedTimes = messageTimes= 0;
  
  for (int i = 0; i < numHosts; i++)
  {
    sm_Host*h = (sm_Host*) graph->getHost (i);
    insist (h);

    if (totalTime < h->getSleepTime ())
      totalTime = h->getSleepTime ();
    
    insist (totalTime >= h->getSleepTime ());

    if (h->getNeverSend ())
      continue;

    messageTimes += (double) h->getTimeUtilized ();
    expectedTimes += (double) (totalTime - h->getSleepTime ());
  }

  insist (expectedTimes > 0);
  
  return messageTimes / expectedTimes;
  
  exception:
  return -1.0;
}

int sm_Simulator::addNulls (mt_Graph*graph)
{
  insist (this);
  insist (graph);

  int numSwitches;
  numSwitches = graph->getNumSwitches ();

  for (int i = 0; i < numSwitches; i++)
  {
    sm_Switch*s = (sm_Switch*) graph->getSwitch (i);
    insist (s);

    int maxNodes = s->getMaxNodes ();

    for (int j = 0; j < maxNodes; j++)
    {
      if (!s->getNode (j))
      {
	sm_Null*n = new sm_Null ("dc", "dc");
	nullList.put (n);
	insistp (n, ("sm_Simulator::addNulls: alloc failed"));
	s->connect (j, 0, n);
	n->connect (0, j, s);
      } 
    }
  }
  
  return 1;  
  exception: return 0;
}

int sm_Simulator::getLetPacketsDie ()
{
  return letPacketsDie;
}

int sm_Simulator::getHostDropPeriod ()
{
  insist (this);
  return hostDropPeriod;
  exception: return 0;
}
int sm_Simulator::getHostDropCount ()
{
  insist (this);
  return hostDropCount;
  exception: return 0;
}
int sm_Simulator::getHostDropType ()
{
  insist (this);
  return hostDropType;
  exception: return 0;
}
int sm_Simulator::getHostDelay ()
{
  insist (this);
  return hostDelay;
  exception: return 0;
}

        
void sm_Simulator::clearRoutes (mt_Job*)
{
}
