/*
  sm_Packet.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <string.h>
#include <stdlib.h>

#include "insist.h"
#include "sm_Callable.h"
#include "sm_Packet.h"
#include "sm_Port.h"
#include "sm_Node.h"
#include "sm_Host.h"
#include "sm_Simulator.h"
#include "mt_Message.h"
#include "mt_htonl.h"

int sm_Packet::numSent;
int sm_Packet::numReceived;

void sm_Packet::clearCounters()
{
  numSent = 0;
  numReceived = 0;
}

sm_Packet::sm_Packet (mt_Route*route, char*p, int length, int extraLength, sm_Time startTime, int letMeDie)
{
  this->myMessage = 0;

  insist (route && p);
  insist (length + extraLength > 0);
  insist (!((length + extraLength) % 4));

  this->letMeDie = letMeDie;
  dead = 0;
  mark = 0;
  
  route->copyOut (bytes);
  byteLength = route->getLength ();
  byteIndex = 0;
  messageLength = length;
  this->length = byteLength + messageLength + extraLength;
  
  if (messageLength <= MAX_LENGTH)
    message = buffer;
  else
  {
    myMessage = message = (char*) malloc (messageLength);
    insistp (this->message, ("sm_Packet::sm_Packet: alloc failed"));
  }
  insist (message);
  insist (!((unsigned) message % sizeof (int)));

  memcpy (message, p, length);
  
  headPort = tailPort = oldTail = 0;
  headPosition = 0;
  tailPosition = 0;
  headFragment = tailFragment = 0;
  next = prev = 0;
  this->startTime = lastTime = startTime;
  blocked = 1;
  id = numSent++;
  lastTimeForHeadByte = lastTimeForTailByte = 0;
  latency = 0;
  mostTimeForByte = 0;
  exception: return;
}

sm_Packet::~sm_Packet ()
{
  if (myMessage) 
    free (myMessage);
}

void sm_Packet::received(sm_Time when)
{
  latency = when - startTime;
  numReceived++;
}

char*sm_Packet::getMessage ()
{
  insist (this);
  return message;
  exception: return 0;
}

int sm_Packet::getMessageLength ()
{
  insist (this);
  return messageLength;
  exception: return 0;
}

int sm_Packet::getMostTimeForByte ()
{
  return mostTimeForByte;
}

int sm_Packet::getLatency ()
{
  return latency;
}

int sm_Packet::getNumReceived ()
{
  return numReceived;
}
int sm_Packet::getNumSent ()
{
  return numSent;
}

void sm_Packet::setSpeed (int timeForByte)
{
  insist (timeForByte > 0);
    
  lastTimeForHeadByte = lastTimeForTailByte = timeForByte;
  exception: return;
}

int sm_Packet::getId ()
{
  return id;
}

sm_Port*sm_Packet::getNextPort ()
{
  insist (this);
  insist (!dead);
  insist (headPort);


  sm_Port*p;
  p = headPort->getNext ();
  if (p)
    return p;
  sm_Node*n;
  n = (sm_Node*) headPort->getNode ();
  insist (n);

  if (byteLength)
  {
    insist (byteIndex <= byteLength);
    if (byteIndex == byteLength)
      return 0;
    if (bytes [byteIndex] == mt_Route::SWITCH_ID)
    {
      insist (messageLength >= (int) sizeof (mt_IdProbeMessage));
      mt_IdProbeMessage*m = (mt_IdProbeMessage*) message;
      insist (m);

      m->id = mt_htonl (n->getId ());
      m->inPort = mt_htons (headPort->getPort ());
      m->numPorts = mt_htons (n->getMaxNodes ());
      
      return n->getOutput (headPort->getPort ());
    }
    else
      return n->getOutput (headPort->getPort () + bytes [byteIndex]);
  }
  else return n->getOutput (headPort->getPort ());
  exception: return 0;
}

int sm_Packet::isBlocked (sm_Port*p)
{
  insist (p);
    
  sm_Packet*k;
  k = p->getPacket ();
  return p->getNextInLine () || (k && (k->getTailPort () != p || k->getTailPosition() == 0));
  exception: return 0;
}

void sm_Packet::call (sm_EventList*list, int type, long)
{
  sm_Time ct = list->getCurrentTime ();
  
  insist (ct >= lastTime);

  if (type == sm_EventList::UNBLOCK)
  {
    insist (blocked);
    lastTime = ct - 1;
    setSpeed (1);
    blocked = 0;
    for (sm_Packet*k = prev; k && k->blocked; k = k->prev)
    {
      k->removeAllEvents (list);
      perr ("%d scheduling unblock event for message %d", id, k->id);

      k->addEvent (list, 1, sm_EventList::UNBLOCK, 0);
    }
    update (list);
  }
  else
  {
    insist (!blocked);
    update (list);
  }
 
  exception: return;
}  
int sm_Packet::getTailPosition ()
{
  return tailPosition;
}
void sm_Packet::setHeadPort (sm_Port*p)
{
  headPort = p;
}
void sm_Packet::setTailPort (sm_Port*p)
{
  tailPort = p;
}
int sm_Packet::getLength ()
{
  return length;
}
sm_Port*sm_Packet::getHeadPort ()
{
  return headPort;
}
sm_Port*sm_Packet::getTailPort ()
{
  return tailPort;
}
void sm_Packet::setHeadPosition (int p)
{
  headPosition = p;
}
void sm_Packet::setTailPosition (int p)
{
  tailPosition = p;
}
void sm_Packet::setStartTime (sm_Time time)
{
  startTime = time;
}
sm_Time sm_Packet::getStartTime ()
{
  return startTime;
}

void sm_Packet::dump (FILE *fp)
{
  if (mark == PRINT_MARK)
    return;

  mark = PRINT_MARK;
  
  fprintf (fp, "packet %d:\n", id);
  fprintf (fp, "head location: node %s %s port %d position %d\n",
	   headPort->getNode ()->getName (),
	   headPort == headPort->getNode()->getOutput (headPort->getPort ()) ? "output" : "input",
	   headPort->getPort (),
	   headPosition);
  fprintf (fp, "tail location: node %s %s port %d position %d\n",
	   tailPort->getNode ()->getName (),
	   tailPort == tailPort->getNode()->getOutput (tailPort->getPort ()) ? "output" : "input",
	   tailPort->getPort (),
	   tailPosition);
  
  fprintf (fp, "message length: %d\n",messageLength);
  fprintf (fp, "route: ");
  
  for (int i = 0; i < byteLength; i++)
    fprintf (fp, "%d ", bytes [i]);

  fprintf (fp, "\nbody:\n");

  for (int i = 0; i < messageLength; i++)
    fprintf (fp, "%2.2x%s", message [i] & 0xff, i && (i % 8 == 0) ? "\n" : " ");
  
  fprintf (fp,"\n\n");
}

void sm_Packet::dumpChain (FILE *fp)
{
  for (sm_Packet*p = this; p; p = p->next)
    p->dump (fp);
}

int sm_Packet::addBlocked (sm_EventList*list, sm_Port*p)
{
  insist (p);
  insist (list);

  insist (!blocked);
  blocked = 1;
  
  perr ("blocked message %d on port (%x) %s  (behind %d)",
	this->id, p, p->getNode()->getName(), p->getPacket()->id);

  if (this == p->getPacket ())
  {
    list->getSimulator ()->abortWithDeadlock (this);
    return 0;
  }
  
  if (p->tail)
    p->tail->next = this;
  else p->head = this;
  p->tail = this;
  p->tail->next = 0;

  if (next)
    next->prev = 0;
  next = 0;

  if (list->getSimulator()->checkForDeadlocks && isDeadlocked ())
  {
    list->getSimulator()->abortWithDeadlock (this);
    return 0;
  }

  return 1;
  exception: return 0;
}

void sm_Packet::removeBlocked (sm_EventList*list, sm_Port*p)
{
  insist (list && p);
    
  if (!p->head)
    return;
  sm_Packet*k;
  k = p->head;
  p->head = p->head->next;
  if (!p->head)
    p->tail = 0;
  perr ("unblocked message %d on port %x (%s)",
	k->id, p, p->getNode()->getName());
  insist (k->blocked);
  insist (!p->getNextInLine ());

  insist (p->getPacket () != k);
  
  p->setNextInLine (k);
  k->next = 0;
  insist (!k->hasEvent (list, sm_EventList::ANY));
  k->addEvent(list, 0, sm_EventList::UNBLOCK, 0);
  exception: return;
}
int sm_Packet::slamIntoNext (sm_EventList*, int td)
{
  insist (next);
  insist (td);
  insist (next->tailPort == headPort && next->tailPosition > headPosition);
  
  int bt;
  bt = lastTimeForHeadByte;
  double md;
  md = (double) td / (double) bt;
  int m;
  m = (int) md;
  headFragment = (double)md - (double) m;
  
  int s;
  s = next->tailPosition - 1 - headPosition;
  
  if (s < m)
  {
    m = s;
    blocked = next->blocked;
  }
  
  headPosition += m;
  return m * bt;
  exception: return 0;
}

int sm_Packet::slamIntoEnd (sm_EventList*, int td)
{
  insist (!next);
  
  int bt;
  bt = lastTimeForHeadByte;
  double md;
  md = (double) td / (double) bt;
  int m;
  m = (int) md;
  headFragment = (double)md - (double) m;
 
  int s;
  s = headPort->getSize () - 1 - headPosition;
  insist (s >= m);

  headPosition += m;
  return m * bt;  
  exception: return 0;
}

int sm_Packet::cross (sm_EventList *list)
{
  insist (headPosition == headPort->getSize () - 1);
  insist (!dead);
  
  sm_Port*p;
  p = getNextPort ();

  insistp (p || letMeDie, ("sm_Packet::cross: packet died"));

  if (!p)
  {
    dead = 1;
    headPort = 0;
    headPosition = 0;
    return 1;
  }
  if (isBlocked (p) && p->getNextInLine () != this)
  {
    perr ("message %d blocked at port %x", id, p);

    addBlocked (list, p);
    return 0;
  }
  perr ("message %d crossed into port %x", id, p);
  
  next = p->getPacket ();
  insist (next != this);
  
  if (next)
    next->prev = this;
  insist (prev != this);
  
  p->setPacket (this);
  p->setNextInLine (0);
  if (p->getNode () == headPort->getNode ())
  {
    byteIndex++;
    length--;
    insist (!headPort->getNext ());
    headPort->setNext (p);
  }
  headPort = p;
  headPosition = 0;

  list->reportMove ();
  
  if (list->getSimulator()->checkForDeadlocks && isDeadlocked ())
  {
    list->getSimulator()->abortWithDeadlock (this);
    return 0;
  }

  return 1;
  exception: return 0;
}

int sm_Packet::inch (sm_EventList*, int td, int ttb)
{
  insist (this);
  
  double md;
  md = (double) td / (double) ttb + tailFragment;

  int m;
  m = (int) md;
  
  if (m == 0 && md > 0.5)
    m = 1;
  
  tailFragment = md - (double) m;

  if (m == 0)
    return 0;

  int s;
  s = tailPort->getSize () - 1 - tailPosition;
  if (s < m)
  {
    sm_Port*p = tailPort->getNext ();
    insist (p || dead);
    if (!p)
    {
      tailPort = 0;
      tailPosition = 0;
      return 0;
    }
    
    if (p->getNode () == tailPort->getNode ())
      tailPort->setNext (0);
    tailPort = p;
    tailPosition = m - s - 1;
    return s;
  }
  tailPosition += m;
  return m;
  exception: return 0;
}

void sm_Packet::reschedule (sm_EventList*list)
{
  insist (list);
  insist (!blocked);
  insist (!dead);

  if (list->getSimulator()->isAborted ())
    return;
  
  removeAllEvents(list);
  sm_Time ct;
  ct = list->getCurrentTime ();
 
  lastTimeForHeadByte = getTimeForHeadByte ();
  lastTimeForTailByte = getTimeForTailByte ();
  perr ("rescheduling message %d (%d, %d)", id, lastTimeForHeadByte, lastTimeForTailByte);
  
  int hd;
  hd = headPort->getSize () - 1 - headPosition;
  int td;
  td = tailPosition == 0 ? 1 : tailPort->getSize () - tailPosition;
  int thb;
  thb = hd * lastTimeForHeadByte;
  int ttb;
  ttb = td * lastTimeForTailByte;

  perr ("%d resc hd = %d, td = %d, thb = %d, ttb = %d",
	id, hd, td, thb, ttb);
  
  
  if (hd == 0)
  {
    insist (!next);
    blocked = 1;
    addEvent(list, headPort->getAddedLatency (), sm_EventList::UNBLOCK, 0);

    perr ("message %d reschedule unblock for  %d", id, (int) (ct + headPort->getAddedLatency ()));
    return;
  }

  if (tailPosition == 1 && (tailPort->getNode()->isHost() ||
			    tailPort->getNode()->getNodeType () == mt_Node::CLOUD))
    return;
  
  addEvent (list, thb < ttb ? thb : ttb, sm_EventList::MOVE, 0);
  perr ("message %d reschedule for %d", id, (int) (ct + (thb < ttb ? thb : ttb)));
  exception: return;
}



int sm_Packet::rescheduleTheDead (sm_EventList*list)
{
  insist (list);
  insist (!blocked);
  insist (dead);
   
  removeAllEvents(list);
  sm_Time ct;
  ct = list->getCurrentTime ();

  if (!tailPort)
  {
    insist (!prev);
    return 0;
  }
  
  lastTimeForTailByte = getTimeForTailByte ();
  perr ("rescheduling dead message %d (%d)", id, lastTimeForTailByte);

  int td;
  td = tailPosition == 0 ? 1 : tailPort->getSize () - tailPosition;
  int ttb;
  ttb = td * lastTimeForTailByte;

  perr ("%d resc  td = %d,  ttb = %d", id, td, ttb);

  if (tailPosition == 1 && (tailPort->getNode()->isHost() ||
			    tailPort->getNode()->getNodeType () == mt_Node::CLOUD))
  {
    insist(0);
    return 0;
  }
  
  addEvent (list, ttb, sm_EventList::MOVE, 0);
  perr ("message %d reschedule for dead %d", id, (int) (ct +  ttb ));
  exception: return 1;
}

void sm_Packet::update (sm_EventList*list)
{
  insist (list);
  sm_Time ct;
  ct = list->getCurrentTime();
  insist (ct >= lastTime);
  insist (!blocked);

  if (list->getSimulator()->isAborted ())
    return;
  
  int td;
  td = ct - lastTime;
  if (td == 0)
    return;
  lastTime = ct;

  perr ("message %d update at time %d head = %d, tail = %d, headPort = %x, tailPort = %x, (%d, %d)",
	id, (int) ct, headPosition, tailPosition, dead ? 0 : headPort, tailPort,
	dead ? 0 : headPort->getSize(), tailPort->getSize());
  
  if (next && !next->blocked)
  {
    perr ("message %d is pushing message %d", id, next->id);
    next->update (list);
  }
  
  int ntd;      
  int ttb;
  ttb = lastTimeForTailByte;

  if (dead)
    ntd = td;
  else if (next)
    ntd = slamIntoNext (list, td);
  else if (headPosition == headPort->getSize () - 1)
    ntd = cross (list);
  else
    ntd = slamIntoEnd (list, td);
  
  sm_Port*ot;
  ot = tailPort;
  int tm;
  tm = ntd == 0 ? 0 : inch (list, ntd, ttb);
  
  if (ot != tailPort)
  {
    insist (tailPosition == 0);
    ot->addEvent (list, 0, sm_EventList::DEPARTURE, (sm_Time) (length + byteLength - byteIndex));
    int ottb = ot->getTimeForByte ();
    if (ottb > mostTimeForByte)
      mostTimeForByte = ottb;
    
    perr ("message %d detected crossing. prev message is %d", id, !prev ? -1 : prev->id);
    if (ot->getPacket() == this)
      ot->setPacket (0);
    
    if (prev)
    {
      perr ("message %d disengaged from message %d in crossing", id, prev->id);
      insist (prev->next == this);
      prev->next = 0;
      prev = 0;
    }
    oldTail = ot;

    if (dead)
      removeBlocked (list, ot);
  }
  
  if (tm && tailPosition == 1)
  {
    insist (tailPort->getPacket() == this);
    
    if (oldTail->getNode ()->isHost () ||  oldTail->getNode()->getNodeType () == mt_Node::CLOUD)
      ((sm_Callable*)oldTail->getNode ())->addEvent(list, 0, sm_EventList::DEPARTURE, (sm_Time) this);
    
    if (tailPort->getNode ()->isHost () || tailPort->getNode()->getNodeType () == mt_Node::CLOUD)
    {
      tailPort->setPacket (0);
      if (prev)
	prev->next = 0;
      perr ("message %d added arrival event for node %s", id, tailPort->getNode()->getName ());
      
      ((sm_Callable*)tailPort->getNode ())->addEvent(list, 0, sm_EventList::ARRIVAL, (sm_Time) this);
    }  
    removeBlocked (list, tailPort);
  }

  if (dead)
  {
    if (!rescheduleTheDead (list))
      delete this;
    return;
  }
  else if (!blocked)
    reschedule (list);
  else
    removeAllEvents(list);
  
  perr ("%d movement: head = %d, tail = %d, headPort = %x, tailPort = %x, %s %s",
	id, headPosition, tailPosition, headPort ? headPort : 0, tailPort,
	blocked ? "(blocked)" : "(not blocked)", dead ? "dead" : "alive");
  exception: return;
}

int sm_Packet::isGoodForHost ()
{
  insist (this);
  return byteIndex == byteLength;
  exception: return 0;
}

int sm_Packet::canDie ()
{
  insist (this);
  return letMeDie;
  exception: return 0;
}

int sm_Packet::isDeadlocked ()
{
  for (sm_Packet*p = next; p; p = p->next)
  {
    insist (!dead);

    if (!p->blocked)
      break;
    
    if (p->mark == DEADLOCK_MARK)
      return 1;
    else p->mark = DEADLOCK_MARK;
 }
  
  for (sm_Packet*p = next; p; p = p->next)
    p->mark = UNMARKED;

  exception:
  return 0;
}

int sm_Packet::getMark ()
{
  insist (this);
  return mark;
  exception: return 0;
}

void sm_Packet::setMark (int mark)
{
  insist (this);
  this->mark = mark;
  exception: return;
}
