/*
  sm_EventList.c
  map tools
  finucane@myri.com (David Finucane)
  dmazzoni@myri.com (Dominic Mazzoni)
*/

#include "insist.h"
#include "sm_EventList.h"
#include "sm_Callable.h"
#include "sm_Simulator.h"

#define parent(x) (((x)-1)/2)
#define left_child(x) (2*(x)+1)
#define right_child(x) (2*(x)+2)

sm_Event::sm_Event ()
{
  qIndex = -1;
  cPrev = 0;
  cNext = 0;
}

sm_Event::sm_Event (int type, sm_Time time, sm_Callable*callable, long data)
{
  this->type = type;
  this->time = time;
  this->callable = callable;
  this->data = data;
  qIndex = -1;
  cPrev = 0;
  cNext = 0;
}

sm_Event::~sm_Event()
{
}

sm_EventList::sm_EventList (sm_Simulator *simulator)
{
  this->simulator = simulator;
  queueMaxSize = 1024;
  queue = new sm_Event*[queueMaxSize+1];

  movements = 0;
  
  for (int i = 0; i < NUM_EVENTS; i++)
    eventCounters [i] = 0;

  insistp (queue, ("sm_EventList: Failed to allocate array of size %d", queueMaxSize));
   
  queueSize = 0;

  time = 1;
  return;
  
  exception:
  queueMaxSize = 0;
  return;
}

sm_EventList::~sm_EventList()
{
  delete[] queue;
}

sm_Simulator *sm_EventList::getSimulator()
{
  return simulator;
}

int sm_EventList::print ()
{
  insist (this);
  
  for (int i = 0; i < queueSize; i++)
  {
    printf ("%d.", queue[i]->type);
  }
  printf ("\n");
  return 1;
  exception: return 0;
}

int sm_EventList::check()
{
  for (int i = 1; i < queueSize; i++)
  {
    insistf (queue[i]->time >= queue[parent(i)]->time);
  }
  return 1;
  exception: return 0;
}

int sm_EventList::inspect (char*s)
{
  char*t = s;
  insist (this);
  insist (s);

  t += sprintf (t, "(%d) ", movements);
  
  for (int i = 0; i < NUM_EVENTS; i++)
    t += sprintf (t, "%d ", eventCounters [i]);
  
  return 1;
  exception: return 0;
}

int sm_EventList::getQueueSize ()
{
  insist (this);

  return queueSize;
  
  exception: return 0;
}

void sm_EventList::swap(int index1, int index2)
{
  insist(index1>=0 && index1<queueSize && index2>=0 && index2<queueSize);

  sm_Event *tempEvent;

  tempEvent = queue[index1];
  queue[index1] = queue[index2];
  queue[index2] = tempEvent;

  queue[index1]->qIndex = index1;
  queue[index2]->qIndex = index2;

  exception: return;
}

void sm_EventList::trickle_up(int index)
{
  insist(index>=0 && index<queueSize);

  if (index==0)
    return;

  if (queue[index]->time < queue[parent(index)]->time)
  {
    swap(index,parent(index));
    trickle_up(parent(index));
  }

  exception: return;
}

void sm_EventList::trickle_down(int index)
{
  insist(index>=0 && index<queueSize);

  if (left_child(index) >= queueSize)
    return;

  if (right_child(index) >= queueSize ||
      queue[left_child(index)]->time < queue[right_child(index)]->time)
  {
    if (queue[index]->time > queue[left_child(index)]->time)
    {
      swap(index,left_child(index));
      trickle_down(left_child(index));
    }
  }
  else
  {
    if (queue[index]->time > queue[right_child(index)]->time)
    {
      swap(index,right_child(index));
      trickle_down(right_child(index));
    }
  }

  exception: return;
}

void sm_EventList::extractEvent(sm_Event *e)
{
  int index;

  insist(e);
  insist(e->qIndex>=0);
  insistx (check());

  index = e->qIndex;
  if (index < queueSize-1)
    swap(index,queueSize-1);

  queueSize--;
  if (index<queueSize)
  {
    trickle_up(index);    
    trickle_down(index);
  }
  
  e->qIndex = -1;

  eventCounters [e->type]--;
  insist (eventCounters [e->type] >= 0);
  
  insistx (check());
  exception: return;
}

int sm_EventList::processEvent ()
{
  if (queueSize==0)
    return 0;

  sm_Time data;
  sm_Event *e;

  //printFormat ("%d", queueSize);

  perr ("time = %d\n", queue[0]->time);

  insist (queue[0]->time >= time);
  insist (queue[0]->callable);
  insistx (check());

  time = queue[0]->time;

  e = queue[0];
  e->callable->extractEvent(this, e);
  data = e->data;

  insistx (check());

  e->callable->call (this, e->type, data);
  delete e;

  insistx (check());

  return 1;
  exception: return 0;
}

void sm_EventList::growQueue()
{
  sm_Event **newQueue = 0;  
  insist (this);

  newQueue = new sm_Event*[queueMaxSize * 2];
  insistp (newQueue, ("sm_EventList: Failed to allocate array of size %d", queueMaxSize));
  
  queueMaxSize *= 2;

  for (int i = 0; i < queueSize; i++)
    newQueue [i] = queue [i];

  delete[] queue;
  queue = newQueue;

  exception:  return;
}

void sm_EventList::reportMove ()
{
  movements++;
}

void sm_EventList::addEvent (sm_Event *e)
{
  insist(e);
  insist (e->type >= 0 && e->type < NUM_EVENTS);

  insistx (check());

  eventCounters [e->type] ++;
  
  if (queueSize == queueMaxSize)
    growQueue();

  insistx (check());

  e->qIndex = queueSize;
  queue[queueSize] = e;
  queueSize++;

  trickle_up(e->qIndex);

  insistx (check());

  exception: return;
}

sm_Time sm_EventList::getCurrentTime ()
{
  return time;
}









