/*
  mt_RouteTable.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdlib.h>
#include <string.h>
#include "insist.h"

#include "mt_RouteTable.h"
#include "mt_RouteFile.h"


mt_RouteTable::mt_RouteTable ()
{
  byteArrays = 0;
  routes = 0;
  numHosts = 0;
  maxRoutes = 0;
  isFromFile = 0;
}

int mt_RouteTable::fromCalculator (mt_Calculator*calculator, mt_Graph*graph)
{
  insist (calculator && graph);
    
  if (!(numHosts = graph->getNumHosts ()))
    return 0;
  
  maxRoutes = calculator->getMaxRoutes();

  byteArrays = new char * [numHosts * maxRoutes];
  insistp (byteArrays, ("mt_RouteTable::fromCalculator: alloc failed."));

  memset (byteArrays, 0, sizeof (char*) * numHosts * maxRoutes);
  
  routes = new char* [maxRoutes * numHosts * (numHosts + 1)];
  insistp (routes, ("mt_RouteTable::fromCalculator: alloc failed."));

  memset (routes, 0, sizeof (char*) * maxRoutes * numHosts * (numHosts + 1));
  //printFormat ("max route index is %d", maxRoutes * numHosts * (numHosts + 1));

  int numHosts;
  numHosts = graph->getNumHosts ();
  
  for (int k = 0; k < maxRoutes; k++)
  {
    for (int i = 0; i < numHosts; i++)
    {
      mt_Node*hi = graph->getHost (i);
      hi->setRouteIndex (i);
    
      int maxBytes = calculator->getNumBytes (i, k);
      insist (maxBytes >= 0);
    
      byteArrays [i + k * numHosts] = new char [maxBytes];
      insistp (byteArrays [i + k * numHosts], ("mt_RouteTable::fromCalculator: alloc failed"));
  
      int numBytes = 0;
    
      for (int j = 0; j < numHosts; j++)
      {
	int r = calculator->getNumRoutes (i, j);

	if (r <= k)
	  continue;
      
	mt_Route route;
	if (!calculator->getRoute (i, j, k, &route))
	  return 0;

	//printFormat ("adding %d to %d index %d route %s", i, j, k,  route.toString ());
	
	insist (route.getLength () + numBytes <= maxBytes);

	insist (!routes [k * numHosts * (numHosts + 1) + i * (numHosts + 1) + j + 1]);
	
	route.copyOut (byteArrays [i + k * numHosts] + numBytes);
	routes [k * numHosts * (numHosts + 1) + i * (numHosts + 1) + j] = byteArrays [i + k * numHosts] + numBytes;
	//printFormat ("entry %x", routes [k * numHosts * (numHosts + 1) + i * (numHosts + 1) + j]);
	
	numBytes += route.getLength ();
	routes [k * numHosts * (numHosts + 1) + i * (numHosts + 1) + j + 1] = byteArrays [i + k * numHosts] + numBytes;
	//printFormat ("entry %x", routes [k * numHosts * (numHosts + 1) + i * (numHosts + 1) + j + 1]);
      }
    }
  }
  return graph->getNumNodes (mt_Node::CLOUD) || check (graph);
  return 1;

  exception: return 0;
}

int mt_RouteTable::fromFile (char*filename, mt_Graph*graph)
{
  routes = 0;
  numHosts = 0;
  mt_Route route;
  mt_RouteFile rf (filename, mt_File::_READ); //yuck
  char lastFrom [mt_Node::NAME_LENGTH + 1];
  char lastTo [mt_Node::NAME_LENGTH + 1];
  char from [mt_Node::NAME_LENGTH + 1];
  char to [mt_Node::NAME_LENGTH + 1];
  *lastTo = *lastFrom = 0;
  int fromIndex = -1;
  int toIndex = 0;
  int routeIndex = 0;
  isFromFile = 1;
  
  insist (filename && *filename && graph);

  int r;
  r = rf.readHeader (&numHosts, &maxRoutes);
  insistp (r, ("mt_RouteTable::mt_RouteTable: error reading header from %s", filename));
  insistp (numHosts == graph->getNumHosts (),
	   ("mt_RouteTable::mt_RouteTable: route file host count doesn't match graph"));

  insist (numHosts > 0 && maxRoutes > 0);
  
  byteArrays = new char * [numHosts];
  insistp (byteArrays, ("mt_RouteTable::fromFile: alloc failed."));

  memset (byteArrays, 0, numHosts);
  
  routes = new char* [numHosts * numHosts * (maxRoutes + 1)];
  insistp (routes, ("mt_RouteTable::fromFile: alloc failed."));
  memset (routes, 0, sizeof (char) * numHosts * numHosts * (maxRoutes + 1));
  
  mt_Node*fromNode,*toNode;
  fromNode = toNode = 0;

  for (int i = 0; i < numHosts; i++)
  {
    int numRoutes, maxBytes, numBytes;
    numRoutes = maxBytes = numBytes = 0;
    
    int r;
    r = rf.readFromLine (from, &numRoutes, &maxBytes);
    insist (r && numRoutes > 0 && maxBytes > 0);
    insist (strcmp (from, lastFrom));
    strcpy (lastFrom, from);
    
    toIndex = 0;
    routeIndex = 0;
    fromIndex++;

    insistp (fromIndex <= numHosts, ("mt_RouteTable::fromFile: too many routes"));

    fromNode = graph->getNode (from);
    insistp (fromNode, ("mt_RouteTable::mt_RouteTable: node \"%s\" not found in graph", from));
    fromNode->setRouteIndex (fromIndex);
    
    byteArrays [i] = new char [maxBytes];
    insistp (byteArrays [i], ("mt_RouteTable::fromFile: alloc failed"));

    for (int j = 0; j < numRoutes; j++)
    {
      mt_Route route;
      
      r = rf.readToLine (to, &route);
      insist (r);
      
      toNode = graph->getNode (to);
      insistp (toNode, ("mt_RouteTable::mt_RouteTable: node \"%s\" not found in graph", to));

      if (j == 0)
	toIndex = 0;
      else if (strcmp (to, lastTo))      
      {
	toIndex++;
	routeIndex = 0;
      }
      else routeIndex++;
    
      strcpy (lastTo, to);
    
      insistp (toIndex <= numHosts*maxRoutes, ("mt_RouteTable::fromFile: too many routes"));

      mt_Node*t = fromNode->getNode (&route);
    
      insistp (t == toNode, ("mt_RouteTable::mt_RouteTable: bad route from \"%s\" to \"%s\".", from, to));
      insist (route.getLength () + numBytes <= maxBytes);
	
      route.copyOut (byteArrays [i] + numBytes);
      routes [i * (numHosts * (maxRoutes + 1)) + toIndex * (maxRoutes + 1) + routeIndex] =  byteArrays [i] + numBytes;
      numBytes += route.getLength ();
      routes [i * (numHosts * (maxRoutes + 1)) + toIndex * (maxRoutes + 1) + routeIndex + 1] = byteArrays [i] + numBytes;
    }
  }
  return check (graph);
  
  exception: return 0;
}

int mt_RouteTable::check (mt_Graph*graph)
{
  insist (this);
  insist (graph);
  insist (graph->getNumHosts () == numHosts);
  char buffer [mt_Route::MAX_ROUTE*5];
  
  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*ni = graph->getHost (i);
    insist (ni);
    
    int ii = ni->getRouteIndex ();
    insist (ii >= 0 && ii < numHosts);
    
    for (int j = 0; j < numHosts; j++)
    {
      mt_Node*nj = graph->getHost (j);
      insist (nj);

      int ji = nj->getRouteIndex ();
      insist (ji >= 0 && ji < numHosts);
	
      for (int k = 0; k < maxRoutes; k++)
      {
	//printFormat ("checking %s to %s number %d", ni->getName (), nj->getName (), k);
	
	mt_Route route;
	int r = getRoute (ii, ji, k, &route);
	if (!r && k > 0)
	  continue;

	if (numHosts == 2 && graph->getNumNodes () == 2)
	{
	  insistp (!r || !route.getLength (), ("mt_RouteTable::check point to point route is nonzero"));
	}
	else
	{
	  insist (r);
	
	  mt_Node*n = ni->getNode (&route);
	  
	  insistp (n == nj || (n == nj->getNode (0) && n->getNodeType () == mt_Node::CLOUD),
		   ("mt_RouteTable::check: route %s doesn't lead from %s to %s", route.toString (buffer), ni->getName (), nj->getName ()));
	}
      }
    }
  }
  return 1;

  exception:
  return 0;
}

mt_RouteTable::~mt_RouteTable ()
{
  if (byteArrays)
  {
    for (int i = 0; i < numHosts * (isFromFile ? 1 : maxRoutes); i++)
    {
      if (byteArrays [i])
	delete byteArrays [i];
    }
    delete [] byteArrays;
  }
  if (routes) delete routes;
  
}

int mt_RouteTable::getMaxRoutes()
{
  insist (this);

  return maxRoutes;

  exception: return 0;
}

int mt_RouteTable::getNumRoutes (int from, int to)
{
  mt_Route r;
  int numRoutes = 0;

  insist (this);
  insist (from >= 0 && from < numHosts);
  insist (to >= 0 && to < numHosts);
  insist (maxRoutes > 0);

  for (int i = 0; i < maxRoutes; i++)
  {
    if (getRoute (from, to, i, &r))
      numRoutes++;
  }
  
  return numRoutes;
  exception: return 0;
}

int mt_RouteTable::getRoute (int from, int to, int index, mt_Route*route)
{
  insist (this);
  insist (from >= 0 && from < numHosts);
  insist (to >= 0 && to < numHosts);
  insist (index >= 0 && index < maxRoutes);
  insist (route && routes);

  char*a1, *a2;

  if (isFromFile)
  {
    a1 = routes [from * (numHosts * (maxRoutes + 1)) + to * (maxRoutes + 1) + index];
    a2 = routes [from * (numHosts * (maxRoutes + 1)) + to * (maxRoutes + 1) + index + 1];
  }
  else
  {
    a1 = routes [index * numHosts * (numHosts + 1) + from * (numHosts + 1) + to];
    a2 = routes [index * numHosts * (numHosts + 1) + from * (numHosts + 1) + to + 1];
  }
  
  //printFormat ("a1 is %x and a2 is %x\n", a1, a2);
  
  if (!a1 || !a2)
    return 0;
 
  insist (a1 && a2);

  int length;
  length = (int) (a2 - a1);
  insist (length >= 0);

  route->fromBytes (a1, length);
  return 1;
  
  exception: return 0;
}
