/*
  mt_RouteFile.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <string.h>
#include "insist.h"

#include "mt_RouteFile.h"
#include "mt_Calculator.h"

static const char*DELIMITERS = "(), \n\r\t";


mt_RouteFile::mt_RouteFile (const char*filename, int mode) : mt_File (filename, mode)
{
  curRoute = numRoutes = -1;
  reader = 0;
}

mt_RouteFile::mt_RouteFile (FILE*fp) : mt_File (fp)
{
  curRoute = numRoutes = -1;
  reader = 0;
}

mt_RouteFile::~mt_RouteFile ()
{
  if (reader) delete reader;
}

int mt_RouteFile::readFromLine (char*from, mt_Token*tokens, int*numRoutes, int*numBytes, int line)
{
  insist (this);
  insist (from && tokens);
  
  if (!tokens->getNext ())
    return 0;
  
  strncpy (from, tokens->getWord (), mt_Node::NAME_LENGTH);
  
  if (!tokens->getNext () || !tokens->isInteger ())
  {
    printFormat ("mt_RouteFile::readFromLine: missing route count on line %d.", line);
    return 0;
  }
  
  *numRoutes = tokens->getInteger ();

  if (*numRoutes <= 0)
  {
    printFormat ("mt_RouteFile::readFromLine: bad route count on line %d.", line);
    return 0;
  }

  if (!tokens->getNext () || !tokens->isInteger ())
  {
    printFormat ("mt_RouteFile::readFromLine: missing numBytes count on line %d.", line);
    return 0;
  }
  
  *numBytes = tokens->getInteger ();

  if (*numBytes <= 0)
  {
    printFormat ("mt_RouteFile::readFromLine: bad numBytes count on line %d.", line);
    return 0;
  }
  
  return 1;

  exception: return 0;
}

int mt_RouteFile::readToLine (char*to, mt_Route *route, mt_Token*tokens, int line)
{
  insist (this);
  insist (to && route && tokens);
  
  if (!tokens->getNext ())
    return 0;
  
  strncpy (to, tokens->getWord (), mt_Node::NAME_LENGTH);
  
  int length;
  length = 0;
  
  while (tokens->getNext ())
  {
    insistp (tokens->isInteger (), ("mt_RouteFile::readToLine: bad hop on line %d.", line));
      
    insistp (length < mt_Route::MAX_ROUTE,
	     ("mt_RouteFile::readToLine: too many hops on line %d.", line));
    
    route->append (tokens->getInteger ());
    length++;
  }
  if (++curRoute == numRoutes)
    curRoute = numRoutes = 0;
  
  return 1;
  exception: return 0;
}

int mt_RouteFile::readHeader (int*numHosts, int*maxRoutes)
{
  mt_Token *tokens = 0;

  insist (this);
  insist (fp);
  insist (!reader);
  insist (numHosts && maxRoutes);
  
  reader = new mt_FileReader (fp, MAX_LINE, "");
  if (!reader)
    goto exception;

  char*line;
  line = reader->readLine ();
  if (!line) return 0;

  tokens = new mt_Token (line, DELIMITERS);

  if (!tokens->getNext () || !tokens->isInteger ())
    goto exception;
  this->numHosts = *numHosts= tokens->getInteger ();

  insistp  (this->numHosts > 0, ("mt_RouteFile::readHeader: bad host count in header"));
  
  if (!tokens->getNext () || !tokens->isInteger ())
    goto exception;
  
  this->maxRoutes = *maxRoutes = tokens->getInteger ();
  insistp  (this->maxRoutes >= 1, ("mt_RouteFile::readHeader: bad max routes in header"));

  if (tokens) delete tokens;
  return 1;

  exception:
  if (tokens) delete tokens;
  return 0;
}

int mt_RouteFile::readFromLine (char*from, int*numRoutes, int*numBytes)
{
  insist (this);
  insist (reader);
  insist (numHosts > 0 && maxRoutes > 0);
  insist (from && numRoutes && numBytes);
  insist (curRoute == this->numRoutes);
  
  {
    char*line = reader->readLine ();
    if (!line) return 0;
    mt_Token tokens (line, DELIMITERS);
    curRoute = 0;
    int r = readFromLine (from, &tokens, numRoutes, numBytes, reader->getLineNumber ());
    this->numRoutes = *numRoutes;
    return r;
  }
  exception: return 0;
}

int mt_RouteFile::readToLine (char*to, mt_Route *route)
{

  insist (this);
  insist (reader);
  insist (numHosts > 0 && maxRoutes > 0);
  insist (to && route);
  insist (curRoute < numRoutes);
  insist (curRoute >= 0 && numRoutes >= 0);
  
  {  
    char*line = reader->readLine ();
    if (!line) return 0;
    mt_Token tokens (line, DELIMITERS);

    route->empty ();
 
    mt_Token more (line, DELIMITERS);
    return readToLine (to, route, &more, reader->getLineNumber ());
  }
  
  exception: return 0;
}

int mt_RouteFile::writeLimits (FILE*fp, int numHosts, int maxRoutes)
{
  insistf (fp);
  insistf (numHosts > 0);
  insistf (maxRoutes > 0);
  
  fprintf (fp, "%d %d\n", numHosts, maxRoutes);
  return 1;
  exception: return 0;
}

int mt_RouteFile::writeFromLine (FILE*fp, char*name, int numRoutes, int numBytes)
{
  insistf (fp);
  insistf (name && numRoutes > 0 && numBytes > 0);
  
  fprintf (fp, "\"%s\" (%d %d)\n", name, numRoutes, numBytes);
  return 1;
  exception: return 0;
}

int mt_RouteFile::writeToLine (FILE*fp, char*name, mt_Route*route)
{
  insistf (fp);
  insistf (name && route);
  fprintf (fp, "\"%s\" %s\n", name, route->toString ());
  return 1;
  exception: return 0;
}

int mt_RouteFile::write (mt_Calculator*calculator)
{  
  insist (this);
  insist (calculator);
  insist (fp);
  
  int numHosts;
  numHosts = calculator->getNumHosts ();
  int maxRoutes;
  maxRoutes = calculator->getMaxRoutes ();

  if (!writeLimits (fp, numHosts, maxRoutes))
    return 0;

  for (int i = 0; i < numHosts; i++)
  {    
    int numRoutes = calculator->getNumRoutes (i);
    
    if (!writeFromLine (fp, calculator->getHost (i)->getName(), numRoutes, calculator->getNumBytes (i)))
      return 0;
    
    for (int j = 0; j < numHosts; j++)
    {
      numRoutes = calculator->getNumRoutes (i, j);
      
      for (int k = 0; k < numRoutes; k++)
      {
	mt_Route route;
	if (!calculator->getRoute (i, j, k, &route))
	  return 0;

	if (!writeToLine (fp, calculator->getHost (j)->getName(), &route))
	  return 0;
      }
    } 
  }
  return 1;
  exception: return 0;
}
