/*
  mt_Message.c
  map tools
  finucane@myri.com
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "insist.h"
#include "mt_htonl.h"
#include "mt_Message.h"


const char*mt_Message::names [mt_Message::NUM_TYPES] =
{
  "scout",
  "old configure",
  "reply",
  "probe",
  "verify",
  "compare",
  "id probe",
  "cloud query",
  "cloud query reply",
  "configure",
  "configure reply"
};

mt_Message::mt_Message ()
{ 
}

mt_Message::mt_Message (int subtype, int port, int phase)
{
  this->type = GM_TYPE;
  this->subtype = subtype;
  this->port = port;
  this->phase = phase;
}

void mt_Message::swap ()
{
  type = mt_htons (type);
  subtype = mt_htons (subtype);
  port = mt_htonl (port);
  phase = mt_htonl (phase);
}

void mt_Message::dump (int length)
{
#define MAX_DUMP 200
  char buffer [MAX_DUMP * 3];
  char*s = buffer;
  if (length > MAX_DUMP ) length = MAX_DUMP;
  
  for (int i = 0; i < length; i++)
    s += sprintf
      (s, "%2.2x%s", ((unsigned char*)this) [i],
       i  && (i + 1) % 16 == 0 ? "\n" : i && i % 2 == 1  ? " " : "");
  mt_Component::printFormat ("%s", buffer);
}

mt_ScoutMessage:: mt_ScoutMessage (int port, int phase, mt_Route*route,
				   mt_Address*address, int level, int command):
  mt_Message (SCOUT, port, phase)
{  
  this->routeLength = route->getLength ();
  if (route->getLength () < OLD_ROUTE_LENGTH)
    mt_Route::physical (route->getHops (), this->route, route->getLength ());
  mt_Route::physical (route->getHops (), this->extendedRoute, route->getLength ());
  address->copyOut (this->address);
  this->level = level;
  this->command = command;
}

void mt_ScoutMessage::swap ()
{
  mt_Message::swap ();
  routeLength = mt_htonl (routeLength);
  level = mt_htonl (level);
  command = mt_htons (command);
}

void mt_ScoutMessage::physical ()
{
  if (routeLength < OLD_ROUTE_LENGTH)
    mt_Route::physical (route, route, routeLength);
  mt_Route::physical (extendedRoute, extendedRoute, routeLength);
}

int mt_ScoutMessage::getSize ()
{
  int size = sizeof (mt_ScoutMessage) - mt_Route::MAX_ROUTE + (routeLength > OLD_ROUTE_LENGTH ? routeLength : 0);
  if (size % 4)
    size += 4 - (size % 4);
  return size;
}

mt_ProbeMessage:: mt_ProbeMessage (int port, int phase) :
  mt_Message (PROBE, port, phase)
{
}

mt_CompareMessage:: mt_CompareMessage (int port, int phase, int index) :
  mt_Message (COMPARE, (port & 0xffff) | ((index << 16) & 0xffff0000), phase)
{
}

int mt_CompareMessage::getIndex ()
{
  short a = (port >> 16) & 0xffff; 
  return (int) a;
}

int mt_CompareMessage::getPort ()
{
  short a = port & 0xffff;
  return (int) a;
}


mt_IdProbeMessage:: mt_IdProbeMessage (int port, int phase) :
  mt_Message (ID_PROBE, port, phase)
{
  id = 0;
  inPort = 0;
  numPorts = 0;
}

void mt_IdProbeMessage::swap ()
{
  mt_Message::swap ();
  id = mt_htonl (id);
  inPort = mt_htons (inPort);
  numPorts = mt_htons (numPorts);
}


mt_ReplyMessage::mt_ReplyMessage (int port, int phase, mt_Address*address,
				  mt_Address*mapperAddress, short gmId,
				  int mapperRunning, int mapVersion, char*hostname, int level,
				  short hostType, short option) :
  mt_Message (REPLY, port, phase)
{
  
  address->copyOut (this->address);
  mapperAddress->copyOut (this->mapperAddress);
  this->gmId = gmId;
  this->mapVersion = mapVersion;
  strncpy (this->oldHostname, hostname, OLD_HOSTNAME_LENGTH);
  this->level = level;
  this->hostType = hostType;
  this->option = option;
  this->mapperRunning = mapperRunning;

  if (option & LONG_HOSTNAME_OPTION)
  {
    strncpy (this->hostname, hostname, HOSTNAME_LENGTH);
  }
}

char*mt_ReplyMessage::getHostname ()
{
  if (option & mt_Message::LONG_HOSTNAME_OPTION)
  {
    hostname [mt_Message::HOSTNAME_LENGTH - 1] = 0;
    return hostname;
  }
  else
  {
    oldHostname [mt_Message::OLD_HOSTNAME_LENGTH - 1] = 0;
    return oldHostname;
  }
}

void mt_ReplyMessage::swap ()
{
  mt_Message::swap ();
  gmId = mt_htons (gmId);
  mapVersion = mt_htonl (mapVersion);
  level = mt_htonl (level);
  hostType = mt_htons (hostType);
  option = mt_htons (option);
}


mt_CloudReplyMessage::mt_CloudReplyMessage (int port, int phase, mt_Address*address,
					    mt_Address*mapperAddress, short gmId,
					    int mapVersion, char*hostname, int level,
					    short hostType, int numHosts) :
  mt_ReplyMessage (port, phase, address, mapperAddress, gmId, 1, mapVersion,
		   hostname, level, hostType, mt_Message::CLOUD_OPTION)
{
  this->numHosts = numHosts;
}

void mt_CloudReplyMessage::swap ()
{
  mt_ReplyMessage::swap ();
  numHosts = mt_htonl (numHosts);
}

mt_CloudQueryMessage::mt_CloudQueryMessage ()
{
}

mt_CloudQueryMessage::mt_CloudQueryMessage (int port, int phase, mt_Route*route, int first, int numHosts) :
  mt_Message (CLOUD_QUERY, port, phase)
{
  this->routeLength = route->getLength ();
  mt_Route::physical (route->getHops (), this->route, route->getLength ());
  this->first = first;
  this->numHosts = numHosts;
}

int mt_CloudQueryMessage::getSize ()
{
  return sizeof (mt_CloudQueryMessage) - mt_Route::MAX_ROUTE + routeLength;
}

void mt_CloudQueryMessage::swap ()
{
  mt_Message::swap ();
  routeLength = mt_htonl (routeLength);
  first = mt_htonl (first);
  numHosts = mt_htonl (numHosts);
}

mt_CloudQueryReplyMessage::mt_CloudQueryReplyMessage (int port, int phase, mt_Address*address, int first, int numHosts):mt_Message (CLOUD_QUERY_REPLY, port, phase)
{  
  address->copyOut (this->address);
  this->first = first;
  this->numHosts = numHosts;
}

void mt_CloudQueryReplyMessage::swap ()
{
  mt_Message::swap ();
  first = mt_htonl (first);
  numHosts = mt_htonl (numHosts);
}

mt_VerifyMessage::mt_VerifyMessage (int phase): mt_Message (VERIFY, 0, phase)
{
}


void*mt_OldConfigureMessage::operator new (size_t, int maxItems)
{
  return malloc (sizeof (mt_OldConfigureMessage) + sizeof (mt_OldConfigureMessage::Item) * maxItems);
}

void mt_OldConfigureMessage::operator delete (void*p)
{
  free (p);
}

mt_OldConfigureMessage::mt_OldConfigureMessage () : mt_Message (OLD_CONFIGURE, 0, 0)
{
  mt_Message::swap ();
}

void mt_OldConfigureMessage::fromNetwork ()
{
  mt_Message::swap ();

  serial = mt_htonl (serial);
  routeLength = mt_htonl (routeLength);
  gmId = mt_htons (gmId);
  mapVersion = mt_htonl (mapVersion);
  numItems = mt_htonl (numItems);

  for (int i = 0; i < numItems; i++)
  {
    items [i].gmId = mt_htons (items [i].gmId);
    items [i].hostType = mt_htons (items [i].hostType);
    items [i].routeLength = mt_htons (items [i].routeLength);
  }
}

void mt_OldConfigureMessage::set (int item, mt_Address*address, int gmId, int hostType,
				  mt_Route*route, char*hostname)
{
  address->copyOut (items [item].address);
  items [item].gmId = mt_htons (gmId);
  items [item].hostType = mt_htons (hostType);
  items [item].routeLength = mt_htons (route->getLength ());
  route->press (items [item].route);
  strncpy (items [item].hostname, hostname, OLD_HOSTNAME_LENGTH);
}


void mt_OldConfigureMessage::setPhase (int phase)
{
  this->phase = mt_htonl (phase);
}

void mt_OldConfigureMessage::setGmId (int gmId)
{
  this->gmId = mt_htons (gmId);
}

void mt_OldConfigureMessage::setAddress (mt_Address*address)
{
  address->copyOut (this->address);
}

void mt_OldConfigureMessage::setMapperAddress (mt_Address*mapperAddress)
{
  mapperAddress->copyOut (this->mapperAddress);
}

void mt_OldConfigureMessage::setRoute (mt_Route*route)
{
  routeLength = mt_htonl (route->getLength ());
  route->press (this->route);
}
void mt_OldConfigureMessage::setPart (int part)
{
  this->port = mt_htonl (part);
}

void mt_OldConfigureMessage::setNumHosts (int numHosts)
{
  this->numHosts = mt_htons (numHosts);
}

void mt_OldConfigureMessage::setMapVersion (int mapVersion)
{
  this->mapVersion = mt_htonl (mapVersion);
}

void mt_OldConfigureMessage::setNumItems (int numItems)
{
  this->numItems = mt_htonl (numItems);
}

void mt_OldConfigureMessage::setSerial (int serial)
{
  this->serial = mt_htonl (serial);
}
  
int mt_OldConfigureMessage::sizeOf (int numItems)
{
  return sizeof (mt_OldConfigureMessage) + sizeof (Item) * (numItems - 1);
}

  
mt_ConfigureMessage::mt_ConfigureMessage ()
{
  type = GM_TYPE;
  subtype = CONFIGURE;
}

void mt_ConfigureMessage::swap ()
{
  mt_Message::swap ();

  hostSection = mt_htonl (hostSection);
  gmId = mt_htons (gmId);
  numHosts = mt_htons (numHosts);
  mapVersion = mt_htonl (mapVersion);
  numItems = mt_htonl (numItems);
  numBytes = mt_htonl (numBytes);
}

char*mt_ConfigureMessage::read16 (char*p, int*var)
{
  mtu16 v;
  memcpy (&v, p, sizeof (mtu16));
  *var = mt_htons (v);
  return p + sizeof (mtu16);
}
char*mt_ConfigureMessage::read32 (char*p, int*v)
{
  memcpy (v, p, sizeof (mtu32));
  *v = mt_htonl (*v);
  return p + sizeof (mtu32);
}

int mt_ConfigureMessage::setHeader (int phase, int host, int hostSection, mt_Address*address, mt_Address*mapperAddress,
				    int gmId, int numHosts, int mapVersion, int numItems, mt_Route*route)
{
  this->type = GM_TYPE;
  this->subtype = CONFIGURE;

  this->phase = phase;
  this->port = host;
  this->hostSection = hostSection;
  
  address->copyOut (this->address);
  mapperAddress->copyOut (this->mapperAddress);
  this->gmId = gmId;
  this->numHosts = numHosts;
  this->mapVersion = mapVersion;
  this->numItems = numItems;
  
  char*p = byteStream;
  
  insistf (route->getLength () < 256);
  *p++ = (mtu8) route->getLength ();
  mt_Route::physical (route->getHops (), p, route->getLength ());

  numBytes = route->getLength () + 1;
  return 1;
  exception: return 0;
}

char*mt_ConfigureMessage::getHost (char*p, mt_Address*address, int*gmId, int*hostType, char*hostname, int*control, int*numRoutes)
{
  int len;
  
  insistf (this);
  insistf (p);
  insistf (address && gmId && hostType && hostname && control && numRoutes);
  insistf (byteStream > (char*)this && byteStream < (char*)this + mt_Network::MTU);
  insistf (p >= byteStream && p < byteStream + mt_Network::MTU);

  address->fromBytes (p);
  p += mt_Address::NUM_BYTES;
  
  p = read16 (p, gmId);
  p = read16 (p, hostType);
  p = read16 (p, control);
  
  len = *p++;

  memcpy (hostname, p, len);
  hostname [len] = 0;
  p += len;

  *numRoutes = *p++;
  return p;
  
  exception: return 0;
}

int mt_ConfigureMessage::append (mt_Address*address, int gmId, int hostType, int control, char*hostname, int numRoutes)
{
  char*o,*p;
  mt_Address ta;
  
  insistf (numBytes > 0 && numBytes <= MAX_BYTE_STREAM);
  insistf (hostname);
  insistf (address);
  
  o = p = byteStream + numBytes;

  insistf (byteStream > (char*)this && byteStream < (char*)this + mt_Network::MTU);
  insistf (p > byteStream && p < byteStream + mt_Network::MTU);
  
  address->copyOut (p);
  p += mt_Address::NUM_BYTES;

  p += append (p, gmId, sizeof (mtu16));
  p += append (p, hostType, sizeof (mtu16));
  p += append (p, control, sizeof (mtu16));
  
  int len;
  len = (int) strlen (hostname);
  insistf (len >= 0 && len < 256);

  *p++ = len;
  insistf (*(p-1) == len);
  strncpy (p, hostname, len);
  insistf (!strncmp (hostname, p, len));
  
  p += len;

  insistf (numRoutes < 256 && numRoutes);
  *p++ = (mtu8) numRoutes;
  insistf (*(p-1) == numRoutes);

  numBytes += p - o;
  insistf (numBytes <= MAX_BYTE_STREAM);

  int tGmId, tHostType, tControl, tNumRoutes;
  char tHostname [mt_Network::HOSTNAME_LENGTH];
  
  getHost (o, &ta, &tGmId, &tHostType, tHostname, &tControl, &tNumRoutes);
  insistf (tGmId == gmId);

  insistf ((tHostType & 0xffff) == (hostType & 0xffff));
  insistf (!strcmp (tHostname, hostname));
  insistf (tNumRoutes == numRoutes);
  insistf (tControl == control);

  return 1;
  exception: return 0;
}

int mt_ConfigureMessage::append (mt_Route*r)
{
  char*o, *p;
  int len;
  
  o = p = byteStream + numBytes;

  insistf (r);
  insistf (numBytes > 0 && numBytes <= MAX_BYTE_STREAM);
  insistf (byteStream > (char*)this && byteStream < (char*)this + mt_Network::MTU);
  insistf (p > byteStream && p < byteStream + mt_Network::MTU);
  
  len = r->getLength ();
  insistf (len >= 0 && len < 256);
  *p++ = (mtu8) len;
  mt_Route::physical (r->getHops (), p, len);
  p += len;
  
  numBytes += p - o;
  insistf (numBytes <= MAX_BYTE_STREAM);
  return 1;
  exception: return 0;
}

int mt_ConfigureMessage::getSize ()
{
  insistf (this);
  
  insistf ((int) MAX_BYTE_STREAM < (int) mt_Network::MTU);
  insistf (byteStream > (char*)this && byteStream < (char*)this + mt_Network::MTU);
  insistf (numBytes > 0 && numBytes < MAX_BYTE_STREAM);  

  int size;
  size = sizeof (mt_ConfigureMessage) - MAX_BYTE_STREAM + numBytes;
  if (size % 4)
    size += 4 - (size % 4);
  return size;

  exception: return 0;
}

mt_ConfigureReplyMessage::mt_ConfigureReplyMessage (int host, int hostSection, int phase, mt_Address*address) :
  mt_Message (CONFIGURE_REPLY, host, phase)
{
  address->copyOut (this->address);
  this->hostSection = hostSection;
}

void mt_ConfigureReplyMessage::swap ()
{  
  mt_Message::swap ();
  hostSection = mt_htonl (hostSection);
}


char*mt_ConfigureMessage::getRoute (char*p, mt_Route*r)
{
  char bytes [mt_Route::MAX_ROUTE];
  int length;

  insistf (this);
  insistf (r);
  insistf (byteStream > (char*)this && byteStream < (char*)this + mt_Network::MTU);
  insistf (p >= byteStream && p < byteStream + mt_Network::MTU);

  length = *p;
  insistf (length >= 0 && length <= mt_Route::MAX_ROUTE);
  
  mt_Route::logical (++p, bytes, length);
  return p + r->fromBytes (bytes, length);
  
  exception: return 0;
}

int mt_ConfigureMessage::append (char*p, int var, int size)
{
  insistf (p);
  
  if (size == sizeof (mtu16))
  {
    mtu16 v = var;
    v = mt_htons (v);
    memcpy (p, &v, sizeof (mtu16));
    return sizeof (mtu16);
  }
  if (size == sizeof (mtu32))
  {
    mtu32 v = var;
    v = mt_htonl (v);
    memcpy (p, &v, sizeof (mtu32));
    return sizeof (mtu32);
  }
  exception: return 0;
}
