/*
  mt_MapFile.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <string.h>
#include <stdio.h>

#include "insist.h"
#include "mt_MapFile.h"

static const char*DELIMITERS = "(), \n\r\t";
static const char*COMMENT_CHARS = "#;";

mt_MapFile::mt_MapFile (const char*filename, int mode) : mt_File (filename, mode)
{
}

mt_MapFile::mt_MapFile (FILE*fp) : mt_File (fp)
{
}


int mt_MapFile::read (mt_Graph*graph)
{
  mt_Tokenizer tokens (fp, DELIMITERS, MAX_LINE, COMMENT_CHARS);
  return parseNodes (&tokens, graph);
}

int mt_MapFile::parseNodeType (char*s)
{
  insist (this && s);
  
  if (!strcmp (s, "h"))
    return mt_Node::HOST;
  else if (!strcmp (s, "s"))
    return mt_Node::SWITCH;
  else if (!strcmp (s, "cloud"))
    return mt_Node::CLOUD;
  exception: return mt_Node::NO_TYPE;
}

int mt_MapFile::write (mt_Graph*graph)
{
  char*buffer = 0;
  
  insist (this);
  insist (graph);

  int numNodes;
  numNodes = graph->getNumNodes ();

  if (!numNodes)
    return 0;
  
  buffer = new char [graph->getMaxNodeStringLength ()];
  insistp (buffer, ("mt_MapFile::write: alloc failed"));
  
  for (int i = 0; i < numNodes; i++)
    fprintf (fp, "%s\n", graph->getNode (i)->toString (buffer));
    
  delete [] buffer;
  
  return 1;
  
  exception:
  if (buffer) delete [] buffer;
  return 0;
}


int mt_MapFile::parseNodes (mt_Tokenizer*tokens, mt_Graph*graph)
{
  int numNodes;

  insist (tokens && graph);

  numNodes = 0;
  
  if (tokens->getNext () && tokens->isInteger () &&
      tokens->getNext () && tokens->isInteger ())
  {
    /*get ready to read the map file, skipping Dominic's node counts if any*/
    tokens->getNext ();
  }  

  while (tokens->getWord ())
  {
    int nodeType = mt_Node::NO_TYPE;
    char type [MAX_LINE + 1];
    
    if (!tokens->isWord ())
      break;

    if ((nodeType = parseNodeType (tokens->getWord ())) == mt_Node::NO_TYPE)
      break;
    
    if (!tokens->getNext ())
      break;

    strncpy (type, tokens->getWord (), MAX_LINE);

    if (!tokens->getNext ())
      break;

    if (!tokens->isWord ())
      break;

    char*name = tokens->getWord ();
    insist (name && *name);
    
    if (strlen (name) > mt_Node::NAME_LENGTH -1)
    {
      printFormat ("name %s is too long", name);
      break;
    }
    
    mt_Node *node = graph->getNode (name);

    if (!node)
    {
      node = graph->newNode (nodeType, name, type);
      insist (node);
      graph->add (node);
      numNodes++;
    }

    if (!tokens->getNext ())
      break;
    if (!tokens->isInteger ())
      break;
    int linkCount = tokens->getInteger ();

    if (nodeType == mt_Node::HOST && linkCount > 1)
    {
      printFormat ("host %s has %d links.", node->getName (), linkCount);
      break;
    }
    for (; linkCount > 0; linkCount--)
    {
      if (!tokens->getNext ())
	break;

      if (!tokens->isInteger ())
	break;
      int np = tokens->getInteger ();

      if (!tokens->getNext ())
	break;
      
      if ((nodeType = parseNodeType (tokens->getWord ())) == mt_Node::NO_TYPE)
	break;
      
      if (!tokens->getNext ())
	break;

      strncpy (type, tokens->getWord (), MAX_LINE);
      
      if (!tokens->getNext ())
	break;
      
      if (!tokens->isWord ())
	break;

      name = tokens->getWord ();
      insist (name && *name);
      
      if (strlen (name) > mt_Node::NAME_LENGTH -1)
      {
	printFormat ("name %s is too long", name);
	break;
      }
    
      mt_Node*r = graph->getNode (name);
      
      if (!r)
      {
	r = graph->newNode (nodeType, name, type);
	insist (r);
	
	graph->add (r);
	
	numNodes++;
      }
	
      if (!tokens->getNext ())
	break;
      if (!tokens->isInteger ())
	break;
      int rp = tokens->getInteger ();

      
      /*detect danny mistakes*/
      if (node->isHost () && np != 0)
      {
	printFormat ("link %d on host %s is impossible", np, node->getName ());
	break;
      }
      if (r->isHost () && rp != 0)
      {
	printFormat ("link %d on host %s is impossible", rp, r->getName ());
	break;
      }
	
      /*detect inktomi mistakes*/

      mt_Node*nt = node->getNode (np);	
      mt_Node*rt = r->getNode (rp);

      
      if ((nt && nt != r) || (rt && rt != node))
      {
	printFormat ("link %d on node %s is inconsistent",
		     np, node->getName());

	nt = node->getNode (np);
	rt = r->getNode (rp);
	
	break;
      }
      
      node->connect (np, rp, r);
      r->connect (rp, np, node);
      insist (node->getNode (np) == r);
      insist (r->getNode (rp) == node);
      insist (node->getOpposite (np) == rp);
      insist (r->getOpposite (rp) == np);
    }
    if (linkCount > 0)
      break;

    for (int n = 0; ; n++)
    {
      if (!tokens->getNext ())
	break;
      
      if (parseNodeType (tokens->getWord ()) != mt_Node::NO_TYPE)
	break;
      
      char option [MAX_LINE + 1];
      strncpy (option, tokens->getWord (), MAX_LINE);

      if (!tokens->getNext ())
	break;

      if (!node->setOption (option, tokens->getWord ()))
	break;
    }  
  }
  
  if (tokens->getWord ())
  {
    printFormat ("line: %d. Syntax error near the text \"%s\".",
		 tokens->getLineNumber (),
		 tokens->getWord ());
    return 0;
  }
  return numNodes;
  exception: return 0;

}














