/*
  mt_HostTable.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdlib.h>
#include <string.h>

#include "insist.h"
#include "mt_HostTable.h"
#include "mt_FileReader.h"
#include "mt_FileWriter.h"
#include "mt_Network.h"

int mt_HostTable::Item::toString (char*s) 
{
  char buffer [HOSTNAME_LENGTH * 2];
  
  return sprintf (s,"%d %d %s \"%s\"", gmId, hostType, address.toString (buffer), hostname);
} 

void mt_HostTable::grow () 
{
  insist (this);
  insist (items && numItems <= maxItems && maxItems > 0);
	
  maxItems*=2;
  items = (Item*)realloc (items, sizeof (Item)* maxItems);
  insist (items);
  lastItem = 0;
  exception: return;
}

mt_HostTable::Item*mt_HostTable::slowFind (mt_Address*address)
{
  insist (this);
  insist (address);
  insist (items && numItems <= maxItems && maxItems > 0);
  
  for (int i = 0; i < numItems; i++)
    if (!items[i].address.compare (address))
      return &items[i];
  exception: return 0;
}

mt_HostTable::Item*mt_HostTable::slowFind (char*hostname)
{
  insist (this);
  insist (hostname);
  insist (items && numItems <= maxItems && maxItems > 0);
  
  for (int i = 0; i < numItems; i++)
    if (!strcmp (items[i].hostname, hostname))
      return &items[i];
  exception: return 0;
}

static int gmIdCompare (const void *k, const void *e)
{
  mt_HostTable::Item*a = (mt_HostTable::Item*)k;
  mt_HostTable::Item*b= (mt_HostTable::Item*)e;;
 
  if (a->gmId == b->gmId)
    return 0;
  if (a->gmId > b->gmId)
    return 1;
  return -1;
}

static int addressCompare (const void *k, const void *e)
{
  mt_HostTable::Item*a = (mt_HostTable::Item*)k;
  mt_HostTable::Item*b= (mt_HostTable::Item*)e;;
 
  return (a->address.compare (&b->address));
}

static int nameCompare (const void *k, const void *e)
{
  mt_HostTable::Item*a = (mt_HostTable::Item*)k;
  mt_HostTable::Item*b= (mt_HostTable::Item*)e;;
 
  return (strcmp (a->hostname, b->hostname));
}

mt_HostTable::Item*mt_HostTable::find (mt_Address*address) 
{
  insist (this);
  insist (items && numItems <= maxItems && maxItems > 0);
  insist (address);
  
  mt_HostTable::Item*m;
  if (lastItem && !lastItem->address.compare (address))
    return lastItem;
    
  if (sortType != BY_ADDRESS)
    return slowFind (address);

  dummy.address = *address;
  
  m = (mt_HostTable::Item*) bsearch (&dummy, items, numItems, sizeof (mt_HostTable::Item), addressCompare);
  return lastItem = m;
  exception: return 0;
}

mt_HostTable::Item*mt_HostTable::find (char*hostname) 
{
  insist (this);
  insist (items && numItems <= maxItems && maxItems > 0);
  insist (hostname && *hostname);
  
  mt_HostTable::Item*m;
  if (lastItem && !strcmp (lastItem->hostname, hostname))
    return lastItem;
    
  if (sortType != BY_ADDRESS)
    return slowFind (hostname);

  strncpy (dummy.hostname, hostname, mt_Network::HOSTNAME_LENGTH);
  
  m = (mt_HostTable::Item*) bsearch (&dummy, items, numItems, sizeof (mt_HostTable::Item), nameCompare);
  return lastItem = m;
  exception: return 0;
}

mt_Address*mt_HostTable::getAddress (int gmId)
{
  insist (this);
  for (int i = 0; i < numItems; i++)
    if (gmId == items[i].gmId)
      return &items[i].address;
  exception: return 0;
}

int mt_HostTable::add (int gmId, int hostType, mt_Address*address, char*hostname) 
{
  insist (this);
  insist (address && !address->isZero());
  insist (items && numItems <= maxItems && maxItems > 0);
  insist (this->getGmId (address) <= 0); /*allow duplicate 0 gmIdes*/
  insist (hostname && *hostname);
  
  if (numItems == maxItems)
    grow();
  
  insist (maxItems > numItems);
  items [numItems].address = *address;
  items [numItems].gmId = gmId;
  items [numItems].hostType = hostType;
  items [numItems].state = FIRST_SEEN;
  strncpy (items [numItems].hostname, hostname, mt_Network::HOSTNAME_LENGTH);
  lastItem = &items [numItems];
    
  numItems++;
  insist (!gmId || gmId != maxGmId);
  if (gmId > maxGmId)
    maxGmId = gmId;
  sortType = UNSORTED;
  
  return gmId;
  exception: return 0;
}

int mt_HostTable::replace (int gmId, mt_Address*address)
{
  insist (this);
  insist (address);
  
  mt_HostTable::Item*m;
  m = find (address);
  insist (m && !m->gmId);
  return m->gmId = gmId;
  exception: return 0;
}

int mt_HostTable::add (int gmId, int hostType, mt_Address*address)
{
  insist (this);
  return add (gmId, hostType, address, address->toString ());
  exception: return 0;
}

int mt_HostTable::add (mt_Address*address, int hostType) 
{
  insist (this);
  return add (maxGmId + 1, hostType, address, address->toString ());
  exception: return 0;
}

int mt_HostTable::getGmId (mt_Address*address) 
{
  insist (this);
  mt_HostTable::Item*m;
  m = find (address);
  if (m)
    return m->gmId;
  exception: return -1;
}

int mt_HostTable::getNumItems()
{
  insist (this);
  return numItems;
  exception: return 0;
}
  
int mt_HostTable::getState (mt_Address*address)
{
  insist (this);
  insist (address);
  
  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
  return m->state;
  exception: return 0;
}

void mt_HostTable::setState (mt_Address*address, int state)
{
  insist (this);
  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
  m->state = state;
  exception: return;
}
  
int mt_HostTable::getOption (mt_Address*address)
{
  insist (this);
  insist (address);
  
  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
  return m->option;
  exception: return 0;
}

void mt_HostTable::setOption (mt_Address*address, int option)
{
  insist (this);
  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
  m->option = option;
  exception: return;
}

int mt_HostTable::getHostType (mt_Address*address)
{
  insist (this);
  insist (address);
  
  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
  return m->hostType;
  exception: return 0;
}

void mt_HostTable::setHostname (mt_Address*address, char*s)
{
  insist (this);
  insist (s);

  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
  strncpy (m->hostname, s, mt_Network::HOSTNAME_LENGTH);
  exception: return;
}

char*mt_HostTable::getHostname (mt_Address*address)
{
  insist (this);
  insist (address);
  
  mt_HostTable::Item*m;
  m = find (address);
  insist (m);
    
  return m->hostname;
  exception: return 0;
}

mt_HostTable::~mt_HostTable ()
{
  if (items) free (items);
}
void mt_HostTable::sortByAddress()
{
  insist (this);
  insist (sortType != BY_ADDRESS);
  qsort (items, numItems, sizeof (mt_HostTable::Item), addressCompare);
  sortType = BY_ADDRESS;
  lastItem = 0;
  exception: return;
}
void mt_HostTable::sortByGmId()
{
  insist (this);
  insist (sortType != BY_ID);
  qsort (items, numItems, sizeof (mt_HostTable::Item), gmIdCompare);
  sortType = BY_ID;
  lastItem = 0;
  exception: return;
}

void mt_HostTable::sortByName()
{
  insist (this);
  insist (sortType != BY_NAME);
  qsort (items, numItems, sizeof (mt_HostTable::Item), nameCompare);
  sortType = BY_NAME;
  lastItem = 0;
  exception: return;
}

void mt_HostTable::initialize ()
{
  maxItems = 100;
  numItems = 0;
  lastItem = 0;
  items = (mt_HostTable::Item*) malloc (sizeof (mt_HostTable::Item)* maxItems);
  insistp (items, ("mt_HostTable::initialize: alloc failed"));
  sortType = UNSORTED;
  maxGmId = 0;
  exception: return;
}

mt_HostTable::mt_HostTable () 
{
  initialize ();
}

int mt_HostTable::toFile (FILE*fp)
{
  insist (this);
  insist (fp);
  
  char t [HOSTNAME_LENGTH * 2];
  int i;
  insist (items && numItems <= maxItems && maxItems > 0);

  if (numItems == 0)
    return 1;
  
  if (sortType != BY_ID)
    sortByGmId ();

  for (i = 0; i < numItems; i++)
  {
    items[i].toString (t);
    fprintf (fp, "%s\n",t);
  }

  sortByAddress ();
  return 1;
  exception: return 0;
}

mt_HostTable::mt_HostTable (char*filename)
{
  mt_FileReader*reader = 0;
  char *t;

  insist (filename && *filename);
  initialize();
  
  FILE*fp;
  fp = fopen (filename, "r");
  if (!fp)
    return;
  fclose (fp);
  
  reader = new mt_FileReader (filename, 200, "");
  insist (reader);
  
  while ((t = reader->readLine ()))
  {
    mt_Token tokens (t, " \t");
    if (!tokens.getNext ())
      continue;

    if (!tokens.isInteger ())
    {
      printFormat ("syntax error on line %d in the file \"%s\". (expected gm id in first column).",
		   reader->getLineNumber(), filename);
      goto exception;
    }    
    int i = tokens.getInteger ();

    if (i < 0)
    {
      printFormat ("syntax error on line %d in the file \"%s\". (gm ids must be >= 0.)",
		   reader->getLineNumber(), filename);
      goto exception;
    }
    
    if (!tokens.getNext ())
    {
      printFormat ("syntax error on line %d in the file \"%s\". (expected host type in column 2).",
		   reader->getLineNumber(), filename);
      goto exception;
    }

    int hostType = tokens.getInteger ();
    
    if (!tokens.getNext ())
    {
      printFormat ("syntax error on line %d in the file \"%s\". (expected 48-bit board address in column 3).",
		   reader->getLineNumber(), filename);
      numItems  = maxGmId = 0;
      lastItem = 0;
      delete reader;
      return;
    }

    
    mt_Address address;
    address.fromString (tokens.getWord());
    tokens.getNext();

    if (getGmId (&address) >= 0 || getAddress (i))
    {
      char buffer [30];
      printFormat ("syntax error on line %d in the file \"%s\". (duplicate entry)",
		   reader->getLineNumber (),
		   filename,
		   address.toString (buffer));
      numItems = maxGmId = 0;
      lastItem = 0;
      delete reader;
      return;
    }
    
    add (i, hostType, &address , tokens.getWord ());
    mt_HostTable::Item*m = find (&address);
    insist (m);
  }
  sortByAddress ();
  delete reader;
  return;
  
  exception:
  if (reader) delete reader;
  numItems = maxGmId = 0;
  lastItem = 0;
  return;
}

int mt_HostTable::resolve (int gmId, int hostType, mt_Address*address)
{
  insist (this);
  insist (address);
  insistx (check ());
  
  int old;
  old = getGmId (address);
  
  if (old < 0)
  {
    if (gmId)
    {
      if (getAddress (gmId))
      {
	add (0, hostType, address);
	insistx (check ());
	return 0;
      }
      add (gmId, hostType, address);
    }
    else
      add (0, hostType, address);
  }
  else if (old == 0)
  {
    if (gmId)
    {
      if (getAddress (gmId))
	return 0;
      replace (gmId, address);
    }
  }
  else if (gmId && gmId != old)
    return 0;
  insistx (check ());
  return 1;
  exception: return 0;
}

int mt_HostTable::assignAddresses (mt_Graph*graph)
{
  if (sortType != BY_NAME)
    sortByName ();

  int numHosts;
  numHosts = graph->getNumHosts ();

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = graph->getHost (i);
    insist (n);

    mt_Address*address = n->getAddress ();
    insist (address);
    
    if (address->isZero ())
    {
      mt_HostTable::Item*m;
      m = find (n->getName ());
      insist (m);
      n->setAddress (&m->address);
    }
  }
  return 1;
  exception: return 0;
  
}
int mt_HostTable::assignGmIds (mt_Graph*graph)
{
  insist (this);
  insist (graph);
  
  if (sortType != BY_ADDRESS)
    sortByAddress ();
  
  int numHosts;
  numHosts = graph->getNumHosts ();

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = graph->getHost (i);
    insist (n);

    int tableId = getGmId (n->getAddress ());
    int nodeId = n->getGmId ();

    if (!tableId)
    {
      replace (++maxGmId, n->getAddress ());
      if (!nodeId)
	n->setGmId (maxGmId);
    }
    else if (!nodeId)
      n->setGmId (tableId);
      
  }
  return 1;
  exception: return 0;
}


int mt_HostTable::getNextGap (int*position, int*length, int*offset)
{
  insist (this);
  insist (position);
  insist (length);
  insist (offset);

  if (*position < numItems && *offset < *length)
    return items [*position].gmId + ++(*offset);

  if (*position == 0 && *length == 0)
  {
    int i;
    for (i = 0; i < numItems && !items [i].gmId; i++);
    if (i < numItems && (*length = items [i].gmId - 1))
        return getNextGap (position, length, offset);
  }

  for (; *position < numItems - 1; (*position)++)
  {
    if (!*position || !items [*position].gmId)
      continue;

    insist (items [*position].gmId < items [*position + 1].gmId);

    *length = items [*position + 1].gmId - items [*position].gmId - 1;
    if (*length >= 1)
    {
      *offset = 0;
      return getNextGap (position, length, offset);
    }
  }
 exception: return 0;
}


int mt_HostTable::assignGmIdsByFillingInGaps (mt_Graph*graph)
{
  insist (this);
  insist (graph);
  
  if (sortType != BY_ID)
    sortByGmId ();
  
  int numHosts;
  numHosts = graph->getNumHosts ();

  int newID, position, length, offset;
  position = length = offset = 0;
  
  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*n = graph->getHost (i);
    insist (n);

    int tableId = getGmId (n->getAddress ());
    int nodeId = n->getGmId ();

    if (!tableId)
    {
      newID = getNextGap (&position, &length, &offset);
      if (!newID)
	newID = ++maxGmId;
      
      sortType = UNSORTED;
      
      replace (newID, n->getAddress ());
      if (!nodeId)
	n->setGmId (newID);
    }
    else if (!nodeId)
      n->setGmId (tableId);
      
  }
  return 1;
  exception: return 0;
}

int mt_HostTable::check ()
{
  insist (this);
  
  for (int i = 0; i < numItems; i++)
    for (int j = i + 1; j < numItems; j++)
      if ((items[i].gmId > 0 && items [i].gmId == items [j].gmId) ||
	  !items[i].address.compare (&items [j].address))
	return 0;
  return 1;
  exception: return 0;
}

int mt_HostTable::setStates (int state)
{
  insist (this);
  insist (items && numItems <= maxItems && maxItems > 0);
  
  for (int i = 0; i < numItems; i++)
    items[i].state = state;
      
  exception: return 0;
}
