/*
  mt_Graph.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "insist.h"

#include "mt_Graph.h"
#include "mt_Host.h"
#include "mt_Switch.h"
#include "mt_Queue.h"
#include "mt_Cloud.h"


static int nodeCompare (const void *k, const void *e)
{
  mt_Node**a = (mt_Node**)k;
  mt_Node**b = (mt_Node**)e;;
 
  insistf (a && *a && b && *b);
  return (strcmp ((*a)->getName (), (*b)->getName ()));

  exception: return 0;
}

mt_Graph::mt_Graph ()
{
  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
  {
    nodes [i] = 0;
    numNodes [i] = 0;
    maxNodes [i] = 0;
    maxStringLengths [i] = 0;
  }
  root = 0;
}

int mt_Graph::getNumNodes ()
{
  return getNumNodes (mt_Node::NODE);
}

int mt_Graph::getNumHosts ()
{
  return getNumNodes (mt_Node::HOST);
}

int mt_Graph::getNumSwitches ()
{
  return getNumNodes (mt_Node::SWITCH);
}

int mt_Graph::getNumNodes (int type)
{
  insist (this);
  insist (type >= 0 && type < mt_Node::NUM_TYPES);
  
  return numNodes [type];
  
  exception: return 0;
}

mt_Node*mt_Graph::getNode (int type, int index)
{
  insist (this);
  insist (type >= 0 && type < mt_Node::NUM_TYPES);
  insist (index >= 0 && index < numNodes [type]);
  
  return nodes [type] [index];
  
  exception: return 0;
}

  
int mt_Graph::clone (mt_Graph*g)
{
  insist (g);

  empty ();
  
  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
    if (!setSize (i, g->numNodes [i]))
      return 0;
  
  for (int i = mt_Node::HOST; i < mt_Node::NUM_TYPES; i++)
  {
    for (int j = 0; j < g->numNodes [i]; j++)
    {
      mt_Node*o = g->getNode (i, j);
      insist (o);
      mt_Node*n = newNode (i, o->getName (), o->getType());
      insist (n);
      *n = *o;      
      n->startClone (o);
      add (n);
    }
  }
  
  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
  {
    insist (numNodes [i] == g->numNodes [i]);
  }
  
  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
    nodes [mt_Node::NODE][i]->finishClone ();

  return 1;
  exception: return 0;
}

int mt_Graph::toString (char*s)
{
  char*t = s;

  insist (this);
  insist (s);
  
  int n;
  n = numNodes [mt_Node::NODE];
  
  mt_Node**sorted;
  sorted = (mt_Node**) malloc (sizeof (mt_Node*) * n);
  insistp (sorted, ("malloc failed"));
  
  for (int i = 0; i < n; i++)
    sorted [i] = nodes [mt_Node::NODE][i];

  qsort (sorted, n, sizeof (mt_Node*), nodeCompare);
  
  for (int i = 0; i < n; i++)
    t += sprintf (t, "%s\n", sorted [i]->toString (t));

  free (sorted);
  
  return (int) (t - s);
  exception: return 0;
}

mt_Node*mt_Graph::unequalMapVersions (mt_Graph*graph)
{
  insist (this);
  insist (graph);
  
  int numHosts;
  numHosts = getNumNodes (mt_Node::HOST);
  
  if (numHosts != graph->getNumNodes (mt_Node::HOST))
    return 0;

  for (int i = 0; i < numHosts; i++)
  {
    mt_Node*h1 = getNode (mt_Node::HOST, i);
    mt_Node*h2 = graph->getHost (h1->getAddress ());
    
    if (!h2 || h1->getMapVersion () != h2->getMapVersion ())
      return h1;  
  }
  
  return 0;
  exception: return 0;
}

int mt_Graph::stringEquals (mt_Graph*graph)
{
  char*s1 = 0;
  char*s2 = 0;
  int n1, n2;
  
  insist (this);
  insist (graph);

  s1 = new char [getMaxStringLength ()];
  s2 = new char [graph->getMaxStringLength ()];

  insistp (s1 && s2, ("mt_Graph::equals: alloc failed"));

  *s1 = *s2 = 0;
  
  n1 = toString (s1);
  n2 = graph->toString (s2);
  
  int e;
  e = *s1 && *s2 && n1 == n2 && !strcmp (s1, s2);  
   
  delete [] s1;
  delete [] s2;

  return e;
  
  exception:
  if (s1) delete [] s1;
  if (s2) delete [] s2;
  return 0;
}

int mt_Graph::matchKids (mt_Node*n1, mt_Node*n2)
{
  insist (this);
  insist (n1 && n2);
  
  int nn1;
  nn1 = n1->getMaxNodes ();
  int nn2;
  nn2 = n2->getMaxNodes ();
  
  if (nn1 != nn2)
    return 0;

  for (int i = 0; i < nn1; i++)
  {
    mt_Node*n1n = n1->getNode (i);
    mt_Node*n2n = n2->getNode (i);

    if (!n1n && !n2n)
      continue;
    
    if (!n1n || !n2n)
      return 0;
  
    if (n1->getOpposite (i) != n2->getOpposite (i))
      return 0;
    
    if (!match (n1n, n2n))
      return 0;
  }
  return 1;
  exception: return 0;
}

int mt_Graph::match (mt_Node*n1, mt_Node*n2)
{
  insist (this);
  insist (n1 && n2);
  insist (n1->getName () && n2->getName ());
  
  if (n1->getMatch () == n2)
  {
    insist (n2->getMatch () == n1);
    return 1;
  }

  if (n1->getMatch () || n2->getMatch ())
    return 0;
  
  if (n1->getNodeType () != n2->getNodeType ())
    return 0;
  
  insist (!n1->isHost () || n1->getAddress ());
  
  if (n1->isHost () && (n1->getAddress ()->compare (n2->getAddress ()) || strcmp (n1->getName (), n2->getName ()) ||
			n1->getHostType () != n2->getHostType ()))
    return 0;

  n1->setMatch (n2);
  n2->setMatch (n1);
   
  return matchKids (n1, n2);

  exception: return 0;
}

int mt_Graph::equals (mt_Graph*graph)
{
  insist (this && graph);
  
  if (getNumNodes () != graph->getNumNodes ())
    return 0;
  
  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
  {
    insist (nodes [mt_Node::NODE][i]);
    insist (graph->nodes [mt_Node::NODE][i]);
    
    nodes [mt_Node::NODE][i]->setMatch (0);
    graph->nodes [mt_Node::NODE][i]->setMatch (0);
  }
    
  mt_Node*h1,*h2;
  h1 = getHost (0);
  insist (h1 && h1->getAddress ());
  h2 = graph->getHost (h1->getAddress ());
  if (!h2) return 0;
  
  return match (h1, h2);
  
  exception: return 0;
}


mt_Graph::~mt_Graph ()
{
  empty ();

  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
  {
    maxStringLengths [i] = 0;
    if (nodes [i]) delete nodes [i];
  }
}

void mt_Graph::empty ()
{
  if (nodes [mt_Node::NODE])
  {
    for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
      delete nodes [mt_Node::NODE][i];
  }
  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
    numNodes [i] = 0;
}

int mt_Graph::setSize (int type, int size)
{
  insist (this);
  insist (type >= 0 && type < mt_Node::NUM_TYPES);
  insist (size >= 0);
  
  if (size > maxNodes [type])
  {
    mt_Node**a = new mt_Node * [size];

    insistp (a, ("mt_Graph::setSize: alloc failed"));
    insist (numNodes [type]  <= maxNodes [type]);
    insist (nodes [type] || numNodes [type] == 0);
    
    for (int i = 0; i < numNodes [type]; i++)
      a [i] = nodes [type] [i];
    if (nodes [type]) delete [] nodes [type];
    nodes [type] = a;
    maxNodes [type] = size;    
  }

  if (type != mt_Node::NODE)
  {
    int n = 0;
    for (int i = mt_Node::NODE + 1; i < mt_Node::NUM_TYPES; i++)
      n += maxNodes [i];
    return setSize (mt_Node::NODE, n);
  }
  return 1;
  exception: return 0;
}

void mt_Graph::add (mt_Node*n)
{
  insist (this);
  insist (n);
  
  /*
  printFormat ("addding %s as %s %d, node %d", n->getName (),
	       n->isHost () ? "host" : "switch",
	       n->isHost () ? numHosts: numSwitches,
	       numNodes);
	       */

  int type;
  type = n->getNodeType ();
  insist (type > mt_Node::NODE && type < mt_Node::NUM_TYPES);
  
  if (numNodes [type] == maxNodes [type])
    setSize (type, maxNodes [type] ? maxNodes [type] * 2 : 10);
  insist (numNodes [type] < maxNodes [type]);
  nodes [type][numNodes [type]] = (mt_Node*) n;

  n->setTypeIndex (numNodes [type]++);
    
  int len;
  len = n->getMaxStringLength ();
  if (len > maxStringLengths [type])
    maxStringLengths [type] = len;
  
  insist (numNodes [mt_Node::NODE] < maxNodes [mt_Node::NODE]);
  nodes [mt_Node::NODE] [numNodes [mt_Node::NODE]] = n;
  n->setNodeIndex (numNodes [mt_Node::NODE]++);
  exception: return;
}

void mt_Graph::remove (mt_Node*n)
{
  insist (this);
  insist (n);
  
  int type;
  type = n->getNodeType ();
  insist (type > mt_Node::NODE && type < mt_Node::NUM_TYPES);
  
  int i;
  i = n->getTypeIndex ();
  insist (i >= 0 && i < numNodes [type]);
  nodes [type][i] = nodes [type] [--numNodes [type]];
  nodes [type][i]->setTypeIndex (i);

  i = n->getNodeIndex ();
  insist (i >= 0 && i < numNodes [mt_Node::NODE]);
  nodes [mt_Node::NODE][i] = nodes [mt_Node::NODE][--numNodes [mt_Node::NODE]];
  nodes [mt_Node::NODE][i]->setNodeIndex (i);
  delete n;

  exception: return;
}

int mt_Graph::removeMarked (int mark)
{
  insist (this);
  
  /*numNodes[][] can change in the loop*/
  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
  {
    insist (nodes [mt_Node::NODE][i]);
    if (nodes [mt_Node::NODE][i]->getMark () == mark)
     remove (nodes [mt_Node::NODE][i]);
  }
  return 1;
  exception: return 0;
}

mt_Node*mt_Graph::getNode (char*s)
{
  insist (this);
  insist (s && *s);

  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
    if (!strcmp (s, nodes [mt_Node::NODE][i]->getName ()))
      return nodes [mt_Node::NODE][i];
  
  exception: return 0;
}

int mt_Graph::getHostIndex (char*s)
{
  insist (this);
  insist (s && *s);

  mt_Node*n;
  n = getNode (s);
  insist (n && n->isHost ());
  return n->getTypeIndex ();
  
  exception: return -1;
}

mt_Node*mt_Graph::getNode (int n)
{
  return getNode (mt_Node::NODE, n);
}

mt_Node*mt_Graph::getHost (int n)
{
  return getNode (mt_Node::HOST, n);
}


mt_Node*mt_Graph::getNode (int type, mt_Address*address)
{
  insist (this);
  insist (type >= 0 && type < mt_Node::NUM_TYPES);
  insist (address);  

  for (int i = 0; i < numNodes [type]; i++)
    if (!address->compare (nodes [type][i]->getAddress ()))
      return nodes [type][i];

  exception: return 0;
}

mt_Node*mt_Graph::getHost (mt_Address*address)
{
  return getNode (mt_Node::HOST, address);
}

mt_Node*mt_Graph::getGmId (int gmId)
{
  insist (this);
  insist (gmId > 0);

  for (int i = 0; i < numNodes [mt_Node::HOST]; i++)
    if (nodes [mt_Node::HOST][i]->getGmId () == gmId)
      return nodes [mt_Node::HOST][i];
  exception: return 0;
}

mt_Node*mt_Graph::getSwitch (int n)
{
  return getNode (mt_Node::SWITCH, n);
}

int mt_Graph::getMaxNodeStringLength ()
{
  insist (this);

  int n;
  n = 0;
  
  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
    if (n < maxStringLengths [i])
      n = maxStringLengths [i];

  return n;
  exception: return 0;
}

int mt_Graph::getMaxStringLength ()
{
  insist (this);

  int n;
  n = 0;
  
  for (int i = 0; i < mt_Node::NUM_TYPES; i++)
    n += maxStringLengths [i] * numNodes [i];
  
  return n;
  exception: return 0;
}

mt_Node*mt_Graph::newNode (int nodeType, char*name, char*type)
{
  insist (this);
  switch (nodeType)
  {
    case mt_Node::HOST:
      return new mt_Host (name, type);
    case mt_Node::SWITCH:
      return new mt_Switch (name, type);
    case mt_Node::CLOUD:
      return new mt_Cloud (name, type);
    default:
      insist (0);
  }
  exception: return 0;
}

void mt_Graph::clearNumbers ()
{
  insist (this);
  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
  {
    insist (nodes [mt_Node::NODE] [i]);
    
    nodes [mt_Node::NODE] [i]->number = -1;
    nodes [mt_Node::NODE] [i]->setHeight (-1);
  }
  exception: return;
}

void mt_Graph::clearMarks ()
{
  insist (this);

  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
  {
    insist (nodes [mt_Node::NODE][i]);
    nodes [mt_Node::NODE] [i]->setMark (mt_Node::_NONE);
  }
  exception: return;
}

int mt_Graph::autoNumber ()
{
  queue.empty ();
  
  insist (this);
  
  clearNumbers ();

  int numHosts;
  numHosts = getNumHosts();
  int numSwitches;
  numSwitches = getNumSwitches();
  int switchesFound;
  switchesFound = 0;
  int maxHeight;
  maxHeight = 0;

  for (int i= 0; i < numHosts; i++)
  {
    mt_Node*h = getHost(i);
    insist (h);

    mt_Node*s = h->getNode(0);
    insist (s || numHosts == 1);
    
    if (!s) return 0;

    if (s->getHeight () == -1)
    {
      s->setHeight (0);
      switchesFound++;
      queue.put(s);
    }
  }

  mt_Node *s;
  while ((s = (mt_Node*) queue.get ()))
  {
    insist (s);
    
    int maxNodes = s->getMaxNodes ();
    
    for (int i = 0; i < maxNodes; i++)
    {
      mt_Node*n = s->getNode (i);
      if (n && n->isSwitch ())
      {
	if (n->getHeight () == -1)
	{
	  n->setHeight (s->getHeight () + 1);
	  if (n->getHeight () > maxHeight)
	    maxHeight = n->getHeight ();
	  queue.put (n);
	}
      }
    }
  }

  int count;
  count = 0;

  for (int ht = maxHeight; ht >= 0; ht--)
    for (int si = 0; si < numSwitches; si++)
      if (getSwitch (si)->getHeight () == ht)
	getSwitch (si)->number = count++;
  
  return 1;
  exception: return 0;
}

int mt_Graph::renumber (int numRoots, char**roots)
{
  mt_Node**a = 0;
  
  insist (this);
  insist (numRoots > 0 && roots);

  a = new mt_Node* [numRoots];
  insistp (a, ("mt_Graph::renumber: alloc failed"));

  for (int i = 0; i < numRoots; i++)
  {
    insist (roots [i]);
    a [i] = (mt_Node*)getNode (roots [i]);
    insistp (a [i], ("node \"%s\" does not exist.", roots [i]));
    insistp (a [i]->isSwitch (), ("node \"%s\" is not a switch.", roots [i]));
  }
  
  int result;
  result = renumber (numRoots, a);

  delete a;
  return result;
  
  exception:
  if (a) delete a;
  return 0;
}

int mt_Graph::renumber (int host)
{
  insist (this);
  insist (host >= 0 && host < numNodes [mt_Node::HOST]);

  mt_Node*h;
  h = nodes [mt_Node::HOST] [host];
  insist (h);

  mt_Node*s;
  s = h->getNode (0);
  return !s || s->isHost () || renumber (1, &s);
  
  exception: return 0;
}

int mt_Graph::renumber (int numRoots, mt_Node**roots)
{
  queue.empty ();
  insist (this);
  insist (numRoots > 0 && roots);
  
  int count;
  count = 0;

  clearNumbers ();
  
  for (int i = 0; i < numRoots; i++)
  {
    insist (roots [i]);
    mt_Node*n = getNode (roots[i]->getName ());
    insist (n && !strcmp (n->getName (), roots [i]->getName ()));

    n->number = count++;
    queue.put (n);
  }
  
  mt_Node*s;
  while ((s = (mt_Node*) queue.get ()))
  {
    int maxNodes = s->getMaxNodes ();
    
    for (int i = 0; i < maxNodes; i++)
    {
      mt_Node*n = s->getNode (i);

      if (n && n->isSwitch ())
      {
	insist (getNode (n->getName ()) == n);

	if (n->number == -1)
	{
	  n->number = count++;
	  queue.put (n);
	}
      }
    }
  }
  return 1;
  exception: return 0;
}

int mt_Graph::setRoutes (mt_Node*root)
{
  mt_Queue queue;
  int count = 0;
  
  insist (this);
  insist (root);  

  clearMarks ();
  
  mt_Node*s;
  s = root->getNode (0);
  if (!s)
  {
    root->setRoute (0, mt_Route ());
    return 1;
  }
  
  if (s->isHost ())
  {
    s->setRoute (0, mt_Route ());
    root->setRoute (0, mt_Route ());
  }
  else
  {
    queue.put (s);
    s->number = ++count;
    s->setIn (root->getOpposite (0));
    s->setRoute (0, mt_Route ());
    root->setRoute (0, mt_Route (s->getRoute (0), 0));
    s->setMark (mt_Node::UP);
  }
  
  while ((s = (mt_Node*) queue.get ()))
  {
    insist (s->isMarked ());
    int maxNodes = s->getMaxNodes ();
    
    for (int i = 0; i < maxNodes; i++)
    {
      mt_Node*n = s->getNode (i);
      
      if (n)
      {
	mt_Route r (s->getRoute (0), i - s->getIn ());
	
	if (n->isSwitch () && !n->isMarked ())
	{
	  n->setIn (s->getOpposite (i));
	  n->setRoute (0, r);
	  queue.put (n);
	  n->setMark (mt_Node::UP);
	}
	else if (n != root && !n->isSwitch ())
	  n->setRoute (0, r);
	insist (root->getNode (n->getRoute (0)) == n);
      }
    }
  }
  return 1;
  exception: return 0;
}


int mt_Graph::getMaxMapVersion ()
{
  insist (this);
  int t, m;
  m = 0;
  
  for (int i = 0; i < numNodes [mt_Node::HOST]; i++)
    if ((t = nodes [mt_Node::HOST] [i]->getMapVersion ()) > m)
      m = t;

  return m;
  exception: return 0;
}


mt_Node*mt_Graph::getNode (int type, int x, int y, int z)
{
  insist (this);
  insist (x >= 0 && y >= 0);
  insist (type >= 0 && type < mt_Node::NUM_TYPES);
  
  for (int i = 0; i < numNodes [type]; i++)
  {
    mt_Node*n = nodes [type] [i];
    insist (n);
    if (n->getMeshX () == x && n->getMeshY () == y && n->getMeshZ () == z)
      return n;
  }
  exception: return 0;
}

mt_Node*mt_Graph::getHost (int x, int y, int z)
{
  insist (this);
  return getNode (mt_Node::HOST, x, y, z);
  exception: return 0;
}

mt_Node*mt_Graph::getNumberedSwitch (int n)
{
  insist (this);

  for (int i = 0; i < numNodes [mt_Node::SWITCH]; i++)
    if (nodes [mt_Node::SWITCH] [i]->number == n)
      return nodes [mt_Node::SWITCH] [i];

  exception: return 0;
}

mt_Node*mt_Graph::getSwitch (int x, int y, int z)
{
  insist (this);
  return getNode (mt_Node::SWITCH, x, y, z);
  exception: return 0;
}


int mt_Graph::fillLevels (mt_Node*node, mt_Node**nodes, mt_Node***levels, int*nodesPerLevel, int maxLevel)
{
  insist (this);
  insist (node && nodes && levels && nodesPerLevel);
  insist (maxLevel > 0);

  mt_Node**source;
  int maxNodes;
  int type;
  type = node->getNodeType ();
  insist (type >= 0 && type < mt_Node::NUM_TYPES);
  
  source = this->nodes [type];
  maxNodes = numNodes [type];
  
  insist (source && numNodes > 0);

  for (int i = 0; i < maxLevel; i++)
    nodesPerLevel [i] = 0;
  
  for (int i = 0; i < maxNodes; i++)
  {
    int level = node->getDistance (source [i]);
    insist (level >= 0);
    
    if (level >= maxLevel)
      level = maxLevel - 1;
    nodesPerLevel [level]++;
  }

  int total;
  total = 0;
  for (int i = 0; i < maxLevel; i++)
  {
    insist (total >= 0 && total <= maxNodes);
    
    levels [i] = &nodes [total];
    total += nodesPerLevel [i];
  }

  insist (total == maxNodes);
  
  for (int i = 0; i < maxLevel; i++) 
    nodesPerLevel [i] = 0;
  
  for (int i = 0; i < maxNodes; i++)
  {
    int level = node->getDistance (source [i]);
    if (level >= maxLevel)
      level = maxLevel - 1;

    insist (nodesPerLevel [level] >= 0 && nodesPerLevel [level] < maxNodes);
    
    (levels [level])[nodesPerLevel [level]] = source [i];    
    nodesPerLevel [level]++;
  }
  return 1;
  exception: return 0;
}

int mt_Graph::removeSteppenwoelfe ()
{
  insist (this);

  for (int i = 0; i <  numNodes [mt_Node::NODE]; i++)
  {
    insist (nodes [mt_Node::NODE][i]);
    if (!nodes [mt_Node::NODE][i]->getNumNodes ())
     remove (nodes [mt_Node::NODE][i]);
  }
  return 1;
  exception: return 0;
}

int mt_Graph::removeLinks (mt_GraphRemoveLinksFunction function)
{
  insist (this);
  insist (function);

  for (int i = 0; i < numNodes [mt_Node::NODE]; i++)
  {
    mt_Node*n = nodes [mt_Node::NODE][i];
    int maxNodes = n->getMaxNodes ();

    for (int j = 0; j < maxNodes; j++)
    {
      if (n->getNode (j) && (*function) (n, n->getNode (j), j))
	n->disconnect (j);
    }
  }
  return 1;
  exception: return 0;
}

void mt_Graph::setRoot (mt_Node*n)
{
  insist (this);
  insist (n);
  root = n;
  exception:;
}

mt_Node*mt_Graph::getRoot ()
{
  insist (this);
  return root;
  exception: return 0;
}


int mt_Graph::computeRoutes (mt_Node*root)
{
  mt_Queue switches;

  int numSwitches = 0;
  insist (this);
  insist (root);
  
  mt_Switch*s;
  s = (mt_Switch*) root->getNode (0);
  insist (s);
  
  clearMarks ();
  clearNumbers ();
  
  root->setHeight (0);
  s->setMark (mt_Node::UP);
  s->setIn (root->getOpposite (0));
  
  switches.empty ();
  switches.put (s);

  while ((s = (mt_Switch*) switches.get ()))
  {
    numSwitches++;
    
    int maxNodes = s->getMaxNodes ();
    
    for (int i = 0; i < maxNodes; i++)
    {
      mt_Node*n = s->getNode (i);

      if (n && !n->isMarked ())
      {
	mt_Route r (s->getRoute (0), i - s->getIn ());  
	mt_Node*fn;
	int fin;
	
	n->setIn (s->getOpposite (i));
	n->setRoute (0, r);
	n->setMark (mt_Node::UP);
	n->setHeight (s->getHeight () + 1);

	insist (n->isHost () || root->follow (n->getRoute (0),  &fn, &fin) && fn == n);
	
	if (n->isSwitch ())
	  switches.put (n);
      }
    }
  }
  insist (numSwitches == getNumSwitches ());
  return numSwitches;
  exception: return 0;
}
