/*
  mt_Calculator.c
  map tools
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "insist.h"
#include "mt_Calculator.h"
#include "mt_RouteFile.h"
#include "mt_MapFile.h"
#include "mt_Switch.h"

mt_Calculator::mt_Calculator ()
{
  maxPositions = MAX_POSITIONS;
  positions = (int*) malloc (sizeof (int) * MAX_POSITIONS);
  insistp (positions, ("mt_Calculator::mt_Calculator malloc failed\n"));
  
  sections = 0;
  links = 0;
  
  return;
  exception:;
}

mt_Calculator::~mt_Calculator ()
{
  if (positions)
    free (positions);
  if (sections)
    free (sections);
  if (links)
    free (links);
}

int mt_Calculator::getLinkPort (mt_Node*from, mt_Node*to)
{
  insist (this);
  insist (links);
  insist (from && to);
  insist (from->isSwitch () && to->isSwitch ());
  
  int fromIndex;
  fromIndex = from->getTypeIndex ();
  insist (from == getSwitch (fromIndex));
  
  Link*link;
  link = 0;

  for (int i = 0; i < mt_Switch::NUM_PORTS && links [mt_Switch::NUM_PORTS * fromIndex + i].other; i++)
  {
    if (links [mt_Switch::NUM_PORTS * fromIndex + i].other == to)
    {
      link = &links [mt_Switch::NUM_PORTS * fromIndex + i];
      break;
    }
  }
  insist (link);
  
  insist (link->other && link->numLinks >= 1 && link->numLinks <= mt_Switch::NUM_PORTS);

  int pick;
  pick = link->count % link->numLinks;
  link->count++;
  int numSeen;
  numSeen = 0;
  
  for (int i = 0; i < mt_Switch::NUM_PORTS; i++)
  {
    mt_Node*node = from->getNode (i);
    if (node == to)
    {
      if (numSeen == pick)
	return i;
      numSeen++;
    }    
  }
  
  exception: return -1;
}

int mt_Calculator::makeLinks ()
{
  insist (this);
  
  links = (Link*) calloc (getNumSwitches () * mt_Switch::NUM_PORTS, sizeof (Link));
  insistp (links, ("mt_Calculator::makeLinks: calloc failed\n"));
  
  for (int i = 0; i < getNumSwitches (); i++)
  {
    mt_Node*node = getSwitch (i);
    
    for (int j = 0; j < mt_Switch::NUM_PORTS; j++)
    {
      mt_Node*other = node->getNode (j);
      if (other && other->isSwitch ())
      {
	for (int k = 0; k < mt_Switch::NUM_PORTS; k++)
	{
	  Link*link = &links [mt_Switch::NUM_PORTS * i + k];
	  if (link->other)
	  {
	    if (link->other == other)
	    {
	      link->numLinks++;
	      break;
	    }
	  }
	  else
	  {
	    link->other = other;
	    link->numLinks = 1;
	    break;
	  }
	}
      }
    }
  }
  return 1;
  exception: return 0;
}

int mt_Calculator::initialize (mt_Graph*graph, mt_Node*)
{
  insist (graph);
  mt_Component::srand ();
  return clone (graph) && makeLinks ();
  
  exception: return 0;
}

int mt_Calculator::initialize (char *mapFile)
{
  insist (this);
  insist (mapFile && *mapFile);
  mt_Component::srand ();
  
  mt_MapFile*mf;
  mf = new mt_MapFile (mapFile, mt_File::_READ);
  if (!mf->read (this))
  {
    delete mf;
    return 0;
  }
  delete mf;
  return makeLinks ();

  exception: return 0;
}

void mt_Calculator::cleanup ()
{
  insist (this);
  mt_Graph::empty ();
  
  if (sections)
  {
    free (sections);
    sections = 0;
  }
  if (links)
  {
    free (links);
    links = 0;
  }
  
  exception: return;
}

int mt_Calculator::writeRoutes (FILE*fp)
{
  mt_RouteFile routeFile (fp);

  // yuck
  insist (this);
  insist (fp);

  return writeRoutes (&routeFile);
  
  exception: return 0;
}

int mt_Calculator::writeRoutes (char*filename)
{
  mt_RouteFile routeFile (filename, mt_File::_WRITE);

  // yuck
  insist (this);
  insist (filename);

  return writeRoutes (&routeFile);
  
  exception: return 0;
}


int mt_Calculator::writeRoutes (mt_RouteFile*routeFile)
{  
  insist (this);
  insist (routeFile);

  return routeFile->write (this);
  exception: return 0;
}

int mt_Calculator::getNumBytes (int from)
{
  int numBytes = 0;
  
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  
  int maxRoutes;
  maxRoutes = getMaxRoutes ();

  for (int j = 0; j < maxRoutes; j++)
    numBytes += getNumBytes (from, j);

  return numBytes;
  exception: return 0;
}

int mt_Calculator::getNumBytes (int from, int index)
{
  int numBytes = 0;
  
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  
  int numHosts;
  numHosts = getNumHosts ();
  for (int i = 0; i < numHosts; i++)
  {
    int numRoutes = getNumRoutes (from, i);
    insist (numRoutes > 0);

    insist (index >= 0);

    if (index >= numRoutes)
      return 0;

    mt_Route r;
    int n = getRoute (from, i, index, &r);
    insist (n);
    numBytes += 1 + r.getLength ();
  }
  return numBytes;
  exception: return 0;
}

int mt_Calculator::growPositions ()
{
  insist (maxPositions && positions);
  int*p;
  p = (int*) malloc (maxPositions * 2 * sizeof (int));
  insistp (p, ("mt_Calculator::growPositions: malloc failed"));
  
  for (int i = 0; i < maxPositions; i++)
    p [i] = positions [i];
  
  int numHosts;
  numHosts = getNumHosts ();
  
  for (int i = 0; i < numHosts; i++)
    sections [i] = p + (sections [i] - positions);
  
  maxPositions *= 2;
  free (positions);
  positions = p;
  return 1;
  
  exception: return 0;
}
  
int mt_Calculator::getNumSections ()
{
  return numSections;
}

int mt_Calculator::getFirstSection (int from)
{
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  insist (sections && positions && maxPositions);
  insist (sections [0] && sections [from]);
  insist (numSections);
  insist (sections [from] - sections [0] >= 0);
  insist (sections [from] - sections [0] < numSections);
  
  return (int) (sections [from] - sections [0]);

  exception: return 0;
}

int mt_Calculator::computeSections (mt_HostTable*hostTable, int sizePerHost, int sizePerRoute, int size)
{
  insist (this);
  insist (hostTable);
  insist (size > 0);
  insist (size > mt_HostTable::HOSTNAME_LENGTH);
  
  int numHosts;
  numHosts = getNumHosts ();
  
  int numPositions;
  numPositions = 0;
  numSections = 0;

  if (sections)
    free (sections);
  
  sections = (int**) calloc (numHosts,  sizeof (int*));
  insistp (sections, ("mt_Calculator::computeSections: calloc failed"));
  
  insist (positions && maxPositions);
  
  for (int i = 0; i < numHosts; i++)
  {
    if (maxPositions == numPositions && !growPositions ())
      return 0;
    insist (maxPositions > numPositions);
 
    sections [i] = &positions [numPositions];
    numSections++;
    positions [numPositions++] = 1;
    
    int numBytes = 0;
    int lastBytes = 0;
    
    for (int j = 0; j < numHosts; j++)
    {      
      if (numBytes >= size)
      {
	if (maxPositions == numPositions && !growPositions ())
	  return 0;
	insist (maxPositions > numPositions);
	
	insist (j);
	positions [numPositions] = j - 1;
	numPositions++;
	sections [i] [0] ++;
	numSections++;
	
	int t = numBytes;
	numBytes -= lastBytes;
	lastBytes = t;
      }
      
      numBytes += sizePerHost
	+ (int) strlen (hostTable->getHostname (getHost (j)->getAddress ()));
      
      int numRoutes = getNumRoutes (i, j);
      numBytes += sizePerRoute * numRoutes;
      
      for (int k = 0; k < numRoutes; k++)
      {
	mt_Route r;
	getRoute (i, j, k, &r);
	numBytes += r.getLength ();
      }
    }
  }
  return numSections;
  exception: return 0;
}

int mt_Calculator::getNumSections (int from)
{
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  insist (sections && positions && maxPositions);
  insist (from < getNumHosts ());
  
  insist (sections [from]);
  insist (sections [from][0] > 0 && sections [from][0] <= getNumHosts () * getNumHosts ());
  return sections [from][0];

  exception:return 0;
}

int mt_Calculator::getSection (int from, int index, int*start, int*stop)
{
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  insist (sections && positions && maxPositions);
  insist (from < getNumHosts ());
  insist (start && stop);
  insist (index >= 0);  

  insist (sections [from]);
  insist (index < sections [from][0]);
  
  *start = index ? sections [from][index] : 0;
  *stop = index < sections [from][0] - 1 ? sections [from][index + 1] : getNumHosts ();

  return 1;
  exception: return 0;
}

int mt_Calculator::getNumRoutes (int from)
{
  int numRoutes = 0;
  int numHosts;

  insist (this);

  numHosts = getNumHosts ();
  
  for (int i = 0; i < numHosts; i++)
    numRoutes += getNumRoutes (from, i);
  return numRoutes;
  exception: return 0;
}

int mt_Calculator::getSpreadRoute (int from, int to, int routeIndex, mt_Route*route)
{
  mt_Route r;
  
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  insist (to >= 0 && to < getNumHosts ());
  insist (getHost (from) && getHost (to));
  
  if (!getRoute (from, to, routeIndex, &r))
    return 0;

  insist (getHost (from)->getNode (&r) == getHost (to));

  int numHops;
  numHops = r.getLength ();
  char*hops;
  hops = r.getHops ();
  insist (hops);
  
  if (numHops <= 1)
  {
    *route = r;
    return 1;
  }
  
  char*newHops;
  newHops = (char*) malloc (numHops);
  insistp (newHops, ("mt_Calculator::getSpreadRoute malloc failed"));
  
  int in;
  in = getHost (from)->getOpposite (0);
  mt_Node*n;
  n = getHost (from)->getNode (0);
  insist (n);
  
  int numNewHops;
  numNewHops = 0;
  int newIn;
  newIn = in;
  
  while (numNewHops < numHops)
  {
    insist (in + hops [numNewHops] >= 0 && in + hops [numNewHops] < mt_Switch::NUM_PORTS);
    
    mt_Node*m;
    m = n->getNode (in + hops [numNewHops]);
    in = n->getOpposite (in + hops [numNewHops]);
    
    insist (m && in >= -mt_Switch::NUM_PORTS && in < mt_Switch::NUM_PORTS);
    insist (numNewHops < numHops);

    int p = m->isSwitch () ? getLinkPort (n, m) : n->getPort (m);
    insist (m == n->getNode (p));

    insist (p - newIn >= - mt_Switch::NUM_PORTS && p - newIn < mt_Switch::NUM_PORTS);
    newHops [numNewHops++] = p - newIn;

    newIn = n->getOpposite (p);
    n = m;
  }
  
  route->fromBytes (newHops, numNewHops);
  if (getHost (from)->getNode (route) != getHost (to))
  {
    printf ("fuck\n");
  }
  
  insist (getHost (from)->getNode (route) == getHost (to));
  
  free (newHops);
  insist (numNewHops == numHops);
  
  return 1;
  exception: return 0;
}

