/*
  gr_Simulation.c
  griddle
  finucane@myri.com (David Finucane)
*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "insist.h"
#include "gr_Simulation.h"
#include "sm_Host.h"
#include "mt_htonl.h"

class gr_Message : mt_Queueable
{
  public:  
  int type;
  int x;
  int y;
  int z;
  int nx;
  int ny;
  int nz;
  double temperature;
  void swap ();
};

void gr_Message::swap ()
{
  type = mt_htonl (type);
  x = mt_htonl (x);
  y = mt_htonl (y);
  z = mt_htonl (z);
}


gr_Job::~gr_Job ()
{
}

gr_Job::gr_Job (mt_Node*node, mt_Network*network, gr_Simulation *simulation)  : mt_Job (node, network)
{
  insist (this);
  insist (node && network && simulation);  

  this->simulation = simulation;
  x = node->getMeshX ();
  y = node->getMeshY ();
  z = node->getMeshZ ();
  cycle = 0;

  numSends = 0;
  numReceives = 0;
  
  for (int i = 0; i < NUM_TYPES; i++)
    typeReceives [i] = typeSends [i] = 0;
  
  insist (x >= 0 && y >= 0 && z >= 0);

  side = simulation->getSide ();
  insist (side > 0); 
  insist (getIndex (x, y, z) == node->getTypeIndex ());
  
  forget ();
  temperature = simulation->getStartTemperature (x, y, z);

  exception: return;
}

double gr_Job::getTemperature ()
{
  insist (this);
  return temperature;
  exception: return 0;
}

  
int gr_Job::isEdgeNode (int x, int y, int)
{
  insist (this);
  return x == 0 || y == 0 || x == side - 1 || y == side - 1;
  exception: return 0;
}

void gr_Job::compute ()
{
  insist (this);
  temperature = 0;

  for (int i = 0; i < NUM_POSITIONS; i++)
    temperature += temperatures [i];
  
  temperature /= NUM_POSITIONS;
  
  exception: return;
}

void gr_Job::forget ()
{
  insist (this);
  memset (receives, 0, NUM_POSITIONS * sizeof (int));
  memset (temperatures, 0, NUM_POSITIONS * sizeof (double));
  exception: return;
}

int gr_Job::ready ()
{
  insist (this);
  
  for (int i = 0; i < NUM_POSITIONS; i++)
    if (!receives [i])
      return 0;
  return 1;
  exception: return 0;
}

int gr_Job::start ()
{
  cycle = 0;
  
  mt_Address address;
  char name [mt_Network::HOSTNAME_LENGTH + 1];
  int type;
  
  if (!getNetwork ()->open (this, &address, name, &type, 0))
    return 0;

  insist (getNetwork ()->getGraph ()->getHost (x, y, 0) == getNode ());
  
  send ();
  
  return 1;
  exception: return 0;
}

void gr_Job::receive (int event, char*p, int length)
{
  insist (this);

  switch (event)
  {
    case mt_Network::RECEIVE:
      {
	insist ((unsigned)length >= sizeof (gr_Message));
	gr_Message*m;
	m = (gr_Message*) p;
	m->swap ();
      
	insist (m->type >=0 && m->type < NUM_TYPES);

	typeReceives [m->type]++;
	
	if (m->type == DATA)
	{
	  insist (m->nx == x && m->ny == y && m->nz == z);
      
	  int position = getPosition (m->x, m->y, m->z);
	  insist (position >= 0 && position < NUM_POSITIONS);
      
	  //insist (isEdgeNode (m->x, m->y, m->z) || !receives [position]);
	  receives [position] = 1;
	  temperatures [position] = m->temperature;
      
	  numReceives++;
      
	  if (isEdgeNode (x, y, 0))
	    send ();
	  else if (ready ())
	  {
	    compute ();
	    mt_Node*h;
	    h = getHotHost ();
	    if (h)
	      send (h);
	    send ();
	    forget ();
	    cycle++;
	  }
	}
	else
	{
	  numReceives++;
	}
	break;
      }
    case mt_Network::SEND_DONE:
      //printFormat ("%s got a send done", getNode ()->getName ());      
      break;
    case mt_Network::TIMEOUT:
      break;
    default:
      insist (0);
  }
  exception: return;
}


int gr_Job::getIndex (int x, int y, int z)
{
  insist (this);
  
  return getNetwork ()->getGraph ()->getHost (x, y, z)->getTypeIndex ();

  exception: return 0;
}

int gr_Job::getPosition (int nx, int ny, int)
{
  insist (this);
  insist (nx >= 0 && nx <= side);
  insist (ny >= 0 && ny <= side);
  
  if (nx == x - 1)
    return UP;
  if (nx == x + 1)
    return DOWN;
  if (ny == y - 1)
    return LEFT;
  if (ny == y + 1)
    return RIGHT;

  exception: return -1;
}

int gr_Job::send ()
{
  insist (this);
  
  for (int i = 0; i < NUM_POSITIONS; i++)
    if (shouldSend (i) && !send (i))
      return 0;
    
  return 1;
  exception: return 0;
}

int gr_Job::getNeighbor (int position, int*x, int*y, int*z)
{
  insist (this);
  insist (x && y && z);
  
  *x = this->x;
  *y = this->y;
  *z = this->z;
  
  switch (position)
  {
    case UP:
      (*x)--;
      break;
    case DOWN:
      (*x)++;
      break;
    case LEFT:
      (*y)--;
      break;
    case RIGHT:
      (*y)++;
      break;
    default:
      insist (0);
      break;
  }
  return 1;
  exception: return 0;
}

mt_Node*gr_Job::getHotHost ()
{
  
  insist (this);
  int period, mod;
  period = simulation->getPeriod ();
  mod = simulation->getMod ();

  if (cycle == 0 || period == 0) return 0;

  if (((cycle % period) == 0) && ((cycle / side) % mod == ((x + side * y) % mod)))
  {
    int h = (2 * ((cycle / period) + 1) * (side / 2)) % (side * side);
    insist (h >= 0 && h <  getNetwork ()->getGraph ()->getNumHosts ());
    return getNetwork ()->getGraph ()->getHost (h);
  }
  
  exception: return 0;
}

int gr_Job::shouldSend (int position)
{
  insist (this);
  int ny, nx, nz;
  
  if (!getNeighbor (position, &nx, &ny, &nz))
    return 0;
  
  if (nx < 0 || nx >= side || ny < 0 || ny >= side || nz < 0 || nz >= side)
    return 0;

  if (isEdgeNode (x, y, z) && isEdgeNode (nx, ny, nz))
    return 0;

  return 1;
  exception: return 0;
}


int gr_Job::send (int position)
{
  mt_Route route;
  mt_Node*other;
  int in;
  gr_Message*m = new gr_Message;
  insist (m);
  
  insist (this);
  insist (position >= 0 && position < NUM_POSITIONS);
  
  getNeighbor (position, &m->nx, &m->ny, &m->nz);
  
  insist (m->nx >= 0 && m->nx < side && m->ny >= 0 && m->ny < side && m->nz >= 0 && m->nz < side);
  
  m->type = DATA;
  m->x = x;
  m->y = y;
  m->z = z;  
  
  m->temperature = temperature;

  int r;
  int me, him;
  me = getIndex (x, y, z);
  him = getIndex (m->nx, m->ny, m->nz);
  
  r = getNetwork ()->getRoute (me, him, mt_Component::rand (getNetwork()->getNumRoutes (me, him)), &route);
  insist (getNode ()->follow (&route, &other, &in) && other == getNetwork ()->getGraph ()->getHost (him));

  insist (r);
  
  int length;
  length = simulation->getPacketLength ();
  if (length < (int) sizeof (gr_Message))
    length = sizeof (gr_Message);

  m->swap ();

  //printFormat ("%s sending to %s", getNode ()->getName (), other->getName ());
  
  r = getNetwork ()->sendWithPadding (this, &route, (char*) m, sizeof (*m), length);
  insist (r);
  
  numSends++;
  typeSends [DATA]++;
  delete m;
  
  return 1;
  exception: return 0;
}


int gr_Job::send (mt_Node*host)
{
  mt_Route route;
  mt_Node*other;
  int in;
  gr_Message*m = new gr_Message;
  insist (m);
  
  insist (this);

  m->type = HOTSPOT;
  m->x = x;
  m->y = y;
  m->z = z;

  int r;
  int me, him;
  me = getIndex (x, y, z);
  him = host->getTypeIndex ();
  
  r = getNetwork ()->getRoute (me, him, mt_Component::rand (getNetwork()->getNumRoutes (me, him)), &route);
  insist (getNode ()->follow (&route, &other, &in) && other == getNetwork ()->getGraph ()->getHost (him));

  insist (r);
  
  int length;
  length = simulation->getPacketLength ();
  if (length < (int) sizeof (gr_Message))
    length = sizeof (gr_Message);

  m->swap ();

  //printFormat ("%s sending to %s", getNode ()->getName (), other->getName ());
  
  r = getNetwork ()->sendWithPadding (this, &route, (char*) m, sizeof (*m), length);
  insist (r);
  
  numSends++;
  typeSends [HOTSPOT]++;
  delete m;
  
  return 1;
  exception: return 0;
}



void gr_Job::dump (FILE*fp)
{
  insist (this);
  insist (fp);

  fprintf (fp, "%s %d/%d: (%d / %d) %5.4f\n", getNode ()->getName (), typeReceives [DATA],
	   typeSends [DATA], typeReceives [HOTSPOT], typeSends [HOTSPOT],  temperature);
  
  exception: return;
}

int gr_Job::willWait ()
{
  return getNode ()->getTypeIndex () == 0;
}


void gr_Job::wait ()
{
  char*p;
  int len;
  
  getNetwork ()->wait (&p, &len);
}

double gr_Simulation::getStartTemperature (int x, int y, int)
{
  insist (this);
  
  if (x == 0)
    return top;
  if (x == side - 1)
    return bottom;
  if (y == 0)
    return left;
  if (y == side - 1)
    return right;
  
  exception: return 0;
}


int gr_Simulation::getSide ()
{
  return side;
}

int gr_Simulation::getMod ()
{
  return mod;
}

int gr_Simulation::getPeriod ()
{
  return period;
}

int gr_Simulation::getPacketLength ()
{
  return packetLength;
}



mt_Job*gr_Simulation::newJob (mt_Node*node, mt_Network*network, mt_Calculator*)
{  
  insist (this);
  insist (node && node->isHost ());
  insist (network);
  
  return new gr_Job (node, network, this);
  exception: return 0;
}

void gr_Simulation::usage ()
{
  printFormat ("-simulation-args:");

  printFormat ("-side <number>");
  printFormat ("-packet-length <length>");
  printFormat ("-edge-temperatures <top> <bottom> <right> <left>");
  printFormat ("-period <integer>");
  printFormat ("-mod <integer>");
}

gr_Simulation::gr_Simulation ()
{
  side = 0;
}

void gr_Simulation::simpleDump (mt_Network*network, FILE *fp)
{
  insist (fp);
  insist (this);
  insist (network);
  
  fprintf (fp, "side is %d\n", side);
  fprintf (fp, "top is %f\n", top);
  fprintf (fp, "bottom is %f\n", bottom);
  fprintf (fp, "right is %f\n", right);
  fprintf (fp, "left is %f\n", left);
  fprintf (fp, "period is %d\n", period);
  fprintf (fp, "mod is %d\n", mod);

  for (int i = 0; i < side; i++)
  {
    for (int j = 0; j < side; j++)
    {
      fprintf (fp, "%7.2f ", ((gr_Job*)((sm_Host*) network->getGraph ()->getHost (i, j))->getJob ())->getTemperature ());
    }
    fprintf (fp, "\n");
  }
  
  exception: return;
}

int gr_Simulation::parseArgs (mt_Args*args)
{
  int argc;  
  char**argv = args->getArgs (mt_Args::SIMULATION, &argc);
  
  side = 0;
  top = bottom = right = left = 0;
  packetLength = sizeof (gr_Message);
  period = 0;
  mod = 1;
  
  for (int i = 0; i < argc; i++)
  {
    insist (argv[i]);
    if (!strcmp (argv[i], "-side"))
    {
      if (++i == argc)
      {
	printFormat ("gr_Simulator::parseArgs: missing side");
	return 0;
      }
      side = atoi (argv[i]);
      if (side <= 0)
      {
	printFormat ("gr_Simulator::parseArgs: %s is a bad side", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-packet-length"))
    {
      if (++i == argc)
      {
	printFormat ("gr_Simulator::parseArgs: missing packet length");
	return 0;
      }
      packetLength = atoi (argv[i]);
      if (packetLength <= 0)
      {
	printFormat ("gr_Simulator::parseArgs: %s is a bad packet length", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-period"))
    {
      if (++i == argc)
      {
	printFormat ("gr_Simulator::parseArgs: missing period");
	return 0;
      }
      period = atoi (argv[i]);
      if (period <= 0)
      {
	printFormat ("gr_Simulator::parseArgs: %s is a bad packet length", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-mod"))
    {
      if (++i == argc)
      {
	printFormat ("gr_Simulator::parseArgs: missing hotspot mod");
	return 0;
      }
      mod = atoi (argv[i]);
      if (mod <= 0)
      {
	printFormat ("gr_Simulator::parseArgs: %s is a bad mod", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-edge-temperatures"))
    {
      if (++i > argc - 4)
      {
	printFormat ("gr_Simulator::parseArgs: missing edge temperatures");
	return 0;
      }
      if (sscanf (argv[i++], "%lf", &top) != 1 ||
	  sscanf (argv[i++], "%lf", &bottom) != 1 ||
	  sscanf (argv[i++], "%lf", &right) != 1 ||
	  sscanf (argv[i], "%lf", &left) != 1)
      {
	printFormat ("gr_Simulator::parseArgs: is a bad temperatures");
	return 0;
      }
    }
    else
    {
      printFormat ("gr_Simulator::parseArgs: bad argument %s", argv [i]);
      return 0;
    }
  }

  if (!side)
  {
    printFormat ("gr_Simulator::parseArgs: must specify side");
    return 0;
  }

  return 1;

  exception:
  return 0;
}
