/*
  bs_Simulation.c
  barrel simulation
  finucane@myri.com (David Finucane)
*/

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "insist.h"
#include "bs_Simulation.h"
#include "sm_Host.h"


class bs_Message
{
  public:
  int number;
  int from;
  int to;
  mt_Int64 startTime;
};

bs_Job::~bs_Job ()
{
  if (receives) delete [] receives;
  if (sends) delete [] sends;
  if (hosts) delete hosts;
}

bs_Job::bs_Job (mt_Node*node, mt_Network*network, bs_Simulation *sim)
  : mt_Job (node, network)
{
  receives = 0;
  sends = 0;
  
  this->sim = sim;
  
  mt_Graph*g = getNetwork ()->getGraph ();
  insist (g);

  numHosts = g->getNumHosts ();
  insist (numHosts > 0);

  me = node->getRouteIndex ();
  insist (me >= 0 && me < numHosts);
  
  receives = new int [numHosts];
  sends = new int [numHosts];

  insistp (receives && sends, ("bs_Job::bs_Job: alloc failed"));

  memset (receives, 0, numHosts);
  memset (sends, 0, numHosts);
  hosts = 0;  
  phase = 0;
  
  exception: return;
}

int bs_Job::start ()
{
  numSends = 0;
  numReceives = 0;
  to = 0;
  mt_Address address;
  char name [mt_Network::HOSTNAME_LENGTH + 1];
  int type;
  
  if (!getNetwork ()->open (this, &address, name, &type, 0))
    return 0;

  this->sendAfterDelay ();
  
  return 1;
}

void bs_Job::receive (int event, char*p, int length)
{
  insist (this);

  switch (event)
  {
    case mt_Network::RECEIVE:
    {
      insist (!getNeverReceive (me));
      
      insist ((unsigned)length >= sizeof (bs_Message));
      bs_Message*m;
      m = (bs_Message*) p;
      insist (m->to == me);
      insist (m->to >= 0 && m->to < numHosts);

      numReceives++;
      receives [m->from]++;
      break;
    }
    case mt_Network::SEND_DONE:
      sendAfterDelay ();
      break;
    case mt_Network::TIMEOUT:
      send ();
      break;
    default:
      insist (0);
  }
  exception: return;
}

void bs_Job::sendAfterDelay()
{
  int delay;

  delay = sim->delay;
  
  insist (delay >= 0);

  if (delay > 0)
    getNetwork()->setTimer (this, delay);
  else
    send();

  exception: return;
}

int bs_Job:: getNeverSend (int i)
{
  
  mt_Graph*g = getNetwork ()->getGraph ();
  insist (g);

  insist (numHosts == g->getNumHosts ());
  
  insist (i >= 0 && i < numHosts);
  return ((sm_Host*) g->getHost (i))->getNeverSend ();
  
  exception: return 0;
}

int bs_Job:: getNeverReceive (int i)
{
  
  mt_Graph*g = getNetwork ()->getGraph ();
  insist (g);

  insist (numHosts == g->getNumHosts ());
  
  insist (i >= 0 && i < numHosts);
  return ((sm_Host*) g->getHost (i))->getNeverReceive ();
  
  exception: return 0;
}

int bs_Job::getNextDestination (int*to, int*delay)
{
  insist (this);
  insist (to && delay);
  char s [mt_Node::NAME_LENGTH];
  
  *delay = 0;
  
  mt_Graph*g;
  g = getNetwork ()->getGraph ();
  insist (g);

  int number;
  number = getNode ()->number;  

  *s = 0;
  switch (sim->traffic)
  {
    case 1:
      sprintf (s, "rcv%d", number);
      break;
    case 2:
      sprintf (s, "rcv%d", phase & 255);
      break;
    case 3:
      if (phase >= number)
	return 0;
      sprintf (s, "rcv%d", (number - phase) & 255);
      break;
    case 4:
    {
      int k = number - phase;
      if (k <= 0) return 0;
      sprintf (s, "rcv%d", 8 * (k & 31) + (k / 32));
    }
    break;
    default:
      insist (0);
  }

  insist (g->getNode (s));
  
  *to = g->getNode (s)->getTypeIndex ();
  insist (*to);

  insist (*to >=0 && *to < numHosts);
  insist (!getNeverReceive (*to));
 
  phase++;
  return 1;
  exception: return 0;
}



void bs_Job::send ()
{
  mt_Route route;
  int delay;
  int to;
  
  insist (this);

  if (getNeverSend (me) || numHosts == 1 && getNeverReceive (me))
    return;
  
  if (!getNextDestination (&to, &delay))
    return;

  if (delay > 0)
  {
    getNetwork()->setTimer (this, delay);
    return;
  }
  
  bs_Message m;
  mt_Network*n;
  n = getNetwork ();
  insist (n);

  int r;
  r = n->getRoute (me, to, 0, &route);
  insist (r);
  insist (route.getLength () > 0);
  
  m.to = to;
  m.from = me;
  m.number = numSends++;
  m.startTime = n->getCurrentTime ();
  insist (m.startTime >= 0);

  r = n->sendWithPadding (this, &route, (char*) &m, sizeof (m), sim->packetLength < (int) sizeof (m) ? sizeof (m) : sim->packetLength);
  insist (r);

  exception: return;
}

void bs_Job::dump (FILE*fp)
{
  insist (this);
  insist (fp);

  //fprintf (fp, "%s %d/%d\n", getNode()->getName (), numSends, numReceives);
  
  exception: return;
}

int bs_Job::willWait ()
{
  return 0;
}

mt_Job*bs_Simulation::newJob (mt_Node*node, mt_Network*network, mt_Calculator*)
{  
  insist (this);
  insist (node && node->isHost ());
  insist (network);
  
  return new bs_Job (node, network, this);
  exception: return 0;
}
void bs_Simulation::usage ()
{
  printFormat ("-simulation-args:");
  printFormat ("-packet-length <bytes>");
  printFormat ("-delay <microseconds>");
  printFormat ("-traffic <1,2,3,4>");
}

bs_Simulation::bs_Simulation()
{
  packetLength = 100;
  delay = 0;
  traffic = 1;
}

void bs_Simulation::dump ()
{
}

void bs_Simulation::simpleDump (FILE *fp)
{
  fprintf(fp, "packet length: %d\n",packetLength);
  fprintf(fp, "delay between packets: %d microseconds\n", delay);  
}

int bs_Simulation::parseArgs (mt_Args*args)
{
  int argc;  
  char**argv = args->getArgs (mt_Args::SIMULATION, &argc);

  for (int i = 0; i < argc; i++)
  {
    insist (argv[i]);
    if (!strcmp (argv[i], "-packet-length"))
    {
      if (++i == argc)
      {
	printFormat ("bs_Simulation::parseArgs: missing packet length");
	return 0;
      }
      packetLength = atoi (argv[i]);
      if (packetLength <= 0)
      {
	printFormat ("bs_Simulation::parseArgs: %s is a bad packet length", argv[i]);
	return 0;
      }
    }
    else if (!strcmp (argv[i], "-delay"))
    {
      if (++i == argc)
      {
	printFormat ("bs_Simulation::parseArgs: missing delay");
	return 0;
      }
      delay = atoi (argv[i]);
      if (delay < 0)
      {
	printFormat ("bs_Simulation::parseArgs: %d is a bad number for delay", delay);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-traffic"))
    {
      if (++i == argc)
      {
	printFormat ("bs_Simulation::parseArgs: missing traffic");
	return 0;
      }
      traffic = atoi (argv[i]);
      if (traffic < 1 || traffic > 4)
      {
	printFormat ("bs_Simulation::parseArgs: %d is a bad traffic number", delay);
	return 0;
      }
    }
    else
    {
      printFormat ("bs_Simulation::parseArgs: bad argument %s", argv [i]);
      return 0;
    }
  }
  return 1;

  exception:
  return 0;
}









