/*
  ac_Calculator.c
  simple calculator
  finucane@myri.com (David Finucane)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "insist.h"

#include "ac_Calculator.h"
#include "sc_Calculator.h"
#include "mt_Switch.h"
#include "mt_Host.h"

ac_Calculator::ac_Calculator () : mt_Calculator ()
{
  algorithm = 0;
  initializeFunction = mt_fp (ac_Calculator::none);
  rowColumnTable = 0;
  for (int i=0; i<LX; i++) V[i] = C[i] = B[0][i] = 0;
}

ac_Calculator::~ac_Calculator ()
{
  if (rowColumnTable)
    delete rowColumnTable;
}

int ac_Calculator::none ()
{
  return 1;
}

void ac_Calculator::cleanup ()
{
  mt_Calculator::cleanup ();
}

int ac_Calculator::initialize (mt_Graph*graph, mt_Node*root)
{
  return mt_Calculator::initialize (graph, root) && (this->*initializeFunction) ();
}

int ac_Calculator::initialize (char*mapFile)
{
  return mt_Calculator::initialize (mapFile) && (this->*initializeFunction) ();
}

mt_Node*ac_Calculator::newNode (int nodeType, char*name, char*type)
{
  mt_Route r;
  
  insist (this);
  
  switch (nodeType)
  {
    case mt_Node::HOST:
      return new mt_Host (name, type);
    case mt_Node::SWITCH:
      return new mt_Switch (name, type);
    default:
      insist (0);
  }
  exception: return 0;
}

int ac_Calculator::getMaxRoutes ()
{
  insist (this);
  return 1;

  exception: return 0;
}

int ac_Calculator::getNumRoutes (int, int)
{
  insist (this);

  return 1;

  exception: return 0;
}

int ac_Calculator::step (mt_Node*node, int)
{
  insist (node);
  insist (numCollectedSwitches >= 0 && numCollectedSwitches < mt_Route::MAX_ROUTE);
  insist (collectedSwitches);
  
  if (node->isSwitch ())
    collectedSwitches [numCollectedSwitches++] = node;
  return 1;
  
  exception: return 0;
}

int ac_Calculator::getSwitches (mt_Route*route, mt_Node*from, mt_Node*to, mt_Node*switches [])
{  
  mt_Node*n;
  int in;  

  collectedSwitches = switches;
  numCollectedSwitches = 0;
  
  from->follow (route, &n, &in, this);
  insist (n == to);
  return numCollectedSwitches;
  exception: return 0;
}
  
int ac_Calculator::getRoute (mt_Route*route, mt_Node*from, mt_Node*to, mt_Node*switches[], int numSwitches, int*ports)
{
  char hops [mt_Route::MAX_ROUTE];
  int numHops = 0;
  
  insist (this);
  insist (route && from && to && switches);
  insist (numSwitches >= 0);

  mt_Node*prev;
  prev = from->getNode (0);
  int in;
  in = from->getOpposite (0);
  
  for (int i = 1; i < numSwitches; i++)
  {
    int p = prev->getPort (switches [i], ports ? ports [i - 1] : -1);
    insist (p >= 0 && p < prev->getMaxNodes ());

    insist (numHops < mt_Route::MAX_ROUTE);
    
    hops [numHops++] = p - in;
    in = prev->getOpposite (p);
    prev = switches [i];
  }

  insist (numHops < mt_Route::MAX_ROUTE);
  
  int p;
  p = prev->getPort (to, ports ? ports [numSwitches - 1] : -1);
  insist (p >= 0 && p < prev->getMaxNodes ());
  hops [numHops++] = p - in;

  route->fromBytes (hops, numHops);

  insist (((mt_Node*)from)->getNode (route) == to);
  return 1;
  
  exception: return 0;
}

int ac_Calculator::getRoute (int from, int to, int routeIndex, mt_Route*route)  
{
  insist (this);
  insist (from >= 0 && from < getNumHosts ());
  insist (to >= 0 && to < getNumHosts ());
  insist (routeIndex == 0);
  insist (route);
  insist (algorithm);

  mt_Node*h1, *h2;
  h1 = getHost (from);
  h2 = getHost (to);
  insist (h1 && h2);
  insist (h1->isHost () && h2->isHost ());
  
  return (this->*algorithm) (h1, h2, route);
  exception: return 0;
}

int ac_Calculator::valid (int s)
{
  int i,j,dd,t ;
  t = s ;
  for (i=1; i<LX; i++) if (C[i]!=0)
  {
    dd = ( (C[i]>0) ? V[i] : -V[i] ) ;
    for (j=0; j<abs(C[i]); j++) if ((t=((t+dd+T)%T))==MN) return 0;
  }
  return 1;
} // valid

void ac_Calculator::update_route (int s)
{
  int i,xl ;

  if (!valid(s)) return ;
  xl = 0 ; 
  for (i=1; i<LX; i++) xl += abs(C[i]) ;
  if (xl<L)
  {
    for (i=1; i<LX; i++) B[0][i] = C[i] ;
    L = xl ;
    cbest = 1;
  }
  else
  {
    for (i=1; i< LX; i++)
      B[cbest][i] = C[i];
    cbest++;
  } 
}// update_route


void ac_Calculator::tryD (int s, int D,int Level)
{
  int i,t,v;
  t = D;
  for (i=LX; i>Level; i--) { t -= C[i]*V[i];}

  if (V[Level]>0)
  {
    v = ( (t>0) ? (t/V[Level]) : -((-t)/V[Level]) );
    if (Level==1)
    {
      C[Level] = v;
      update_route (s);
      return;
    }
    for (i=-R; i<=R; i++)
    {
      C[Level] = v + i;
      tryD(s,D,Level-1);
    }
  } 
  else // if V[Level]==0
  {
    C[Level] = 0;
    tryD(s,D,Level-1);
  }
}


// get routes (with chords)
int ac_Calculator::get_routes (int s, int d, mt_Node*switches [])
{

  int i,D;
  int numSwitches = 0;
  L = 999;
  if ((D=d-s)==0) 
  {
    for (i=0; i<LX; i++) B[0][i] = 0;
    return 0;
  }
  tryD(s,D,LX-3);
  if (D>0) tryD(s,D-T,LX-3);
  if (D<0) tryD(s,D+T,LX-3);
  SL += L;
  NL++;

  numSwitches = 0;
  
  int xl,dd,k,t;
  t = s;
  xl = 0;

  insist (cbest >= 1);
  k = ::rand () % cbest;
  
  for (i=1; i<LX; i++) xl += abs(B[k][i]);

  
  for (i=1; i<LX; i++) if (B[k][i]!=0)
  {
    dd = B[k][i] > 0 ? V[i] : -V[i];
    
    for (int j=0; j< abs (B[k][i]); j++, numSwitches++)
    {
      insist (numSwitches < mt_Route::MAX_ROUTE);
      switches [numSwitches] = getNumberedSwitch (t=((t+dd+T)%T));
      insist (switches [numSwitches]);
    }
  }
  return numSwitches;
  exception: return 0;
}



int ac_Calculator::initialize_spiral ()
{
  insist (this);

  L = SL = NL = 0;
  
  M = numColumns;
  N = numRows;
  MN = M*N;
  T = MN + 1;
  cbest = 0;

  insist (V[1] == 1);

  for (int i = 1; i < LX; i++)
  {
    insist (!(V[i] > M) || V[i] % M == 0);
  }
  
  return 1;
  exception: return 0;
}

int ac_Calculator::algorithm_spiral (mt_Node*h1, mt_Node*h2, mt_Route*route)
{
  insist (this);
  insist (h1 && h2 && route);

  initialize_spiral ();
  
  mt_Node*switches [mt_Route::MAX_ROUTE];
  int numSwitches;
  numSwitches = 0;

  mt_Node*s1, *s2;
  
  s1 = h1->getNode (0);
  s2 = h2->getNode (0);
  insist (s1 && s2);

  switches [numSwitches++] = s1;

  if (s1->number != s2->number)
    numSwitches += get_routes (s1->number, s2->number, &switches [1]);

  insist (numSwitches > 0 && numSwitches <= mt_Route::MAX_ROUTE);
  return getRoute (route, h1, h2, switches, numSwitches);

  exception: return 0;
}

int ac_Calculator::algorithm_8x8 (mt_Node*h1, mt_Node*h2, mt_Route*route)
{
  insist (this);
  insist (h1 && h2 && route);

  mt_Node*switches [mt_Route::MAX_ROUTE]; 
  int numSwitches;
  numSwitches = 0;

  mt_Node*s1, *s2;
  
  s1 = h1->getNode (0);
  s2 = h2->getNode (0);
  insist (s1 && s2);
  
  int x1, y1, x2, y2;
  
  x1 = s1->getMeshX ();
  y1 = s1->getMeshY ();
  x2 = s2->getMeshX ();
  y2 = s2->getMeshY ();

  insist (x1 >= 0 && y1 >= 0);
  insist (x2 >= 0 && y2 >= 0);

  if (s1 == s2)
    switches [numSwitches++] = s1;
  
  if (x1 != x2)
  {
    switches [numSwitches++] =  getSwitch (x1, y1);
    if (x1 % 2 == x2 % 2)
    {
      if (x1 < x2)
	switches [numSwitches++] =  getSwitch (x2 - 1, y1);
      else
	switches [numSwitches++] =  getSwitch (x2 + 1, y1);
    }
    switches [numSwitches++] = getSwitch (x2, y1);
  }
  
  if (y1 != y2)
  {
    if (numSwitches == 0)
      switches [numSwitches++] =  getSwitch (x2, y1);
    
    if (y1 % 2 == y2 % 2)
    {
      if (y1 < y2)
	switches [numSwitches++] =  getSwitch (x2, y2  - 1);
      else
	switches [numSwitches++] =  getSwitch (x2, y2 + 1);
    }
    switches [numSwitches++] =  getSwitch (x2, y2);
  }
  return getRoute (route, h1, h2, switches, numSwitches);
  
  exception: return 0;
}

int ac_Calculator::algorithm_3_level (mt_Node*h1, mt_Node*h2, mt_Route*route)
{
  insist (this);
  insist (h1 && h2 && route);
  
  mt_Node*s1, *s2;
  
  s1 = h1->getNode (0);
  s2 = h2->getNode (0);
  insist (s1 && s2);
  
  int k1, j1, i1;
  int k2, j2, i2;

  insist (h1->number != h2->number || h1 == h2);
  
  k1 = (h1->number >> 6) & 0xf;
  k2 = (h2->number >> 6) & 0xf;
  j1 = (h1->number >> 3) & 0x7;
  j2 = (h2->number >> 3) & 0x7;
  i1 = h1->number & 0x7;
  i2 = h2->number & 0x7;

  insist (8 * (8 * k1 + j1) + i1 == h1->number);
  insist (8 * (8 * k2 + j2) + i2 == h2->number);

  int numHops;
  numHops = 0;
  char hops [mt_Route::MAX_ROUTE];
  
  if (k1 == k2)
  {
    if (j1 == j2)
      hops [numHops++] = i2;
    else
    {
      hops [numHops++] = i1 + 8;
      hops [numHops++] = j2;
      hops [numHops++] = i2;
    }
  }
  else
  {
    hops [numHops++] = i1 + 8;
    hops [numHops++] = j1 + 8;
    hops [numHops++] = k2;
    hops [numHops++] = j2;
    hops [numHops++] = i2;
  }

  mt_Node*n;
  n = h1->absoluteToRelative (numHops, hops);
  insist (n == h2);
  route->fromBytes (hops, numHops);
  return 1;
  
  exception: return 0;
}

static int switchesAreInDifferentRows (mt_Node*n1, mt_Node*n2, int)
{
  insistf (n1 && n2);
  return n1->isSwitch () && n2->isSwitch () && n1->getMeshY () != n2->getMeshY ();
  exception: return 0;
}

static int switchesAreInDifferentColumns (mt_Node*n1, mt_Node*n2, int)
{
  insistf (n1 && n2);
  return n1->isSwitch () && n2->isSwitch () && n1->getMeshX () != n2->getMeshX ();
  exception: return 0;
}

int ac_Calculator::initialize_rowColumn ()
{
  sc_Calculator simple;
  mt_Args args (mt_Args::CALCULATOR, 5);
  
  insist (this);
  insist (rowColumnTable);
  insist (numRows && numColumns);

  args.addArg ("-use-numbers");
  args.addArg ("-margin"); args.addArg ("1");
  args.addArg ("-weight"); args.addArg ("1");

  if (!simple.parseArgs (&args))
    return 0;
  
  if (!simple.initialize (this, 0))
    return 0;

  if (!simple.removeLinks (switchesAreInDifferentRows))
    return 0;
  
  char hops [mt_Route::MAX_ROUTE];
    
  for (int i = 0; i < numRows; i++)
  {
    for (int j = 0; j < numColumns; j++)
    {
      mt_Node*s1, *h1;
      s1 = getSwitch (j, i);
      insist (s1);
      
      for (int k = 0; k < numColumns; k++)
      {
	mt_Node*s2,*h2;

	s2 = getSwitch (k, i);
	insist (s2);

	h1 = s1->getAnyHost ();
	insist (h1);
      	
	h2 = s2->getAnyHost ();
	insist (h2);

	if (s1->getMeshY () == s2->getMeshY () && s1->getMeshX () != s2->getMeshX ())
	{
	  mt_Route r;
	  if (!simple.getRoute (h1->getTypeIndex (), h2->getTypeIndex (), 0, &r))
	    return 0;

	  insist (r.getLength () > 1);
	  r.copyOut (hops);
	  
	  if (!h1->relativeToAbsolute (r.getLength (), hops))
	    return 0;
	  
	  if (!rowColumnTable->setRowRoute (i, s1->getMeshX (), s2->getMeshX (), hops, r.getLength ()))
	    return 0;
	}
      }
    }
  }
  simple.cleanup ();
  
  if (!simple.initialize (this, 0))
    return 0;

  if (!simple.removeLinks (switchesAreInDifferentColumns))
    return 0;
  
  for (int i = 0; i < numColumns; i++)
  {
    for (int j = 0; j < numRows; j++)
    {
      mt_Node*s1, *h1;
      s1 = getSwitch (i, j);
      insist (s1);

      
      for (int k = 0; k < numRows; k++)
      {
	mt_Node*s2,*h2;

	s2 = getSwitch (i, k);
	insist (s2);

	h1 = s1->getAnyHost ();
	insist (h1);

	h2 = s2->getAnyHost ();
	insist (h2);
	
	if (s1->getMeshX () == s2->getMeshX () && s1->getMeshY () != s2->getMeshY ())
	{
	  mt_Route r;
	  if (!simple.getRoute (h1->getTypeIndex (), h2->getTypeIndex (), 0, &r))
	    return 0;

	  insist (r.getLength () > 1);
	  r.copyOut (hops);
	  
	  if (!h1->relativeToAbsolute (r.getLength (), hops))
	    return 0;
	  
	  if (!rowColumnTable->setColumnRoute (i, s1->getMeshY (), s2->getMeshY (), hops, r.getLength ()))
	    return 0;
	}
      }
    }
  }

  simple.cleanup ();
  return 1;
  
  exception: return 0;
}

int ac_Calculator::algorithm_rowColumn (mt_Node*h1, mt_Node*h2, mt_Route*route)
{
  insist (this);
  insist (h1 && h2 && route);
  insist (numRows && numColumns);
  insist (rowColumnTable);

  mt_Node*s1, *s2;
  
  s1 = h1->getNode (0);
  s2 = h2->getNode (0);
  insist (s1 && s2);
  
  int x1, y1, x2, y2;
  
  x1 = s1->getMeshX ();
  y1 = s1->getMeshY ();
  x2 = s2->getMeshX ();
  y2 = s2->getMeshY ();

  insist (x1 >= 0 && y1 >= 0);
  insist (x2 >= 0 && y2 >= 0);

  int numHops;
  numHops = 0;
  char hops [mt_Route::MAX_ROUTE];
  
  if (x1 == x2 && y1 == y2)
  {
    insist (s1 == s2);
    return getRoute (route, h1, h2, &s1, 1);
  }
  
  if (x1 != x2 && y1 != y2)
  {
    mt_Route*r1 = rowColumnTable->getRowRoute (y1, x1, x2);
    mt_Route*r2 = rowColumnTable->getColumnRoute (x2, y1, y2);

    insist (r1 && r2);
    insist (r1->getLength ());
    insist (r2->getLength ());
    
    r1->copyOut (hops);
    r2->copyOut (hops + r1->getLength () - 1);
    numHops =  r1->getLength () + r2->getLength () - 1;
  }
  else if (x1 == x2)
  {
    mt_Route*r = rowColumnTable->getColumnRoute (x2, y1, y2);
    insist (r);
    r->copyOut (hops);
    numHops =  r->getLength ();
  }
  else
  {
    insist (y1 == y2);
    
    mt_Route*r = rowColumnTable->getRowRoute (y2, x1, x2);
    insist (r);
    r->copyOut (hops);
    numHops = r->getLength ();
  }
  hops [numHops -1] = h2->getOpposite (0);
  
  mt_Node*n;
  n = h1->absoluteToRelative (numHops, hops);
  insist (n == h2);
  route->fromBytes (hops, numHops);

  mt_Node*switches [mt_Route::MAX_ROUTE];

  return getRoute (route, h1, h2, switches, getSwitches (route, h1, h2, switches));
  
  return 1;
  exception: return 0;
}

void ac_Calculator::usage ()
{
  printFormat
  (
   "route-args:\n"
   "-8x8\n"
   "-3-level\n"
   "-row-column\n"
   "-spiral\n"
   "-num-rows <number>\n"
   "-num-columns <number>\n"
   "-steps <num steps> <step> <step> <step> ...\n"
   "-barrel\n"
   );
}

int ac_Calculator::parseArgs (mt_Args*args)
{
  insist (args);
  
  int argc;  
  char**argv;
  argv = args->getArgs (mt_Args::CALCULATOR, &argc);

  algorithm = 0;
  numRows = numColumns = 0;
  
  for (int i = 0; i < argc; i++)
  {
    insist (argv[i]);
    
    if (!strcmp (argv [i], "-8x8"))
      algorithm = mt_fp (ac_Calculator::algorithm_8x8);
    else if (!strcmp (argv [i], "-3-level"))
      algorithm = mt_fp (ac_Calculator::algorithm_3_level);
    else if (!strcmp (argv [i], "-spiral"))
    {
      algorithm = mt_fp (ac_Calculator::algorithm_spiral);
      initializeFunction = mt_fp (ac_Calculator::initialize_spiral);
    }
    else if (!strcmp (argv [i], "-row-column"))
    {
      algorithm = mt_fp (ac_Calculator::algorithm_rowColumn);
      initializeFunction = mt_fp (ac_Calculator::initialize_rowColumn);
    }
    else if (!strcmp (argv [i], "-barrel"))
    {
      algorithm = mt_fp (ac_Calculator::algorithm_barrel);
    }
    else if (!strcmp (argv [i], "-num-rows"))
    {
      if (++i == argc)
      {
	printFormat ("ac_Calculator::parseArgs: missing num rows");
	return 0;
      }
      numRows = atoi (argv [i]);
      if (numRows < 0)
      {
	printFormat ("ac_Calculator::parseArgs: %s is not a legal num rows number", argv [i]);
	return 0;
      }
    }
    else if (!strcmp (argv [i], "-num-columns"))
    {
      if (++i == argc)
      {
	printFormat ("ac_Calculator::parseArgs: missing num columns");
	return 0;
      }
      numColumns = atoi (argv [i]);
      if (numColumns < 0)
      {
	printFormat ("ac_Calculator::parseArgs: %s is not a legal num columns number", argv [i]);
	return 0;
      }
    }

    else if (!strcmp (argv [i], "-steps"))
    {
      if (++i == argc)
      {
	printFormat ("ac_Calculator::parseArgs: missing num steps");
	return 0;
      }
      int numSteps = atoi (argv [i]);

      if (numSteps < 0 || numSteps > LX)
      {
	printFormat ("ac_Calculator::parseArgs: bad num steps");
	return 0;
      }
      if (numSteps == 0)
	continue;
      
      for (int j = 0; j < numSteps; j++)
      {
	if (++i == argc)
	{
	  printFormat ("ac_Calculator::parseArgs: missing step");
	  return 0;
	}
	
	V[j + 1] = atoi (argv [i]);
	
	if (V [j + 1] < 0)
	{
	  printFormat ("ac_Calculator::parseArgs: %s is a bad step", argv [i]);
	  return 0;
	}
      }
    }
    else 
    {
      printFormat ("ac_Calculator::parseArgs: bad option \"%s\"", argv[i]);
      return 0;
    }
  }

#ifndef MT_intel_ms
  //microsoft compiler crashes here
  if (!algorithm)
  {
    printFormat ("ac_Calculator::parseArgs: no algorithm specified");
    return 0;
  }
#endif
  
  if (algorithm == mt_fp (ac_Calculator::algorithm_rowColumn) || algorithm == mt_fp (ac_Calculator::algorithm_spiral))
  {
    if (!numRows || !numColumns)
    {
      printFormat ("ac_Calculator::parseArgs: must specify -num-rows and -num-columns");
      return 0;
    }
    else rowColumnTable = new (numRows, numColumns) ac_RowColumnTable (numRows, numColumns);
    insistp (rowColumnTable, ("ac_Calculator::parseArgs: couldn't allocate row cloumn table"));
  }
  
  return 1;
  exception: return 0;
}

int ac_Calculator::algorithm_barrel (mt_Node*h1, mt_Node*h2, mt_Route*route)
{
  insist (this);
  insist (h1 && h2 && route);
  
  int ports [mt_Route::MAX_ROUTE];
  mt_Node*switches [mt_Route::MAX_ROUTE];
  int numSwitches;
  numSwitches = 0;

  mt_Node*s1, *s2;
  
  s1 = h1->getNode (0);
  s2 = h2->getNode (0);
  insist (s1 && s2);

  for (int i = 0; i < mt_Route::MAX_ROUTE; i++)
    ports [i] = -1;
  
  if (!strcmp (h1->getType (), "sender") && !strcmp (h2->getType (), "receiver"))
  {
    int s = h1->number;
    int d = h2->number;
    
    int x = s / 8;
    int z = d / 8;
    int y = (x & 24) + (z & 7);
    int p = ( (d/32) & 1);

    switches [numSwitches++] = getNumberedSwitch (0 + x);
    insist (switches [0] == s1);
    
    switches [numSwitches++] = getNumberedSwitch (32 + y);
    insist (switches [numSwitches - 1]);
    
    switches [numSwitches++] = getNumberedSwitch (64 + z);
    insist (switches [numSwitches - 1] == s2);

    ports [numSwitches -2] = p;
    
    return getRoute (route, h1, h2, switches, numSwitches, ports);
  }
  else if (!strcmp (h1->getType (), "sender") && !strcmp (h2->getType (), "sender"))
  {
    
    int ss1 = h1->number;
    int ss2 = h2->number;
    int x1 = ss1 / 8;
    int x2 = ss2 / 8;
    int y1 = x1;
    int z = y1;
    int y2 = (x2 & 24) +  (x1 & 7);
    
    switches [numSwitches++] = getNumberedSwitch (0 + x1);
    insist (switches [0] == s1);
    
    switches [numSwitches++] = getNumberedSwitch (32 + y1);
    insist (switches [numSwitches - 1]);
    
    switches [numSwitches++] = getNumberedSwitch (64 + z);
    insist (switches [numSwitches - 1]);

    switches [numSwitches++] = getNumberedSwitch (32 + y2);
    insist (switches [numSwitches - 1]);

    switches [numSwitches++] = getNumberedSwitch (0 + x2);
    insist (switches [numSwitches - 1] == s2);
  }
  else if (!strcmp (h1->getType (), "receiver") && !strcmp (h2->getType (), "receiver"))
  {
    int d1 = h1->number;
    int d2 = h2->number;
    int z1 = d1 / 8;
    int z2 = d2 / 8;
    int y1 = z1;
    int x =  y1;
    int y2 = (z1 & 24) + (z2 & 7);
    
    switches [numSwitches++] = getNumberedSwitch (64 + z1);
    insist (switches [0] == s1);
    
    switches [numSwitches++] = getNumberedSwitch (32 + y1);
    insist (switches [numSwitches - 1]);
    
    switches [numSwitches++] = getNumberedSwitch (0 + x);
    insist (switches [numSwitches - 1]);

    switches [numSwitches++] = getNumberedSwitch (32 + y2);
    insist (switches [numSwitches - 1]);

    switches [numSwitches++] = getNumberedSwitch (64 + z2);
    insist (switches [numSwitches - 1] == s2);
    
  }
  else if (!strcmp (h1->getType (), "receiver") && !strcmp (h2->getType (), "sender"))
  {
    int z = h1->number / 8;
    int x = h2->number / 8;
    int y = (x & 24) + (z & 7);
    
    switches [numSwitches++] = getNumberedSwitch (64 + z);
    insist (switches [0] == s1);
    
    switches [numSwitches++] = getNumberedSwitch (32 + y);
    insist (switches [numSwitches - 1]);
    
    switches [numSwitches++] = getNumberedSwitch (0 + x);
    insist (switches [numSwitches - 1] == s2);    
  }
  else
  {
    insist (0);
  }
  
  insist (numSwitches > 0 && numSwitches <= mt_Route::MAX_ROUTE);
  return getRoute (route, h1, h2, switches, numSwitches);

  exception: return 0;
}

