/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

#define GM_DEBUG_TIMER 0

/****************************************************************
 * Timer state machine
 ****************************************************************/

_MARK_LABEL (L_timer,)
{
  /************
   * Handle high-frequency timer events.
   ************/

  /* update the user-readable real-time clock at SRAM address 0. */

  *(gm_u64_t *) 0 = rtc64 ();

  if ((gm.timeout_time - RTC) <= 0)
    {
      /************
       * Handle low-frequency timer events.
       ************/

      /* If debugging, print something about once a second. */

      if (GM_DEBUG_TIMER)
	{
	  static gm_u32_t last_second;

	  if (last_second != (RTC & 0xffe00000))
	    {
	      fflush (stdout);
	      gm_printf ("A nominal second has passed.\n");
	      LOG_DISPATCH (0, "A nominal second has passed.");
	      print_state ();
	      fflush (stdout);
	      last_second = (RTC & 0xfff00000);
	      gm_always_assert (gm.event_index_table_crc
				== gm_crc (_gmp.event_index,
					   sizeof (_gmp.event_index)));
	    }
	}

      /* check for parity errors */

      gm_check_parity ();

      /* print resend_cnt, but only if it changed */

      if (0 && GM_ENABLE_ERROR_COUNTERS)
	{
	  static gm_u32_t resend_cnt;

	  if (resend_cnt != GM_READ_ERROR_CNT (gm.resend_cnt))
	    {
	      resend_cnt = GM_READ_ERROR_CNT (gm.resend_cnt);
	      gm_printf ("0x%x resends\n", resend_cnt);
	      fflush (stdout);
	    }
	}

      /* re-interrupt if an interrupt was not handled. */

      if ((get_ISR () & HOST_SIG_BIT) == 0 &&
	  gm.interrupt.type != GM_NO_INTERRUPT)
	{
	  prepare_to_interrupt ("found lost interrupt\n");
	  gm.interrupt.print.string = "found lost interrupt\n";
	  gm_interrupt (GM_PRINT_INTERRUPT);
	}

      /* Decay the backlog decay timer in case some senders defect from the
         queue for any reason. */

      if (1)
	{
	  gm.backlog_delay -= gm.backlog_delay >> 3;
	}
      else
	{
	  if (gm.backlog_delay > GM_SEND_TIMEOUT / 8)
	    gm.backlog_delay -= GM_SEND_TIMEOUT / 8;
	  else
	    gm.backlog_delay = 0;
	}

      /************
       * Alarms
       ************/

      /* Generate any scheduled alarm */

      if (gm.first_port_with_alarm)
	{
	  gm_port_protected_lanai_side_t **port;

	  port = &gm.first_port_with_alarm;
	  do
	    {
	      if ((RTC - (*port)->alarm_time) > 0)
		{
		  /* Hack: pass send queue slot token for the alarm
		     back to the host before the alarm event.  This is
		     necessary to ensure that alarm events take no
		     more than 1 send queue slot at a time. */
		  update_host_sent_queues ();

		  gm_report (*port, GM_NEW_ALARM_EVENT);
		  (*port)->alarm_set = 0;
		  *port = (*port)->next_with_alarm;
		}
	      else
		{
		  port = &(*port)->next_with_alarm;
		}
	      gm_morse_async ("4 ");
	    }
	  while (*port);
	}

      /************
       * Host synchronization
       ************/

      /* If the Host has requested exclusive access to the LANai memory data
         structures, grant it. */

      if (gm.pause_rqst)
	{
	  if (GM_DEBUG_TIMER)
	    {
	      gm_printf (GM_STR ("pausing\n"));
	      fflush (stdout);
	    }
	  gm.pause_cnt++;

	  /* Wait for any previous host interrupt to complete before
	     queueing another interrupt.  Also wait for any DMA to
	     complete, because only then is the LANai really paused. */

	  prepare_to_interrupt ("1 ");
	  await_free_DMA_engine ();

	  gm_interrupt (GM_PAUSE_INTERRUPT);

	  /* Flush the cache because it might be updated during the pause. */

#if GM_ENABLE_VM
	  flush_page_hash_cache ();
#endif

	  /* Wait to resume. */

	  prepare_to_interrupt ("2 ");
	  while (gm.pause_rqst)
	    {
	    }

	  /* Ack the resume request. */

	  gm_interrupt (GM_UNPAUSE_INTERRUPT);
	  if (GM_DEBUG_TIMER)
	    {
	      gm_printf (GM_STR ("unpaused\n"));
	      fflush (stdout);
	    }

	  /* Check if the driver has requested that a port be closed. */

	  if (gm.port_to_close != -1)
	    {
	      gm_always_assert (gm.port_to_close < GM_NUM_PORTS);
	      if (GM_DEBUG_TIMER)
		{
		  gm_printf (GM_STR ("resetting port\n"));
		  fflush (stdout);
		}
	      reset_port (gm.port_to_close);
	      gm.port_to_close = -1;
	      if (GM_DEBUG_TIMER)
		{
		  gm_printf (GM_STR ("reset port\n"));
		  fflush (stdout);
		}
	    }
	  if (gm.port_to_open != -1)
	    {
	      gm_always_assert (gm.port_to_close < GM_NUM_PORTS);
	      if (GM_DEBUG_TIMER)
		gm_printf (GM_STR ("opening port\n"));
	      open_port (&gm_port[gm.port_to_open]);
	      gm.port_to_open = -1;
	      if (GM_DEBUG_TIMER)
		{
		  gm_printf (GM_STR ("opened port\n"));
		  fflush (stdout);
		}
	    }
	}

      /* Schedule next timeout and timer event. */

      gm.timeout_time = RTC + GM_LOW_FREQ_TIMER_PERIOD;
    }

  /* flash the LED */
  heartbeat ();

  /* DMA the aggregated lists of sent messages to the host, or
     arrange to do so when the current DMA completes. */

  if (gm.first_port_with_sent_packets)
    {
      if (get_ISR () & DMA_INT_BIT)
	{
	  /* The DMA engine is free, so simply perform the queue
	     updates now. */

	  update_host_sent_queues ();
	}
      else if (GM_STATE & (SDMAING | RDMAING))
	{
	  /* The DMA engine is busy, and an event will be generated
	     upon completion of the DMA.  Intercept this event.  The
	     event will either be a FINISH_SDMA_EVENT or a
	     FINISH_RDMA_EVENT. */

	  void *finish_sdma, *finish_rdma;

	  finish_sdma = GET_HANDLER (FINISH_SDMA_EVENT);
	  finish_rdma = GET_HANDLER (FINISH_RDMA_EVENT);
	  SET_HANDLER (FINISH_SDMA_EVENT,
		       L_timer__update_host_sent_queues__sdma,);
	  SET_HANDLER (FINISH_RDMA_EVENT,
		       L_timer__update_host_sent_queues__rdma,);
	  gm.timer_state.pushed_sdma_handler = finish_sdma;
	  gm.timer_state.pushed_rdma_handler = finish_rdma;

	  /* HACK: defer setting the next timer interrupt until we
	     have completed the queue update.  This prevents
	     overwriting the gm.timer.pushed_*_handler fields. */

	  set_ISR (TIME_INT_BIT);
	  DISPATCH (137, "");
	}
      else
	{
	  /* OK, we have to spin waiting for the current DMA to
	     complete as a last resort.  However, since we handled the
	     large DMA cases above (SDMAING and RDMAING packet
	     segments), waiting should not take long. */

	  update_host_sent_queues ();
	}
    }

  /* Schedule next timer interrupt */
  set_IT (GM_TIMER_PERIOD);

  DISPATCH (138, "");
}
GM_END_HANDLER;

/****************
 * handlers for updating the host SDMA queues.
 ****************/

_MARK_LABEL (L_timer__update_host_sent_queues__sdma,)
{
  void *sdma_handler, *rdma_handler;

  update_host_sent_queues ();

  sdma_handler = gm.timer_state.pushed_sdma_handler;
  rdma_handler = gm.timer_state.pushed_rdma_handler;

  /* Schedule next timer interrupt.  This was deferred until this
     point (rather than doing it in the L_timer handler) to ensure
     that gm.timer.pushed_{s,r}dma_handler was not overwritten. */

  set_IT (GM_TIMER_PERIOD);

  /* Restore the pushed handlers */

  GET_HANDLER (FINISH_SDMA_EVENT) = sdma_handler;
  GET_HANDLER (FINISH_RDMA_EVENT) = rdma_handler;

  /* avoid the extra dispatch overhead, since we know what needs to
     be done. */

  LOG_DISPATCH (139, "update_host_sent_queues__sdma");
  GOTO_PREFETCHED_HANDLER (sdma_handler);
}
GM_END_HANDLER;

_MARK_LABEL (L_timer__update_host_sent_queues__rdma,)
{
  void *sdma_handler, *rdma_handler;

  update_host_sent_queues ();

  sdma_handler = gm.timer_state.pushed_sdma_handler;
  rdma_handler = gm.timer_state.pushed_rdma_handler;

  /* Schedule next timer interrupt.  This was deferred until this
     point (rather than doing it in the L_timer handler) to ensure
     that gm.timer.pushed_{s,r}dma_handler was not overwritten. */

  set_IT (GM_TIMER_PERIOD);

  /* Restore the pushed handlers */

  GET_HANDLER (FINISH_SDMA_EVENT) = sdma_handler;
  GET_HANDLER (FINISH_RDMA_EVENT) = rdma_handler;

  /* avoid the extra dispatch overhead, since we know what needs to
     be done. */

  LOG_DISPATCH (140, "update_host_sent_queues__rdma");
  GOTO_PREFETCHED_HANDLER (rdma_handler);
}
GM_END_HANDLER;

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
