/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 1999 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file contains external (i.e.: non-inline-able) error handlers
   so that error handling will not contend with the critical path for
   registers. */

#include <stdio.h>
#include "gmcp.h"
#include "gm_error_handlers.h"
#include "gm_enable_directed_send.h"
#include "gm_enable_error_counters.h"
#include "gm_bootstrap.h"

#include "gm_dispatch_logging.h"

/****************************************************************
 * RDMA error handlers
 ****************************************************************/

/* Nack the most recent message for the connection, which had the
   indicated length.  This should be called ONLY when a receive token
   could not be allocated for a message with the correct seqno and
   sesno. */

static inline void
gm_nack (gm_connection_t * c, unsigned int length)
{
  /* BAD: make sure fast */
  gm.backlog_delay += length / GM_BYTES_PER_USEC;
  c->ack_packet.usecs_delay = gm.backlog_delay;
  c->ack_packet.subtype = GM_NACK_SUBTYPE;
  c->ack_packet.sexno.parts.sesno += 2;
  gm_assert (1 & c->ack_packet.sexno.parts.sesno);
  GM_INCR_ERROR_CNT (gm.nack_cnt);
  gm_reack (c);
}

/* Nack the message, indicating that the target port for the first
   nacked message is shutting down so the message is undeliverable and
   should not be resent. */

static inline void
gm_nack_down (gm_connection_t * c)
{
  c->ack_packet.subtype = GM_NACK_DOWN_SUBTYPE;
  c->ack_packet.sexno.parts.sesno += 2;
  gm_assert (1 & c->ack_packet.sexno.parts.sesno);
  c->ack_packet.usecs_delay = 0;
  GM_INCR_ERROR_CNT (gm.nack_down_cnt);
  gm_reack (c);
}

/* Nack the message, indicating that target port for the first nacked
   message has rejected that message so the message is undeliverable and
   should not be resent */

static inline void
gm_nack_reject (gm_connection_t * c)
{
  c->ack_packet.subtype = GM_NACK_REJECT_SUBTYPE;
  c->ack_packet.sexno.parts.sesno += 2;
  gm_assert (1 & c->ack_packet.sexno.parts.sesno);
  c->ack_packet.usecs_delay = 0;
  GM_INCR_ERROR_CNT (gm.nack_reject_cnt);
  gm_reack (c);
}

/* This function is called whenever a packet is received for a
   connection and one end of the connection is known to have been
   reset (the driver has been reloaded or the machine rebooted). */
static inline void
handle_connection_reset_request (gm_connection_t * c,
				 const gm_packet_header_t * p)
{
  /* Ingnore duplicate requests using a timer */

  GM_INCR_ERROR_CNT (gm.handle_connection_reset_request_cnt);

  if (p->sexno.parts.sesno)
    {
      if (c->ack_packet.sexno.parts.sesno == 0)
	{
	  /* Shut down the old connection by having the peer set its
	     send_sexno to (0,1). */
	  c->ack_packet.subtype = GM_NACK_CLOSE_CONNECTION_SUBTYPE;
	  /* Record the sequence number that was seen during a close. */
	  c->close_sexno.whole = p->sexno.whole;
	  GM_INCR_ERROR_CNT (gm.nack_send_close_connection_cnt);
	}
      else
	{
	  GM_INCR_ERROR_CNT (gm.nack_send_nothing1_cnt);
	}
    }
  else				/* p->sexno.parts.sesno == 0 */
    {
      if (((rtc64 () - (c->open_time + GM_CONNECTION_TIMEOUT)) > 0)
	  || (c->ack_packet.sexno.whole == 0) /* feldy hack */ )
	{
	  /* reset the connection */
	  c->open_time = rtc64 ();
	  /* Arrange to resend all unacked packets with the new seqnos */
	  /* Tell peer to start using the new initial seqno.  The
	     session number is chosen such that it is odd and
	     somewhat greater than the sequenc number used before
	     to guaranteed stale packets are harmless. */
	  c->ack_packet.subtype = GM_NACK_OPEN_CONNECTION_SUBTYPE;
	  c->ack_packet.sexno.whole = c->close_sexno.whole;
	  c->ack_packet.sexno.parts.sesno += 1000;
	  c->ack_packet.sexno.parts.sesno |= 1;
	  c->ack_packet.sexno.parts.seqno += 1000;

	  GM_INCR_ERROR_CNT (gm.nack_send_open_connection_cnt);
	}
      else
	{
	  GM_INCR_ERROR_CNT (gm.nack_send_nothing2_cnt);
	}
    }
  /* Always resend some sort of ack. */
  gm_reack (c);
}

void
rdma_handle_out_of_sequence (int mylzero, int mylone)
{
  gm_connection_t *c;
  const gm_packet_header_t *p;
  GM_PARAMETER_MAY_BE_UNUSED (mylone);
  
  p = &gm.recv_chunk[mylzero].packet.as_gm.header;
  c = &gm_connection[p->sender_node_id];

  if (!p->sexno.parts.sesno || !c->ack_packet.sexno.parts.sesno)
    handle_connection_reset_request (c, p);
  else
    gm_reack (c);		/* Must re-ack in case ack was lost. */

  if (GM_ENABLE_ERROR_COUNTERS)
    {
      GM_INCR_ERROR_CNT (gm.out_of_sequence_cnt);
      gm.packet_sexno.whole = p->sexno.whole;
      gm.ack_sexno.whole = c->ack_packet.sexno.whole;
      GM_INCR_ERROR_CNT (gm.drop_cnt);
    }

  /* drop the chunk */

  TOGGLE_STATE (gm.free_recv_chunk_cnt * RDMA_PENDING);
  gm.free_recv_chunk_cnt++;
  NOTICE (FREE_RECV_CHUNK);
  gm_printf_p ("Bad seqno.\n");
}

void
rdma_nack_reject (int mylzero, int mylone)
{
  gm_connection_t *c;
  const gm_packet_header_t *p;
  GM_PARAMETER_MAY_BE_UNUSED (mylone);
  
  p = &gm.recv_chunk[mylzero].packet.as_gm.header;
  c = &gm_connection[p->sender_node_id];

  /* Nack the message, indicating the reason. */

  gm_nack_reject (c);

  /* drop the chunk */

  TOGGLE_STATE (gm.free_recv_chunk_cnt * RDMA_PENDING);
  gm.free_recv_chunk_cnt++;
  NOTICE (FREE_RECV_CHUNK);
  gm_printf_p ("Could not alloc recv token.\n");
}

void
rdma_nack (int mylzero, int mylone)
{
  gm_connection_t *c;
  const gm_packet_header_t *p;
  GM_PARAMETER_MAY_BE_UNUSED (mylone);
  
  p = &gm.recv_chunk[mylzero].packet.as_gm.header;
  c = &gm_connection[p->sender_node_id];

  /* Nack the message, indicating the reason. */

  gm_nack (c, p->length);

  /* drop the chunk */

  TOGGLE_STATE (gm.free_recv_chunk_cnt * RDMA_PENDING);
  gm.free_recv_chunk_cnt++;
  NOTICE (FREE_RECV_CHUNK);
  gm_printf_p ("Could not alloc recv token.\n");
}

void
rdma_nack_down (int mylzero, int mylone)
{
  gm_connection_t *c;
  const gm_packet_header_t *p;
  GM_PARAMETER_MAY_BE_UNUSED (mylone);
  
  p = &gm.recv_chunk[mylzero].packet.as_gm.header;
  c = &gm_connection[p->sender_node_id];

  gm_nack_down (c);

  /* drop the chunk */

  TOGGLE_STATE (gm.free_recv_chunk_cnt * RDMA_PENDING);
  gm.free_recv_chunk_cnt++;
  NOTICE (FREE_RECV_CHUNK);
  /* gm_printf_p ("Could not alloc recv token.\n"); */
}

#if GM_ENABLE_DIRECTED_SEND
void
rdma_nack_directed (int mylzero, int mylone)
{
  /* Nack the message, indicating the reason. */

  gm_connection_t *c;
  const gm_packet_header_t *p;
  GM_PARAMETER_MAY_BE_UNUSED (mylone);
  
  p = &gm.recv_chunk[mylzero].packet.as_gm.header;
  c = &gm_connection[p->sender_node_id];

  gm_nack (c, p->length);	/* Ask that message be resent later. */

  /* drop the chunk */

  TOGGLE_STATE (gm.free_recv_chunk_cnt * RDMA_PENDING);
  gm.free_recv_chunk_cnt++;
  NOTICE (FREE_RECV_CHUNK);
  gm_printf_p ("Could not alloc recv token.\n");
}
#endif /* GM_ENABLE_DIRECTED_SEND */

#define GM_DEBUG_BAD_SENDS 0
void
handle_bad_send (gm_port_protected_lanai_side_t * port, gm_u8_t type,
		 void *event, unsigned int length)
{
  if (GM_DEBUG_BAD_SENDS)
    {
      gm_u8_t *type_ptr;

      type_ptr = ((gm_u8_t *) event + length - 1);
      gm_printf ("bad send event of type %u:\n", type);
      fflush(stdout);
      gm_hex_dump (event, length);
      fflush(stdout);
      *type_ptr = 0;
    }
  gm_report_error (port, GM_NEW_BAD_SEND_DETECTED_EVENT);
}

void
handle_bad_crc (const gm_packet_header_t * p, unsigned int bytes_received)
{
  /* Record a probable CRC error for the corresponding connection
     if this is a GM packet. */
  if (p->type == GM_PACKET_TYPE
      && bytes_received >= sizeof (gm_packet_header_t)
      && p->sender_node_id <= gm.max_node_id)
    {
      gm.connection[p->sender_node_id].bad_crc_cnt++;
      if (++gm.connection[p->sender_node_id].probable_crc_error_cnt == 0)
	gm_connection[p->sender_node_id].probable_crc_error_cnt = ~0;
    }

  GM_INCR_ERROR_CNT (gm.drop_cnt);

  if (p->type & 0x8000)
    {
      /* this is a routing byte assume it is not corrupted? */
      unsigned char *cptr = (unsigned char *)&p->type;
      gm_u16_t type;
      int count=GM_MAX_NETWORK_DIAMETER;
      while ((*cptr & 0x80) && (count--))
	 cptr++;
      type = cptr[0];
      type = (type<<8) + cptr[1];
      if ((type == GM_PACKET_TYPE) || (type == GM_MAPPING_PACKET_TYPE))
	GM_INCR_ERROR_CNT (gm.badroute_cnt);
      else 
	GM_INCR_ERROR_CNT (gm.badcrc_cnt);
    }
  else 
    {
      GM_INCR_ERROR_CNT (gm.badcrc_cnt);
      gm_printf_p ("bad packet CRC\n");
    }
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
