/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file abstracts the LANai hardware packet and DMA engines so that the
   single API presented here can be used to control any LANai. */

#ifndef _GM_bootstrap_h_
#define _GM_bootstrap_h_

#include <stdio.h>

#include "gm_call_trace.h"
#include "gm_compiler.h"
#include "gm_debug.h"
#include "gm_debug_lanai_dma.h"
#include "gm_enable_crc32.h"
#include "gm_lanai_def.h"
#include "gmcp.h"

/****************************************************************
 * Forward declarations
 ****************************************************************/

static void set_ISR (gm_u32_t);
static gm_u32_t get_ISR (void);

/****************************************************************
 * LANai hardware initialization.
 ****************************************************************/

void gm_init_lanai_hardware (void);

/****************************************************************
 * Abstraction of state bits, including ISR encodings.
 ****************************************************************/

/* Introduce a barrier between a write to a special register and
   the subsequent read of ISR.

   Here we pass SPECIAL as a pointer to prevent the compiler from from
   re-reading the volatile special register after it is written. */

static inline void
isr_barrier (gm_u32_t * special)
{
#if L4 | L5 | L6
  asm ("nop ! ISR_BARRIER()": : "m" (*special):"isr");
#elif L7 | L8
  GM_PARAMETER_MAY_BE_UNUSED (special);
  /* no need to introduce delay */
#elif L9
  asm ("nop ! ISR_BARRIER()": "=m" (ISR):"m" (*special));
#else
#error unrecognized LANai type
#endif
}

#define ISR_BARRIER(special) isr_barrier ((gm_u32_t *) &(special))

/****************
 * Version-specific state bit definitions
 ****************/

/****
 * L4
 ****/

#if L4
enum gm_state_bits
{
  /* 0x00000001 */ FREE_RECV_CHUNK = BYTE_RDY_BIT,
/* 0x00000002 *//* RECV_INT_BIT */
/* 0x00000004 *//* BUFF_INT_BIT */
/* 0x00000008 *//* SEND_INT_BIT */
/* 0x00000010 *//* DMA_INT_BIT  */
/* 0x00000020 *//* TIME_INT_BIT */
  /* 0x00000040 */ FREE_SEND_CHUNK = WDOG_INT_BIT,
  /* 0x00000080 */ ACK_PENDING = TAIL_INT_BIT,
  /* 0x00000100 */ RDMA_PENDING = OFF_BY_1_BIT,
  /* 0x00000200 */ RDMAING = OFF_BY_2_BIT,
  /* 0x00000400 */ RECEIVING = WAKE_INT_BIT,
  /* 0x00000800 */ SENDING = NRES_INT_BIT,
  /* 0x00001000 */ SDMAING = LINK_INT_BIT,
  /* 0x00002000 */ SEND_PENDING = SEND_RDY_BIT,
  /* 0x00004000 */ SDMA_PENDING = HALF_RDY_BIT,
};
#define BIGGEST_STATE_BIT	SDMA_PENDING

/****
 * L5
 ****/

#  elif L5
enum gm_state_bits
{
  /* 0x00000001 */ FREE_RECV_CHUNK = HEAD_INT_BIT,
/* 0x00000002 *//* RECV_INT_BIT */
/* 0x00000004 *//* BUFF_INT_BIT */
/* 0x00000008 *//* SEND_INT_BIT */
  /* 0x00000010 */ DMA_INT_BIT = (E2L_INT_BIT | L2E_INT_BIT),
/* 0x00000020 *//* TIME_INT_BIT */
  /* 0x00000040 */ RDMA_PENDING = WDOG_INT_BIT,
  /* 0x00000080 */ RDMAING = OFF_BY_1_BIT,
  /* 0x00000100 */ RECEIVING = OFF_BY_2_BIT,
  /* 0x00000200 */ SENDING = OFF_BY_4_BIT,
  /* 0x00000400 */ SDMAING = NRES_INT_BIT,
  /* 0x00000800 */ FREE_SEND_CHUNK = WAKE_INT_BIT,
  /* 0x00001000 */ SEND_PENDING = 0x00002000,
  /* 0x00002000 */ SDMA_PENDING = 0x00004000,
  /* 0x00004000 */ ACK_PENDING = 0x00008000,
};

#define BIGGEST_STATE_BIT       ACK_PENDING

/****
 * L6
 ****/

#elif L6
enum gm_state_bits
{
  /* 0x00000001 */ FREE_RECV_CHUNK = HEAD_INT_BIT,
/* 0x00000002 *//* RECV_INT_BIT */
/* 0x00000004 *//* BUFF_INT_BIT */
/* 0x00000008 *//* SEND_INT_BIT */
  /* 0x00000010 */ FREE_SEND_CHUNK = LAN8_SIG_BIT,
  /* 0x00000020 */ ACK_PENDING = LAN9_SIG_BIT,
  /* 0x00000040 */ TIME_INT_BIT = TIME0_INT_BIT,
  /* 0x00000080 */ RDMA_PENDING = TIME1_INT_BIT,
  /* 0x00000100 */ RDMAING = OFF_BY_1_BIT,
  /* 0x00000200 */ RECEIVING = OFF_BY_2_BIT,
  /* 0x00000400 */ SENDING = OFF_BY_4_BIT,
  /* 0x00000800 */ SDMAING = NRES_INT_BIT,
/* 0x00001000 *//* DMA_INT_BIT */
  /* 0x00002000 */ SEND_PENDING = TIME2_INT_BIT,
  /* 0x00004000 */ SDMA_PENDING = MEM_INT_BIT,
};
#define BIGGEST_STATE_BIT	SDMA_PENDING

/****
 * L7
 ****/

#elif L7
/* Do not remove: acts as feature switch */
#undef POLL_PENDING _POLL_PENDING
enum gm_state_bits
{
  /* 0x00000001 */ FREE_RECV_CHUNK = HEAD_INT_BIT,
/* 0x00000002 *//* RECV_INT_BIT */
/* 0x00000004 *//* BUFF_INT_BIT */
/* 0x00000008 *//* SEND_INT_BIT */
  /* 0x00000010 */ FREE_SEND_CHUNK = LAN8_SIG_BIT,
  /* 0x00000020 */ ACK_PENDING = LAN9_SIG_BIT,
  /* 0x00000040 */ TIME_INT_BIT = TIME0_INT_BIT,
  /* 0x00000080 */ RDMA_PENDING = TIME1_INT_BIT,
  /* 0x00000100 */ RDMAING = OFF_BY_1_BIT,
  /* 0x00000200 */ RECEIVING = OFF_BY_2_BIT,
  /* 0x00000400 */ SENDING = OFF_BY_4_BIT,
  /* 0x00000800 */ SDMAING = NRES_INT_BIT,
/* 0x00001000 *//* WAKE_INT_BIT */
  /* 0x00002000 */ SEND_PENDING = TIME2_INT_BIT,
  /* 0x00004000 */ SDMA_PENDING = MEMORY_INT_BIT,
  _POLL_PENDING = PARITY_INT_BIT,
};

#ifdef POLL_PENDING
#define BIGGEST_STATE_BIT	POLL_PENDING
#else
#define BIGGEST_STATE_BIT	SDMA_PENDING
#endif

/****
 * L8, L9
 ****/

#elif L8 | L9
/* Do not remove: acts as feature switch */
#undef POLL_PENDING _POLL_PENDING
enum gm_state_bits
{
  /* 0x00000001 */ FREE_RECV_CHUNK = HEAD_INT_BIT,
/* 0x00000002 *//* RECV_INT_BIT */
/* 0x00000004 *//* BUFF_INT_BIT */
/* 0x00000008 *//* SEND_INT_BIT */
  /* 0x00000010 */ FREE_SEND_CHUNK = WAKE1_INT_BIT,
  /* 0x00000020 */ ACK_PENDING = WAKE2_INT_BIT,
  /* 0x00000040 */ TIME_INT_BIT = TIME0_INT_BIT,
  /* 0x00000080 */ RDMA_PENDING = TIME1_INT_BIT,
  /* 0x00000100 */ RDMAING = ORUN1_INT_BIT,
  /* 0x00000200 */ RECEIVING = ORUN2_INT_BIT,
  /* 0x00000400 */ SENDING = ORUN4_INT_BIT,
  /* 0x00000800 */ SDMAING = NRES_INT_BIT,
/* 0x00001000 *//* WAKE0_INT_BIT */
  /* 0x00002000 */ SEND_PENDING = TIME2_INT_BIT,
  /* 0x00004000 */ SDMA_PENDING = MEM_INT_BIT,
  _POLL_PENDING = PAR_INT_BIT,
};

#ifdef POLL_PENDING
#define BIGGEST_STATE_BIT	POLL_PENDING
#else
#define BIGGEST_STATE_BIT	SDMA_PENDING
#endif

/****************
 * unrecognized
 ****************/

#else /* L[456789] not defined */
#error LANai version not recognized
#endif

#define IMR_INIT_VAL		(TIME_INT_BIT 				\
				 | DMA_INT_BIT				\
				 | SEND_INT_BIT				\
				 | RECV_INT_BIT				\
				 | BUFF_INT_BIT)

/**********************************************************************
 * DMA interface code for LANais
 **********************************************************************/

/****************************************
 * DMA interface code for LANais
 ****************************************/

#include "gm_lanai_def.h"
#include "gm_types.h"
#include "gm_debug.h"
#include "gm_debug_lanai_dma.h"
#include "gm_stbar.h"
#include "gcc_version.h"
#include "gm_lanai_globals.h"

#if L4
/* nothing to do */
#elif L5
extern void *LAR;
extern void *EAR;
extern int DMA_DIR;
#elif L6 | L7 | L8 | L9
/* nothing to do */
#else
#error
#endif

/****************
 * ISR
 ****************/

static inline gm_u32_t
get_ISR (void)
{
  /* No need for any sort of barrier here.  That is taken care of
     in set_RML() et al. */

#if (L4 | L5 | L6)		/*  && GCC_VERSION >= GCC (2,8) */
  register gm_u32_t isr asm ("isr");
  asm volatile ("nop ! ISR barrier":"=r" (isr));
  return isr;
#elif L7 | L8 | L9
  return ISR;
#endif
}

static inline void
set_ISR (gm_u32_t val)
{
#if defined lanai3		/*  && GCC_VERSION >= GCC (2,8) */
  /* BAD: circumvent volatile register bug in lanai3-gcc-2.8.1 */
  asm volatile ("mov %0,%?isr\n\t"::"r" (val):"isr", "imr", "memory");
#else
  ISR = (int) val;
#endif
}

/****************
 * IMR
 ****************/

static inline gm_u32_t
get_IMR (void)
{
#if L4 | L5 | L6
  register gm_u32_t imr asm ("imr");
  asm volatile ("! IMR barrier":"=r" (imr));
  return imr;
#elif L7 | L8 | L9
  gm_u32_t ret;

  ret = IMR_INIT_VAL;
  return ret;
#else
#error unrecognize processor type
#endif
}

static inline void
set_IMR (gm_u32_t val)
{
#if L4 | L5 | L6
#if GCC_VERSION >= GCC(2,95) || 1
  /* This compiler does not support volatile registers */
  asm volatile ("mov %0,%?imr"::"r" (val):"isr", "imr", "memory");
#else
  IMR = val;
#endif
#elif L7 | L8 | L9
  /* No IMR register on this machine. */
  gm_assert (get_IMR () == val);
#else
#error Do not know what to do.
#endif
  GM_STBAR ();
}

/****************
 * SML, SMLT
 ****************/

/* start a send DMA using the previously set SA and SMP pointers.
   "sml" is rounded down to a GM_PACKET_GRANULARITY boundary, and
   the resulting sml_prime pointer points to the first byte that will not be
   sent.  sml_prime must be larger than the previously set SMP. */

static inline void
set_SML (void *sml)
{
  gm_assert (((void *) (sml) > (void *) gm.send_chunk[0].route &&
	      (void *) (sml) <= (void *) (&gm.send_chunk[0] + 1)) ||
	     ((void *) (sml) > (void *) gm.send_chunk[1].route &&
	      (void *) (sml) <= (void *) (&gm.send_chunk[1] + 1)) ||
	     ((void *) (sml) > (void *) &gm_connection[0] &&
	      (void *) (sml) <= (void *) (&gm_connection
					  [GM_NUM_PRIORITIES *
					   (gm.max_node_id + 1)])));
  gm_assert ((void *) (sml) > (void *) SMP);

  GM_STBAR ();
#if L4
  SML = (void *) ((char *) (sml) - 4);
#elif L5 | L6 | L7 | L8 | L9
  SML = (void *) ((char *) (sml) - 8);
#else
#error
#endif
  ISR_BARRIER (SML);
}

/* Like SML, marks the last flit as a tail flit. */

static inline void
set_SMLT (void *sml)
{
#if !GM_ENABLE_GALVANTECH_WORKAROUND
  gm_assert (((void *) (sml) > (void *) gm.send_chunk[0].route &&
	      (void *) (sml) <= (void *) (&gm.send_chunk[0] + 1)) ||
	     ((void *) (sml) > (void *) gm.send_chunk[1].route &&
	      (void *) (sml) <= (void *) (&gm.send_chunk[1] + 1)) ||
	     ((void *) (sml) > (void *) &gm_connection[0] &&
	      (void *) (sml) <= (void *) (&gm_connection
					  [GM_NUM_PRIORITIES *
					   (gm.max_node_id + 1)]))
	     || ((void *) (sml) > (void *) &gm.ethernet.send.chunk[0].packet
		 && (void *) (sml) <=
		 (void *) (&gm.ethernet.send.chunk[0].packet + 1))
	     || ((void *) (sml) > (void *) &gm.ethernet.send.chunk[1].packet
		 && (void *) (sml) <=
		 (void *) (&gm.ethernet.send.chunk[1].packet + 1)));
#endif
  gm_assert ((void *) (sml) > (void *) SMP);

  GM_STBAR ();
#if L4
  SMLT = (void *) (((char *) sml + GM_PACKET_GRANULARITY - 1) - 4);
#elif L5 | L6 | L7 | L8 | L9
  /* Do NOT subract 8 for the L5/6 SMLT */
  SMLT = (void *) ((char *) sml + GM_PACKET_GRANULARITY - 1);
#else
#error
#endif
  ISR_BARRIER (SMLT);
}

/* Start a receive using the previously set RMP value.  rml is rounded
   down to a GM_PACKET_GRANULARITY boundary and the resulting rml_prime
   pointer points to the first byte past the end of the receive buffer. */

static inline void
set_RML (void *rml)
{
  gm_assert ((void *) (rml) == (void *) (&gm.recv_chunk[0].end) ||
	     (void *) (rml) == (void *) (&gm.recv_chunk[1].end));
  gm_assert ((void *) RMP == (void *) gm.recv_chunk[0].packet.as_bytes ||
	     (void *) RMP == (void *) gm.recv_chunk[1].packet.as_bytes);
  gm_assert ((void *) (rml) > (void *) RMP);

  GM_STBAR ();
#if L4
  RML = (void *) ((char *) (rml) - 4);
#elif L5 | L6 | L7 | L8 | L9
  RML = (void *) ((char *) (rml) - 8);
#else
#error
#endif
  ISR_BARRIER (RML);
}

static inline void
set_SMH (void *ptr)
{
#if GM_ENABLE_CRC32 && (L7 | L8 | L9)
  if (ptr)
    {
      SMH = (ptr);
    }
#else
  GM_PARAMETER_MAY_BE_UNUSED (ptr);
#endif
}


/* DMA direction macros for use with set_DMA_CTR. */
#if L4
#define GM_SDMA_DIR (-1)
#define GM_RDMA_DIR 0
#elif L5 | L6 | L7 | L8 | L9
#define GM_SDMA_DIR DMA_E2L
#define GM_RDMA_DIR DMA_L2E
#else
#error
#endif

/* foo */

static inline void
await_free_DMA_engine (void)
{

#if L4 | L5
  while (!(get_ISR () & DMA_INT_BIT));
#elif L6 | L7 | L8 | L9

  GM_STBAR ();
  if (DMA_INT_BIT == DMA_TERMINAL)
    {
      while (!(get_ISR ()
	       & (gm_u32_t) gm.dma_descriptor.next_with_flags & DMA_INT_BIT))
	GM_STBAR ();
    }
  else
    {
      while (((get_ISR () & DMA_INT_BIT)
	      + ((gm_u32_t) gm.dma_descriptor.next_with_flags & DMA_TERMINAL))
	     != (DMA_INT_BIT + DMA_TERMINAL))
	GM_STBAR ();
    }
  GM_STBAR ();
#else
#error
#endif
}

#define set_EAR(ea, dma_dir) do {					\
  GM_PRINT (GM_TRACE_LANAI_DMA, ("set_EAR(0x%qx,%d) called\n",		\
				      (gm_u64_t) (ea), (dma_dir)));	\
  gm_assert ((ea) >= GM_PAGE_LEN);					\
  _set_EAR ((ea), (dma_dir));						\
} while (0)

static inline void
_set_EAR (gm_dp_t ea, int dma_dir)
{
#if L4 | L6
  GM_PARAMETER_MAY_BE_UNUSED (dma_dir);
  EAR = (gm_lp_t) (((gm_u32_t) ea) & ((gm_u32_t)(~3)));  /* round down to fix SBus dirsend problem */
#elif L5
  if (dma_dir == GM_SDMA_DIR)
    {
      EAR = (gm_lp_t) (gm_u32_t) ea;
      E2L_EAR = (gm_lp_t) (gm_u32_t) ea;
    }
  else
    {
      EAR = (gm_lp_t) (gm_u32_t) ea;
      L2E_EAR = (gm_lp_t) (gm_u32_t) ea;
    }
#elif L7 | L8 | L9
  GM_PARAMETER_MAY_BE_UNUSED (dma_dir);
  if (sizeof (ea) == 8)
    {
      gm_assert (&gm.dma_descriptor.ebus_addr_high + 1
		 == &gm.dma_descriptor.ebus_addr_low);
      asm volatile ("nop	! begin EAR write");
      *(gm_dp_t *) & gm.dma_descriptor.ebus_addr_high = ea;
      asm volatile ("nop	! end EAR write");
      GM_PRINT (GM_TRACE_LANAI_DMA,
		("EAR was set to 0x%qx\n",
		 *(gm_u64_t *) & gm.dma_descriptor.ebus_addr_high));
    }
  else
    {
      gm_assert (gm.dma_descriptor.ebus_addr_high == 0);
      gm.dma_descriptor.ebus_addr_low = ea;
    }
#else
#error Unrecognized LANai version.
#endif
}

static inline gm_dp_t
get_EAR (void)
{
#if L4 | L5 | L6
  return (gm_u32_t) EAR;
#elif L7 | L8 | L9
  return *(gm_dp_t *) & gm.dma_descriptor.ebus_addr_high;
#endif
}

/*#undef EAR
  #define EAR FIXME*/

static inline void
set_LAR (gm_lp_t la, int dma_dir)
{
#if L4 | L6
  GM_PARAMETER_MAY_BE_UNUSED (dma_dir);
  LAR = la;
#elif L5
  if (dma_dir == GM_SDMA_DIR)
    {
      LAR = la;
      E2L_LAR = la;
    }
  else
    {
      LAR = la;
      L2E_LAR = la;
    }
#elif L7 | L8 | L9
  GM_PARAMETER_MAY_BE_UNUSED (dma_dir);
  gm.dma_descriptor.lanai_addr = la;
#else
#error Unrecognized LANai version.
#endif
}

static inline void
set_LED (unsigned int x)
{
#if L5  
  /* Don't touch the CSPI LED, since it is used to control the host
     CPU reset. Only CSPI uses the l5*/
#else
  LED = x;
#endif
}


/* set the DMA_CTR and initiate a DMA.  DC is rounded down to
   GM_DMA_GRANULARITY before being used.  DC most not be zero or be
   rounded down to zero. */

#define set_DMA_CTR(count,dir) __set_DMA_CTR((count),(dir),__FILE__,__LINE__)

static inline void
__set_DMA_CTR (unsigned dc, unsigned dma_dir, const char *file, int line)
{
  GM_PARAMETER_MAY_BE_UNUSED (file);
  GM_PARAMETER_MAY_BE_UNUSED (line);
  
  gm_assert ((dc) > 0);
  gm_assert ((dc) <= GM_MAX_DMA_CTR);
  gm_assert ((dc) >= GM_DMA_GRANULARITY);

#if GM_DMA_GRANULARITY > 1 && GM_DMA_GRANULARITY_ROUNDUP == 0
  gm_assert (GM_DMA_ALIGNED (dc));
  gm_assert (dc > 0 && dc < GM_MAX_DMA_CTR);
  gm_assert (GM_DMA_ALIGNED (gm.dma_descriptor.ebus_addr_low));
  gm_assert (GM_DMA_ALIGNED (gm.dma_descriptor.lanai_addr));
#endif

#if L4
  DMA_DIR = dma_dir;
  GM_STBAR ();

#if GM_DEBUG_LANAI_DMA
  gm_assert ((get_EAR () & 3) == 0);
  {
    gm_dp_t p, limit;

    p = get_EAR ();
    gm_always_assert(p);/* OK, this can become an abusive assertion some day */
    limit = GM_DMA_ALIGN (u32, p + dc);
    while (p < limit)
      {
	if (gm_noticed_dma_addr (&gm.debug.dma_page_bitmap, p) != GM_SUCCESS)
	  {
	    GM_PANIC (("p=0x%qx dc=0x%x file=\"%s\" line=%d\n",
		       (gm_u64_t) p, dc, file, line));
	  }
	p += GM_PAGE_LEN;
      }
  }
#endif /* GM_DEBUG_LANAI_DMA */

  gm_always_assert (DMA_CTR == 0);
  DMA_CTR = (dc);
  ISR_BARRIER (DMA_CTR);

#elif L5

  /* CSPI special L5 code */
  set_ISR (E2L_INT_BIT | L2E_INT_BIT);
  DMA_DIR = dma_dir;
  if (dma_dir == GM_SDMA_DIR)
    {
      E2L_CTR = (dc);
      ISR_BARRIER (E2L_CTR);
    }
  else
    {
      L2E_CTR = (dc);
      ISR_BARRIER (L2E_CTR);
    }
#elif L6 | L7 | L8 | L9

  /* clear DMA_INT_BIT, since the L6 hardware does not do this */

  set_ISR (DMA_INT_BIT);

  /* record DMA direction while maintaining other settings */

  gm.dma_descriptor.next_with_flags
    = (gm_lp_t) (&gm.dma_descriptor) + dma_dir + DMA_WAKE;

  gm_assert (GM_DMA_ALIGNED (&gm.dma_descriptor));

  /* record DMA length */

  gm.dma_descriptor.len = dc;

  /* Flush descriptor and start DMA */
  GM_STBAR ();
  PULSE = 2;
  ISR_BARRIER (PULSE);
#else
#error
#endif
}


/****************************************************************
 * Start a simple RDMA
 ****************************************************************/

/* In these functions, lar, ear, and len are each rounded down to
   GM_DMA_GRANULARITY before use.  (The rounding usually is done
   by the hardware.) */

#define start_RDMA(lar, ear, len) do {					\
  GM_PRINT (GM_TRACE_LANAI_DMA,						\
		 ("_start_RDMA(0x%p,0x%qx,0x%x) called\n",		\
		  lar, (gm_u64_t) ear, len));				\
  gm_assert (len >= GM_DMA_GRANULARITY);				\
  GM_CHECK_IRIX_DMA_ADDR (ear,__LINE__,__LINE__); /* fixme 2nd param */ \
  _start_RDMA (lar, ear, len);						\
} while (0)

static inline void
_start_RDMA (gm_lp_t lar, gm_dp_t ear, gm_u32_t len)
{
  GM_STBAR ();
  set_LAR (lar, GM_RDMA_DIR);
  set_EAR (ear, GM_RDMA_DIR);
  set_DMA_CTR (len, GM_RDMA_DIR);
}

/* start a simple RDMA into the recv queue */

#define start_GRANULAR_RDMA(lar, ear, len) do {				\
  gm_assert (GM_RDMA_ALIGNED (ear));					\
  gm_assert (GM_RDMA_ALIGNED (len));					\
  start_RDMA (lar, ear, len);						\
} while (0)

/* start a simple SDMA.  The start of the DMA must be aligned, but the
   length may be misaligned, in which case a bit extra will be DMAd. */

#define start_SDMA(ear, lar, len, ref_line) do {			\
  GM_PRINT (GM_TRACE_LANAI_DMA,						\
		 ("start_SDMA(0x%qx,0x%p,0x%x) called\n",		\
		  (gm_u64_t) (ear), (lar), (len)));			\
  gm_assert ((len) >= GM_DMA_GRANULARITY);				\
  GM_CHECK_IRIX_DMA_ADDR (ear,ref_line,__LINE__);                       \
  _start_SDMA ((ear), (lar), (len));					\
} while (0)

static inline void
_start_SDMA (gm_dp_t ear, gm_lp_t lar, gm_u32_t len)
{
  set_LAR (lar, GM_SDMA_DIR);
  set_EAR (ear, GM_SDMA_DIR);
  GM_STBAR ();
  set_DMA_CTR (len, GM_SDMA_DIR);
}

/* set the interrupt timer */

static inline void
set_IT (signed it)
{
  GM_STBAR ();
#if L4 | L5
  IT = (it);
  ISR_BARRIER (IT);
#elif L6 | L7 | L8 | L9
  IT0 = (it);
  ISR_BARRIER (IT0);
#else
#error
#endif
}

/* return the ethernet checksum for a just-completed DMA or 0 if
   there is none. */

static inline gm_u16_t
rdma_checksum (void)
{
#if L7
  /* ensure the DMA descriptor writeback has occurred. */
  await_free_DMA_engine ();
  return gm.dma_descriptor.csum[1];
#elif L6
  /* ensure the DMA descriptor writeback has occurred. */
  await_free_DMA_engine ();
  {
    gm_u32_t a, b;

    a = gm.dma_descriptor.csum[0];
    a += gm.dma_descriptor.csum[1];
    a += a >> 16;		/* add back any 16-bit carry. */
    return (gm_u16_t) (0xffff & a);
  }
#else
  return 0;
#endif
}

/****************************************************************
 * Parity checking
 ****************************************************************/

static inline void
gm_check_parity (void)
{
  int error = 0;

#if L7
  error = ((get_ISR () & PARITY_INT_BIT) != 0);
#elif L8 | L9
  error = ((get_ISR () & PAR_INT_BIT) != 0);
#endif

  if (error)
    {
      gm_printf ("LANai detected a PARITY error in board SRAM"
		 " at rtc time = 0x%x%08x\n", *(gm_u32_t *) 0, *(gm_u32_t *) 4);
      fflush (stdout);
      gm_always_assert (0);
    }
}

#endif /* _gm_bootstrap_h_ defined */

/*
  This file uses GM standard indentation.

  Local Variables:
  c-file-style:"gnu"
  c-backslash-column:72
  tab-width:8
  End:
*/
