/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* This file provides alloc, calloc, and free routines to manage an
   externally specified "zone" of memory.  

   This package use buddy-system memory allocation to allocate
   (2^n)-byte regions of memory, where "n" is referred to as
   the "size" of the allocated area of memory.

   All allocated (or freed) areas are maximally aligned. A zone is
   a chunk of memory starting and ending on page boundaries. The
   size and state of each area are encoded in a pair of bit-arrays.
   One array has a bit set for each position corresponding to a buddy-
   boundary.  The second array has a bit set for each position corresponding
   to an area that is not free.

   In the code, variables named with "size" are in logarithmic units
   and those named "length" are in real units.

   DEBUGGING
   *********

   If debugging is turned on, we "mark" al valid areas in the zone,
   using the gm_mark API, and we check each area passed to a function.
   This should catch any DMA overruns type corruption of the data structures
   stored in the managed memory. */

#include "gm_internal.h"

/****************************************************************
 * Types
 ****************************************************************/

/****************
 * Zone areas
 ****************/

/* Zones are divided into managed buffers called "areas", which may either
   represent free buffers or buffers holding user data. */

typedef struct gm_zone_area
{
#if !GM_DEBUG
  struct gm_zone_area   *next;
  struct gm_zone_area   *prev;
#else  /* GM_DEBUG */
  gm_mark_t mark;
  gm_size_t pad_afer_mark;
  /* 8 */
  gm_mark_t free_mark_2;
  struct gm_zone_area   *next;
  struct gm_zone_area   *prev;
  gm_mark_t free_mark_1;
  gm_size_t pad[2];
#endif
} gm_zone_area_t;

GM_TOP_LEVEL_ASSERT (GM_POWER_OF_TWO (sizeof (gm_zone_area_t)));

/* Define the offset of user data within an area */

#if !GM_DEBUG
#define GM_ZONE_AREA_PTR_OFFSET 0
#else  /* GM_DEBUG */
#define GM_ZONE_AREA_PTR_OFFSET GM_OFFSETOF (gm_zone_area_t, free_mark_2)
#endif

#define GM_PTR_FOR_AREA(a) ((void *) ((char *)(a) + GM_ZONE_AREA_PTR_OFFSET))
#define GM_AREA_FOR_PTR(ptr) ((gm_zone_area_t *)			\
			      ((char *) (ptr) - GM_ZONE_AREA_PTR_OFFSET))

typedef struct gm_zone
{
  gm_zone_area_t	*base;
  gm_size_t		length;
  gm_zone_area_t	free_list[32];
  gm_u32_t		*boundary_bits;
  gm_u32_t		*reserved_bits;
  struct gm_mark_set	*user_mark_set;
  struct gm_mark_set	*free_mark_set;
} gm_zone_t;

/****************************************************************
 * Macros
 ****************************************************************/

#define power_of_two(n)		(!((n)&((n)-1)))
#define get_zone_bit(a, i)	(a[(i) >> 5] & (1 << ((i) & 0x1f)))
#define set_zone_bit(a, i)	(a[(i) >> 5] |= (1 << ((i) & 0x1f)))
#define clear_zone_bit(a, i)	(a[(i) >> 5] &= ~(1 << ((i) & 0x1f)))

#define get_area_boundary(z,a)  get_zone_bit((z)->boundary_bits,(a)-((z)->base))
#define set_area_boundary(z,a)  set_zone_bit((z)->boundary_bits,(a)-((z)->base))
#define clear_area_boundary(z,a) clear_zone_bit((z)->boundary_bits,(a)-((z)->base))
#define area_is_reserved(z,a)   get_zone_bit((z)->reserved_bits,(a)-((z)->base))
#define area_is_free(z,a)       (!get_zone_bit((z)->reserved_bits,(a)-((z)->base)))
#define mark_area_reserved(z,a) set_zone_bit((z)->reserved_bits,(a)-((z)->base))
#define mark_area_free(z,a)     clear_zone_bit((z)->reserved_bits,(a)-((z)->base))

/****************************************************************
 * debugging functions
 ****************************************************************/

static gm_inline
void
validate_user_area (gm_zone_t *zone, gm_zone_area_t *area)
{
  GM_PARAMETER_MAY_BE_UNUSED (zone);
  GM_PARAMETER_MAY_BE_UNUSED (area);
     
#if GM_DEBUG
  if (!gm_mark_is_valid (zone->user_mark_set, &area->mark))
    {
      gm_perror ("user freed a free buffer or overrun corrupted buffer",
		 GM_INTERNAL_ERROR);
      gm_abort ();
    }
#endif
}

static gm_inline
void
validate_free_area (gm_zone_t *zone, gm_zone_area_t *area)
{
  GM_PARAMETER_MAY_BE_UNUSED (zone);
  GM_PARAMETER_MAY_BE_UNUSED (area);

#if GM_DEBUG
  if (!gm_mark_is_valid (zone->free_mark_set, &area->mark))
    {
      gm_perror ("zone free area corrupted (probably by an overrun)",
		 GM_INTERNAL_ERROR);
      gm_abort ();
    }
  else if (!gm_mark_is_valid (zone->free_mark_set, &area->free_mark_1)
	   || !gm_mark_is_valid (zone->free_mark_set, &area->free_mark_2))
    {
      gm_perror ("zone adminstrative data has been corrupted",
		 GM_INTERNAL_ERROR);
      gm_abort ();
    }
#endif
}

static gm_inline
void
unmark_area (gm_zone_t *zone, gm_zone_area_t *area)
{
  GM_PARAMETER_MAY_BE_UNUSED (zone);
  GM_PARAMETER_MAY_BE_UNUSED (area);

#if GM_DEBUG
  if (gm_mark_is_valid (zone->user_mark_set, &area->mark))
    {
      gm_unmark (zone->user_mark_set, &area->mark);
    }
  else if (gm_mark_is_valid (zone->free_mark_set, &area->mark))
    {
      gm_unmark (zone->free_mark_set, &area->free_mark_2);
      gm_unmark (zone->free_mark_set, &area->free_mark_1);
      gm_unmark (zone->free_mark_set, &area->mark);
    }
  else
    {
      gm_perror ("Cannot unmark unmarked area", GM_INTERNAL_ERROR);
      gm_abort ();
    }
#endif
}

static gm_inline
void
mark_user_area (gm_zone_t *zone, gm_zone_area_t *area)
{
#if GM_DEBUG
  gm_status_t status;

  status = gm_mark (zone->user_mark_set, &area->mark);
  gm_always_assert (status == GM_SUCCESS);
#else
  GM_PARAMETER_MAY_BE_UNUSED (zone);
  GM_PARAMETER_MAY_BE_UNUSED (area);
#endif
}

static gm_inline
void
mark_free_area (gm_zone_t *zone, gm_zone_area_t *area)
{
#if GM_DEBUG
  gm_status_t status;

  status = gm_mark (zone->free_mark_set, &area->mark);
  gm_always_assert (status == GM_SUCCESS);
  status = gm_mark (zone->free_mark_set, &area->free_mark_1);
  gm_always_assert (status == GM_SUCCESS);
  status = gm_mark (zone->free_mark_set, &area->free_mark_2);
  gm_always_assert (status == GM_SUCCESS);
#else
  GM_PARAMETER_MAY_BE_UNUSED (zone);
  GM_PARAMETER_MAY_BE_UNUSED (area);
#endif
}

/****************************************************************
 * implementation
 ****************************************************************/

/* determine the size of an area by finding its boundary, using a
   binary search, starting at the extreme points of the zone,
   and stopping when a midpoint does not fall a boundary.  */

static unsigned int
sizeof_area (gm_zone_t *zone, gm_zone_area_t *area)
{
  gm_zone_area_t *start, *middle, *stop;
  
  gm_assert (zone && area);
  gm_assert (gm_zone_addr_in_zone (zone, area));

  start = zone->base;
  gm_assert (GM_POWER_OF_TWO (sizeof (gm_zone_area_t)));
  stop = zone->base + (zone->length / sizeof (gm_zone_area_t));
  middle = start + (stop - start)/2;

  while (middle != start && get_area_boundary (zone, middle))
    {
      if (area >= middle)
	start = middle;
      else
	stop = middle;

      middle =  start + (stop - start)/2;
      gm_assert (stop > start);
    
    }
  gm_assert (start == area && stop > area);
  
  /*return the size, in logarithmic units */
  return (unsigned int) gm_log2_roundup (GM_STATIC_CAST (unsigned long,
					 (gm_u8_t *)stop - (gm_u8_t *)start));
}

static void
free_area (gm_zone_t *zone, gm_zone_area_t *area, unsigned int size)
{
  validate_free_area (zone, area);
  
  gm_assert (size == sizeof_area (zone, area));

  /* mark it as free in the bit array */
  mark_area_free (zone, area);

  gm_assert (area_is_free (zone, area));
  gm_assert (get_area_boundary (zone, area));

  /* Add marks to try to detect DMA corruption of the administrative fields */
  
#if GM_DEBUG
  gm_mark (zone->free_mark_set, &area->free_mark_1);
  gm_mark (zone->free_mark_set, &area->free_mark_2);
#endif
  
  /* add it to the head of free list for its size */

  area->next =  zone->free_list[size].next;
  area->prev =  &zone->free_list[size];
  area->next->prev = area;
  zone->free_list[size].next = area;
}

static gm_zone_area_t *
remove_from_free_list (gm_zone_t *zone, gm_zone_area_t *area)
{
  validate_free_area (zone, area);
  
  gm_assert (area);
  gm_assert (area_is_free (zone, area));
  
  /* remove the area from its free list */
  
  area->prev->next = area->next;
  area->next->prev = area->prev;

  return area;
}

static gm_zone_area_t *
reserve_free_area (gm_zone_t *zone, gm_zone_area_t *area)
{
  validate_free_area (zone, area);
  
  gm_assert (area_is_free (zone, area));
  
  /* remove the area from its free list */
  remove_from_free_list (zone, area);
 
  /* mark the area as reserved */
  mark_area_reserved (zone, area);

  gm_assert (area_is_reserved (zone, area));
   
  return area;
}


static gm_zone_area_t *
area_buddy (gm_zone_t *zone, gm_zone_area_t *area)
{
  gm_offset_t offset;
  int size;
  
  validate_free_area (zone, area);
  size = sizeof_area (zone, area);
  offset = GM_STATIC_CAST(unsigned int, (char *)area - (char *)zone->base);
  offset ^= 1 << size;
  return (gm_zone_area_t *) ((char *)zone->base + offset);
}

/**********************************************************************
 * Externally visable entry points follow
 **********************************************************************/

GM_ENTRY_POINT struct gm_zone *
gm_zone_create_zone (void *base, gm_size_t length)
{
  unsigned int size;
  gm_size_t bit_array_length;
  unsigned int i;
  gm_zone_t *zone;

  zone = gm_calloc (1, sizeof (*zone));
  if (!zone)
    goto abort_with_nothing;

  gm_assert (length >= sizeof (gm_zone_area_t) && power_of_two (length));
  gm_assert (sizeof (unsigned) == 4);
  
  zone->base = (gm_zone_area_t *) base;
  zone->length = length;
  size = (unsigned int) gm_log2_roundup (GM_STATIC_CAST (unsigned long, length));

  gm_assert ((unsigned)(1 << size) == length);
  
  gm_assert (GM_POWER_OF_TWO (sizeof (gm_zone_area_t)));
  bit_array_length =  length / sizeof (gm_zone_area_t) / sizeof (int);

  /* allocate space for the bit-arrays.  Allocate one buffer
     big enough for two, and split it.*/
  
  zone->boundary_bits = (gm_u32_t *) gm_calloc (sizeof(int),
						 2 * bit_array_length);
  if (! zone-> boundary_bits)
    goto abort_with_zone;
  zone->reserved_bits =  zone->boundary_bits + bit_array_length;

  /* Allocate a mark set for marking the boundaries of each zone aread */

  if (GM_DEBUG)
    {
      if (gm_create_mark_set (&zone->free_mark_set, 0) != GM_SUCCESS)
	{
	  goto abort_with_boundary_bits;
	}
    }
  
  if (GM_DEBUG)
    {
      if (gm_create_mark_set (&zone->user_mark_set, 0) != GM_SUCCESS)
	{
	  goto abort_with_free_mark_set;
	}
    }
  
  /* make all the free lists empty */
  for (i = 0; i < 32; i ++)
    zone-> free_list [i].next =  zone-> free_list [i].prev =
      & zone-> free_list [i];

  /* make one big area and free it */
  {
    gm_zone_area_t *the_area;

    the_area = (gm_zone_area_t *) base;
    mark_free_area (zone, the_area);
    set_area_boundary (zone, the_area);
    free_area (zone, the_area, size);

    gm_assert (sizeof_area (zone, the_area) ==  size);
    gm_assert (zone->free_list[size].next == base);
  }
  
  return zone;

 abort_with_free_mark_set:
  if (zone->free_mark_set)
    {
      gm_destroy_mark_set (zone->free_mark_set);
    }
 abort_with_boundary_bits:
  gm_free (zone->boundary_bits);
 abort_with_zone:
  gm_free (zone);
 abort_with_nothing:
  return 0;
}

GM_ENTRY_POINT void
gm_zone_destroy_zone (gm_zone_t *zone)
{
  gm_assert (zone && zone->boundary_bits);
  if (zone->user_mark_set)
    {
      gm_destroy_mark_set (zone->user_mark_set);
    }
  if (zone->free_mark_set)
    {
      gm_destroy_mark_set (zone->free_mark_set);
    }
  gm_free (zone->boundary_bits);
  gm_free (zone);
}

GM_ENTRY_POINT void *
gm_zone_free (gm_zone_t *zone, void *a)
{
  gm_zone_area_t *area, *buddy;
  unsigned int area_size, buddy_size, zone_size;
  
  area = GM_AREA_FOR_PTR (a);
  validate_user_area (zone, area);
  unmark_area (zone, area);
  mark_free_area (zone, area);
  
  area_size = sizeof_area (zone, area);
  
  gm_assert (area_is_reserved (zone, area));
  
  zone_size = (unsigned int) gm_log2_roundup (GM_STATIC_CAST (unsigned long,
					      zone->length));
  
  while (area_size < zone_size)
    {
      buddy = area_buddy (zone, area);
      gm_assert (area != buddy);
      buddy_size = sizeof_area (zone, buddy);
    
      if (area_is_free (zone,buddy) && buddy_size == area_size)
	{
      
	  /* merge area and buddy */
	  remove_from_free_list (zone, buddy);
	  
	  area_size ++;
      
	  if (area > buddy)
	    {
	      unmark_area (zone, area);
	      clear_area_boundary (zone, area);
	      gm_assert (!get_area_boundary (zone, area));
	      area = buddy;
	    }
	  else
	    {
	      unmark_area (zone, buddy);
	      clear_area_boundary (zone, buddy);
	      gm_assert (!get_area_boundary (zone, buddy));
	    }

	  if (area_size != sizeof_area (zone, area))
	    gm_assert (area_size == sizeof_area (zone,area));
      
	}
      else
	break;
    }
  free_area (zone, area, area_size);
  return (void *)0;
}

GM_ENTRY_POINT void *
gm_zone_malloc (gm_zone_t *zone, gm_size_t length)
{
  unsigned size,start;
  gm_zone_area_t *area;

  if (!length)
    return (void *)0;

  /* reserve space for the debugging marker at the start of the allocation */
  length += GM_ZONE_AREA_PTR_OFFSET;
  
  if (length < sizeof (gm_zone_area_t))
    length = sizeof (gm_zone_area_t);
  
  /* find the smallest size free area 
     big enough to hold length bytes. */
  
  start = (unsigned int) gm_log2_roundup (GM_STATIC_CAST (unsigned long, length));
  
  for (size = start; size <=31; size++)
    {
      if (zone->free_list[size].next != &zone->free_list[size])
	break;
    }
  
  /* If there is not a free block of the
     requested size, return NULL; */

  if (size == 32)
    return 0;
  
  /* Allocate a big enough buffer */

  area = zone->free_list[size].next;
  validate_free_area (zone, area);
  reserve_free_area (zone, zone->free_list[size].next);
   
  /* Free the high half of the buffer
     as many times as possible without
     making the buffer too small. */
  
  for (; --size >= start; )
    {
      gm_zone_area_t *right;
      gm_assert (GM_POWER_OF_TWO (sizeof (gm_zone_area_t)));
      right = (gm_zone_area_t *) (area + (1<<size)/sizeof (gm_zone_area_t));
      mark_free_area (zone, right);
      set_area_boundary (zone,right);
      gm_assert (get_area_boundary (zone, right));
    
      free_area (zone, right, size);
    }
  gm_assert (area_is_reserved (zone, area));
  gm_assert (get_area_boundary (zone, area));
  gm_assert (sizeof_area (zone, area)
	     == gm_log2_roundup (GM_STATIC_CAST (unsigned long, length)));

  /* Transfer the area to the user */

  unmark_area (zone, area);
  mark_user_area (zone, area);
  return GM_PTR_FOR_AREA (area);
}

GM_ENTRY_POINT void *
gm_zone_calloc (gm_zone_t *zone, gm_size_t count, gm_size_t length)
{
  void *ret;

  length *= count;
  ret = gm_zone_malloc (zone, length);
  if (ret)
    gm_bzero (ret, length);
  return ret;
}

GM_ENTRY_POINT int
gm_zone_addr_in_zone (gm_zone_t *zone, void *p)
{
  int ret;

  ret = ((gm_u8_t *)zone->base <= (gm_u8_t *)p
	 && (gm_u8_t *)p < (gm_u8_t *)zone->base + zone->length);
  return ret;
}
