/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_debug.h"
#include "gm_enable_directed_send.h"
#include "gm_internal.h"

#define GM_DEBUG_UNKNOWN 0

static
void
forge_recv (struct gm_port *p, gm_recv_event_t *e,
	    enum gm_recv_event_type type)
{
  /* Copy the message into the buffer and forge a recv event. */
  
  e->recv.type = gm_hton_u8 ((gm_u8_t) type);
  p->recv_queue_slot = (gm_recv_queue_slot_t *) (e + 1) - 1;
}


#if GM_ENABLE_DIRECTED_SEND
GM_ENTRY_POINT
void
gm_handle_directed_send_notification (struct gm_port *p, gm_recv_event_t *e)
{
  gm_u8_n_t *buffer, *scb, *ecb;
  gm_up_t start_misalignment, end_misalignment;
  int length;
  
  buffer = (void *) gm_ntoh_up (e->directed_send_notification.buffer);
  length = gm_ntoh_u32 (e->directed_send_notification.length);

  /* copy any misaligned start bytes */

  scb = ((gm_u8_n_t *)&e->directed_send_notification.start_copy_bytes[0]) - 1;
  start_misalignment = (gm_up_t) buffer & (GM_DMA_GRANULARITY-1);
  if (start_misalignment)
    {
      gm_u8_n_t *from, *to;
      
      from = scb + start_misalignment;
      to = buffer;
      if (start_misalignment + length < 2*GM_DMA_GRANULARITY)
	{
	  while (length--)
	    *to++ = *from++;
	  goto done_copying;
	}
      else
	{
	  while (start_misalignment++ < GM_DMA_GRANULARITY)
	    *to++ = *from++;
	}
    }

  /* copy the misaligned end bytes */

  ecb = e->directed_send_notification.end_copy_bytes;
  end_misalignment = (gm_up_t) (buffer + length) & (GM_DMA_GRANULARITY-1);
  if (end_misalignment)
    {
      gm_u8_n_t *from, *to;
      
      to = GM_DMA_ALIGN (ptr, buffer+length);
      from = ecb;
      while (to < buffer + length)
	*to++ = *from++;
    }
  
 done_copying:
  
  /* Recycle the directed receive token. */
  gm_provide_receive_buffer (p,
			     (void *) 1, /* any non-null value */
			     GM_DIRECTED_TAG_SIZE,
			     GM_LOW_PRIORITY);
}
#endif

gm_status_t
_gm_unknown (gm_port_t * p, gm_recv_event_t * e)
{
  switch (gm_ntoh_u8 (e->recv.type))
    {
    case GM_NO_RECV_EVENT:
      break;
      
    case GM_ALARM_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_ALARM_EVENT\n"));
      _gm_handle_alarm (p);
      break;

    case _GM_FLUSHED_ALARM_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("_GM_FLUSHED_ALARM_EVENT\n"));
      _gm_handle_flushed_alarm (p);
      break;

      /* When our sleep request has been
         granted, go to sleep.  If there has
         been any receive since the sleep
         was granted, the sleep ioctl will
         return immediately. */
    case _GM_SLEEP_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("_GM_SLEEP_EVENT\n"));
      p->sleep_pending = 0;
      {
	gm_status_t status;
	
	status = _gm_sleep (p);
	gm_assert ((status == GM_SUCCESS) || (status == GM_INTERRUPTED));
      }
      break;

      /* convert fancy receives into normal receives */

    case GM_FAST_RECV_EVENT:
    case GM_FAST_PEER_RECV_EVENT:
      gm_memorize_message ((void *)gm_ntoh_up (e->recv.message),
			   (void *)gm_ntoh_up (e->recv.buffer),
			   gm_ntoh_u32 (e->recv.length));
      e->recv.message = e->recv.buffer;
      /* fall through */
    case GM_PEER_RECV_EVENT:
      forge_recv (p, e, GM_NEW_RECV_EVENT);
      break;
      
    case GM_FAST_HIGH_RECV_EVENT:
    case GM_FAST_HIGH_PEER_RECV_EVENT:
      gm_memorize_message ((void *)gm_ntoh_up (e->recv.message),
			   (void *)gm_ntoh_up (e->recv.buffer),
			   gm_ntoh_u32 (e->recv.length));
      e->recv.message = e->recv.buffer;
      /* fall through */
    case GM_HIGH_PEER_RECV_EVENT:
      forge_recv (p, e, GM_NEW_HIGH_RECV_EVENT);
      break;

      /* Drop the received message and recycle the receive buffer. */

    case GM_RECV_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_RECV_EVENT\n"));
      gm_provide_receive_buffer (p, (void *)gm_ntoh_up (e->recv.buffer),
				 gm_ntoh_u8 (e->recv.size), GM_LOW_PRIORITY);
      break;
    case GM_HIGH_RECV_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_HIGH_RECV_EVENT\n"));
      gm_provide_receive_buffer (p, (void *)gm_ntoh_up (e->recv.buffer),
				 gm_ntoh_u8 (e->recv.size), GM_HIGH_PRIORITY);
      break;
      
#if 0
    case GM_FREE_SEND_TOKEN_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_FREE_SEND_TOKEN_EVENT\n"));
      GM_WARN (("client did not catch GM_FREE_SEND_TOKEN_EVENT\n"));
      break;

    case GM_FREE_HIGH_SEND_TOKEN_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_FREE_HIGH_SEND_TOKEN_EVENT\n"));
      GM_WARN (("client did not catch GM_FREE_HIGH_SEND_TOKEN_EVENT\n"));
      break;
#endif

#if GM_ENABLE_DIRECTED_SEND
    case _GM_DIRECTED_SEND_NOTIFICATION_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("_GM_DIRECTED_SEND_NOTIFICATION_EVENT\n"));
      gm_handle_directed_send_notification (p, e);
      break;

    case GM_DROPPED_SEND_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_DROPPED_SEND_EVENT\n"));
      gm_free_send_token
	(p, GM_SUBPORT_PRIORITY (gm_ntoh_u8 (e->failed_send.target_subport_id)));
      break;

#endif /* GM_ENABLE_DIRECTED_SEND */

    case GM_SENT_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_SENT_EVENT\n"));
      GM_WARN (("gm_unknown: broken gm program, got a SENT event.\n"));
      return GM_FAILURE;

    case GM_BAD_SEND_DETECTED_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_BAD_SEND_DETECTED_EVENT\n"));
      _GM_WARN (("Program passed bad send message to GM.\n"));
      _GM_WARN (("Port %d disabled.\n", p->id));
      return GM_FAILURE;

    case GM_BAD_RESEND_DETECTED_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_BAD_RESEND_DETECTED_EVENT\n"));
      _GM_WARN (("Program passed bad resend command to GM.\n"));
      _GM_WARN (("Port %d disabled.\n", p->id));
      return GM_FAILURE;

    case GM_SEND_TOKEN_VIOLATION_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_SEND_TOKEN_VIOLATION_EVENT\n"));
      GM_WARN (("Program passed send to LANai without send token.\n"));
      _GM_WARN (("Note: Such errors are not always detected.\n"));
      _GM_WARN (("Port %d disabled.\n", p->id));
      return GM_FAILURE;

    case GM_RECV_TOKEN_VIOLATION_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_RECV_TOKEN_VIOLATION_EVENT\n"));
      GM_WARN (("Program passed too many recv tokens to LANai.\n"));
      _GM_WARN (("Note: Such errors are not always detected.\n"));
      _GM_WARN (("Port %d disabled.\n", p->id));
      return GM_FAILURE;

    case GM_BAD_RECV_TOKEN_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_BAD_RECV_TOKEN_EVENT\n"));
      GM_WARN (("Program passed a bad recv token to the LANai.\n"));
      _GM_WARN (("Port %d disabled.\n", p->id));
      return GM_FAILURE;

    case GM_ALARM_VIOLATION_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_ALARM_VIOLATION_EVENT\n"));
      GM_WARN (("Program set an alarm when an alarm was already pending.\n"));
      _GM_WARN (("Port %d disabled.\n", p->id));
      return GM_FAILURE;

    case GM_ORPHANED_SEND_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_ORPHANED_SEND_EVENT\n"));
      {
        int this_node=-1;
        gm_get_node_id (p, (unsigned int *)&this_node);
        GM_WARN (("GM_ORPHANED_SEND_EVENT: node %d:\n", this_node));
	_GM_WARN (("    Could not deliver pkt to node %d port = %d\n",
		   gm_ntoh_u16 (e->failed_send.target_node_id),
		   GM_SUBPORT_PORT (gm_ntoh_u8
				    (e->failed_send.target_subport_id))));
	return GM_FAILURE;
      }

    case GM_REJECTED_SEND_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_REJECTED_SEND_EVENT\n"));
      GM_WARN (("Peer rejected packet.\n"));
      return GM_FAILURE;

    case GM_BAD_RECV_VMA_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_BAD_RECV_VMA_EVENT\n"));
      goto bad_vma;
    case GM_BAD_SEND_VMA_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_BAD_SEND_VMA_EVENT\n"));
    bad_vma:
      GM_WARN (("User attempted to %s non-DMAable memory.\n",
		gm_ntoh_u8 (e->recv.type) == GM_BAD_SEND_VMA_EVENT
		? "send from"
		: "recv to"));
      return GM_FAILURE;

    case GM_SENT_TOKENS_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_SENT_TOKENS_EVENT\n"));
      /* Call the callbacks for all the completed sends and build a list
	 of completed non-callback sends. */

      p->sent_list_slot = &p->sent_list[0];
      gm_handle_sent_tokens (p, e);
      *p->sent_list_slot = gm_hton_up (0);

      /* If any non-callback send completed, forge a GM_SENT_EVENT. */

      if (gm_ntoh_up (p->sent_list[0]))
	{
	  GM_PRINT (GM_DEBUG_UNKNOWN,
			 ("Forging GM_NEW_SENT_EVENT.\n"));
	  e->sent.message_list = gm_hton_up ((gm_up_t) p->sent_list);
	  e->sent.type = gm_hton_u8 (GM_NEW_SENT_EVENT);
	  p->recv_queue_slot = (gm_recv_queue_slot_t *) (e + 1) - 1;
	}
      return GM_SUCCESS;

    case GM_IGNORE_RECV_EVENT:
      GM_PRINT (GM_DEBUG_UNKNOWN, ("GM_IGNORE_RECV_EVENT\n"));
      return GM_SUCCESS;

    default:
      GM_WARN (("Program passed bad type %d to gm_unknown( ).\n",
	       (int) gm_ntoh_u8 (e->recv.type)));
      return GM_FAILURE;
    }
  return GM_SUCCESS;
}

GM_ENTRY_POINT void
gm_unknown (gm_port_t * p, gm_recv_event_t * e)
{
  GM_PRINT (0, ("gm_unknown got event type %u\n",
		     gm_ntoh_u8 (e->recv.type)));

  if (_gm_unknown (p, e) != GM_SUCCESS) {
    /* read the lanai_globals into user space so it will be saved in the dump */
    _gm_preserve_lanai_globals(p);
    GM_PANIC (("\n"));
  }
}

/*
 * never call this explicitly - if GM_DEBUG_BUFFERS is defined, then
 * gm_unknown is automatically redefined to point here
 */
GM_ENTRY_POINT void
_gm_unknown_debug_buffers (gm_port_t * p, gm_recv_event_t * e)
{
  switch (gm_ntoh_u8 (e->recv.type))
    {
      /* Aborting here is sort of iffy - we'll assume that messages
	 are uncaught because of programmer error.  They could also
	 come from a malicious user in which case we just want to
	 drop them. */

    case GM_RECV_EVENT:
    case GM_FAST_RECV_EVENT:
    case GM_PEER_RECV_EVENT:
    case GM_FAST_PEER_RECV_EVENT:
      GM_WARN (("gm_unknown: uncaught low priority receive\n"));
      gm_abort ();
      break;

    case GM_HIGH_RECV_EVENT:
    case GM_FAST_HIGH_RECV_EVENT:
    case GM_HIGH_PEER_RECV_EVENT:
    case GM_FAST_HIGH_PEER_RECV_EVENT:
      GM_WARN (("gm_unknown: uncaught high priority receive\n"));
      gm_abort ();
      break;

    default:
      gm_unknown (p, e);
    }
}
