/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_internal.h"

#define GM_DEBUG_UNIQUE_ID_TO_NODE_ID 0

#if GM_DEBUG_UNIQUE_ID_TO_NODE_ID
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

gm_status_t
gm_unique_id_to_node_id (gm_port_t *port, char *unique,
			 unsigned int *n)
{
#if GM_KERNEL
  unsigned int node_id, *node_id_ptr;
  gm_unique_id_64_t uid;
  gm_port_state_t *ps;
  gm_instance_state_t *is;
  static struct gm_hash *hash;

  GM_CALLED ();
  
  ps = port->kernel_port_state;
  gm_assert (ps);
  is = ps->instance;
  gm_assert (is);
  hash = is->ethernet.addr_tab.cache_hash;
  gm_assert (hash);

  /* Copy in the unique ID */

  /* Clear the unused part of the unique ID. */
  uid.as_unswapped_u64 = 0;

  gm_copyin (ps, unique, &uid, 6);

  /* attempt to fetch and verify a cached translation */
     
  GM_PRINT (GM_DEBUG_UNIQUE_ID_TO_NODE_ID, ("Fetching Cached translation.\n"));
  node_id_ptr
    = (unsigned int *) gm_hash_find (is->ethernet.addr_tab.cache_hash,
				     &uid);
  if (node_id_ptr)
    {
      /* verify cached value is up-to-date */
      node_id = *node_id_ptr;
      if (uid.as_unswapped_u64 != (is->ethernet.addr_tab
				   .volatile_ethernet_addr[node_id]
				   .as_unswapped_u64))
	{
	  gm_hash_remove (hash, &uid);
	  node_id_ptr = 0;
	}
    }

  /* if no cached value was found, attempt to find and cache a translation */

  if (node_id_ptr)
    node_id = *node_id_ptr;
  else
    {
      GM_PRINT (GM_DEBUG_UNIQUE_ID_TO_NODE_ID,
		("Scanning for translation.\n"));
      for (node_id = 1; node_id < is->max_node_id; node_id++)
	{
	  if (uid.as_unswapped_u64 == (is->ethernet.addr_tab
				       .volatile_ethernet_addr[node_id]
				       .as_unswapped_u64))
	    {
	      gm_hash_insert (hash, &uid, &node_id); /* try to cache */
	      goto success;
	    }
	}

      /* Failed */

      if (GM_DEBUG_UNIQUE_ID_TO_NODE_ID)
	{
	  GM_PRINT (GM_DEBUG_UNIQUE_ID_TO_NODE_ID,
		    ("No translation found.\n"));
	  gm_pause_lanai (is);
	  for (node_id = 1; node_id < is->max_node_id; node_id++)
	    {
	      gm_u8_n_t *c;
	      if (0 == (is->ethernet.addr_tab
			.volatile_ethernet_addr[node_id].as_unswapped_u64))
		{
		  continue;
		}
	      c = (is->ethernet.addr_tab.volatile_ethernet_addr[node_id]
		   .as_bytes);
	      GM_PRINT (GM_DEBUG_UNIQUE_ID_TO_NODE_ID,
			("%d %02x:%02x:%02x:%02x:%02x:%02x\n",
					node_id,
					gm_ntoh_u8 (c[0]), gm_ntoh_u8 (c[1]),
					gm_ntoh_u8 (c[2]), gm_ntoh_u8 (c[3]),
					gm_ntoh_u8 (c[4]), gm_ntoh_u8 (c[5])));
	    }
	  gm_unpause_lanai (is);
	}
      GM_RETURN_STATUS (GM_FAILURE);
    }
 success:
  GM_PRINT (GM_DEBUG_UNIQUE_ID_TO_NODE_ID, ("Found translation.\n"));
  gm_copyout (ps, &node_id, n, sizeof (n));
  GM_RETURN_STATUS (GM_SUCCESS);
#else  /* !GM_KERNEL */
  gm_status_t status;
  
  union 
  {
    char unique[6];
    unsigned node;
  } both;

  gm_bcopy (unique, both.unique, 6);
  status = _gm_user_ioctl (port, GM_UNIQUE_ID_TO_NODE_ID, &both,
			   sizeof (both));
  if (status != GM_SUCCESS)
    return status;
  *n = both.node;
  return status;
#endif /* !GM_KERNEL */
}
