#include "gm_internal.h"

#include "gm_tick.h"

#if GM_CPU_alpha && GM_ENABLE_TRACE

#if !GM_KERNEL
#include <stdlib.h>
#include <sys/time.h>
#else
#include <asm/hwrpb.h>
#endif

#define TICK_LOWBITS ((1UL << 32)-1)
#define TICK_HIGHBITS (~((1UL << 32)-1))

static gm_inline unsigned tick32(void)
{
  gm_u32_t t;
  __asm__ volatile("rpcc %0" : "=r" (t));
  return t;
}

#if GM_KERNEL
#define gettimeofday(a,b) do_gettimeofday(a)
#endif

static gm_u64_t cycles_per_ms;
gm_u32_t gm_tick_high;
gm_u32_t gm_tick_last=~0;/* force first rdtsc to go in recompute */
static int rounding_mode = -1;
static int ref_time = 0;

void gm_tick_init(void)
{
  gm_u64_t ref_dist;
  gm_s64_t residual;
  gm_u32_t cc;
  char *s;
  struct timeval t;

  if (!cycles_per_ms) {
#if GM_KERNEL
    cycles_per_ms = hwrpb->cycle_freq/1000;
#else
    if (!(s = getenv("GM_CYCLE_FREQ"))) {
      GM_PANIC (("no GM_CYCLE_FREQ specified\n"));
    }
    cycles_per_ms = atol(s)/1000;
#endif
    GM_PRINT (1, ("hw counter cycle freq (per msec)=%ld\n",cycles_per_ms));
    if (!cycles_per_ms) {
      GM_PRINT (1, ("faking cycle freq to 500Mhz\n"));
      cycles_per_ms = 500000;
    }
  }
  /* ROUNDING : 0 means start time reference low 32 bit should be
     close to 0, 1 means should be close to of (1<<31), this info may
     be necessary to solve border case of the equation converting the
     imprecise gettimeofday to hw cycles */
#if !GM_KERNEL
  if (rounding_mode == -1 && (s = getenv("GM_TICK_ROUNDING"))) {
    rounding_mode = atoi(s);
  }
#endif
  /* get time with both imprecise and precise counter */
  gettimeofday(&t,0);
  cc = tick32();
  /* origin point is the previous point with seconds a multiple of 65536 (bug once a day) */
  if (!ref_time)
    ref_time = t.tv_sec - (t.tv_sec % 65536);
  ref_dist = (t.tv_sec - ref_time)*1000*cycles_per_ms+(t.tv_usec*cycles_per_ms)/1000;
  
  residual = (gm_s64_t)cc - (ref_dist & TICK_LOWBITS);
  if (rounding_mode == -1) {
    if (residual < -(3L<<30)) {
      gm_tick_high = (ref_dist >> 32) + 1;
      rounding_mode = 0;
    } else if (residual < -(1L<<30)) {
      GM_PRINT (1, ("gm_tick_init: round away from 0\n"));
      gm_tick_high = (ref_dist>> 32) + 1;
      rounding_mode = 1;
    } else if (residual < (1L << 30)) {
      gm_tick_high = (ref_dist>> 32) + 0;
      rounding_mode = 0;
    } else if (residual < (3L << 30)) {
      GM_PRINT (1, ("gm_tick_init: round away from 0\n"));
      gm_tick_high = (ref_dist>> 32) + 0;
      rounding_mode = 1;
    } else {
      gm_tick_high = (ref_dist>> 32) - 1;
      rounding_mode = 0;
    }
  } else if (rounding_mode == 0) {
    if (residual < -(2L<<30)) {
      gm_tick_high = (ref_dist>> 32) + 1;
    } else if (residual < (2L << 30)) {
      gm_tick_high = (ref_dist>> 32) + 0;
    } else {
      gm_tick_high = (ref_dist>> 32) - 1;
    }
  } else { /* rounding_mode >= 0 */     
    if (residual < 0) {
      gm_tick_high = (ref_dist>> 32) + 1;
    } else {
      gm_tick_high = (ref_dist>> 32) + 0;
    }
  }
  gm_tick_last = cc;
}

gm_tick_t gm_tick_recompute(void)
{
  gm_u32_t cc = tick32();
  gm_u32_t diff = cc - gm_tick_last;
  if (cc < gm_tick_last && diff < (1 << (32-8)) && ref_time) {
    gm_tick_last = cc;
    gm_tick_high += 1;
    GM_PRINT (1, ("gm_tick wraparound\n"));
  } else {
    if (ref_time && diff > (1 << (32-4)))
      GM_PRINT (1, ("long delay between gm_tick\n"));
    gm_tick_init(); 
  }
  return gm_tick_last + ((gm_u64_t)gm_tick_high << 32);
}

#endif
