/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#ifdef WIN32
#include <stdio.h>
#endif

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_enable_fast_small_send.h"
#include "gm_enable_galvantech_workaround.h"
#include "gm_enable_trace.h"
#include "gm_internal.h"
#include "gm_send_queue.h"

#define GM_DEBUG_SEND 0

#if GM_DEBUG_SEND
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

static unsigned int gm_sent_count = 0;

GM_ENTRY_POINT
void
gm_send_with_callback (gm_port_t *p, void *message, unsigned int size,
		       gm_size_t len, unsigned int priority,
		       unsigned int target_node_id,
		       unsigned int target_port_id,
		       gm_send_completion_callback_t callback,
		       void *context)
{
  GM_CALLED ();

  GM_LOG_EVT(GM_SEND_EVENT);

  gm_assert (p);
  gm_assert (message);
  gm_assert (gm_min_size_for_length (len) <= size);
  gm_assert (size >= GM_MIN_MESSAGE_SIZE && size <= 32);
  gm_assert (priority == GM_LOW_PRIORITY ||
	       priority == GM_HIGH_PRIORITY);
  
#if GM_DEBUG_SEND
  gm_printf ("gm_send_with_callback "
	     "(%p, %p, %d, 0x%lx, %s, %d, %d, %p, %p) called.\n",
	     p, message, size, len,
	     priority ? "GM_HIGH_PRIORITY" : "GM_LOW_PRIORITY",
	     target_node_id, target_port_id, callback, context);
#if 0
  {
    unsigned int i;
    for (i=0; i<len; i++)
      {
	gm_printf ("%02x", ((gm_s8_t *) message)[i]);
	if ((i & 0x7) == 0x7)
	  gm_printf ("\n");
	else if ((i & 0x3) == 0x3)
	  gm_printf (" ");
      }
    if ((i & 0x7) != 0x7)
      gm_printf ("\n");
  }
#endif
#endif

#if GM_DEBUG_BUFFERS
  {
    struct gm_buf_handle *bh = gm_find_buf (message);
    if (!bh) {
      fprintf (stderr, "gm_send: got unregistered buffer %p\n", message);
      abort();
    }
    if (bh->size != -1 && bh->size != (int) size) {
      fprintf (stderr, "gm_send: buffer %p initialized with size %d, arrived with size %d\n",
	       message, bh->size, size);
      abort();
    }
    if (bh->status != gm_in_app) {
      fprintf (stderr, "gm_send: buffer %p should have status %s not %s\n",
	       message,
	       gm_get_buf_status_name(gm_in_app),
	       gm_get_buf_status_name(bh->status));
      abort();
    }
    bh->status = gm_in_send;
  }
#endif /* GM_DEBUG_BUFFERS */

  GM_SIMULATE_SIMULATE ( );

  /* Pass send token to LANai.  Care must be taken to flush the message to
     memory before flushing the token, and the token should be flushed to
     memory immediately. */

  {
    struct gm_send_send_event volatile *se;
    struct gm_send_send_event batch_write;

    se = GM_SEND_QUEUE_SLOT (p, callback, context, send);

    if (GM_STRUCT_WRITE_COMBINING)
      {
	GM_PRINT (GM_PRINT_LEVEL >= 8,
		  ("gm_send (%p, %p, %d, 0x%lx, %s, %d, %d) called.\n",
		   p, message, size, len,
		   priority ? "GM_HIGH_PRIORITY" : "GM_LOW_PRIORITY",
		   target_node_id, target_port_id));
	GM_PRINT (GM_PRINT_LEVEL >= 8,
		  ("gm_send: port = %p  send = %p\n", p, se));
	
#if GM_ENABLE_GALVANTECH_WORKAROUND
	batch_write.ip_checksum = gm_htons (_gm_ip_checksum (message, len));
#endif
	batch_write.message = gm_hton_up ((gm_up_t) message);
	batch_write.length = gm_hton_u32 ((gm_u32_t) len);
	batch_write.target_node_id = gm_hton_u16 ((gm_u16_t) target_node_id);
	batch_write.target_subport_id
	  = gm_hton_u8 ((gm_u8_t) GM_SUBPORT (priority, target_port_id));
	batch_write.type = gm_hton_u8 ((gm_u8_t) (GM_SEND_EVENT_0 + size));

	if (GM_FAST_SMALL_SEND && len <= GM_FAST_SEND_LEN)
	  {
	    unsigned int i;
	    volatile gm_ptr_t *data;

	    data = (volatile gm_ptr_t *) &se->message;
	    data -= len / sizeof (gm_ptr_t);
	    for (i=0; i<len/GM_SIZEOF_VOID_P; i++)
	      {
		data[i] = ((gm_ptr_t *) message)[i];
	      }
	  }

	GM_COPY_TO_IO_SPACE (*se, batch_write);
      }
    else
      {
#if GM_ENABLE_GALVANTECH_WORKAROUND
	se->ip_checksum = gm_htons (_gm_ip_checksum (message, len));
#endif
	se->message = gm_hton_up ((gm_up_t) message);
	se->target_node_id = gm_hton_u16 ((gm_u16_t) target_node_id);
	se->length = gm_hton_u32 ((gm_u32_t) len);
	se->target_subport_id =
	  gm_hton_u8 ((gm_u8_t) GM_SUBPORT (priority, target_port_id));
	GM_STBAR ();
	se->type = gm_hton_u8 ((gm_u8_t) (GM_SEND_EVENT_0 + size));
      }
    GM_FLUSH_SEND_EVENT (p, se);
  }
  GM_RETURN_NOTHING ();
}

/* Handle legacy-style send completion by building a send completion
   list to be passed to the user in a GM_SENT_EVENT. */

void
_gm_sent (struct gm_port *port, void *context, gm_status_t status)
{
  gm_u32_t my_id=0;
  gm_u8_t my_name[GM_MAX_HOST_NAME_LEN+1]={0};

  GM_CALLED_WITH_ARGS (("%p,%p,%d", port, context, status));

  gm_sent_count++;

  if (status != GM_SUCCESS)
    {
      (void)gm_get_node_id(port,&my_id);
      (void)gm_get_host_name(port, (char *)my_name);
      GM_WARN (("_gm_sent: gmID=%d '%s' msg_cnt=%d send failed to complete"
		" (see kernel log for details)", my_id, my_name,gm_sent_count));
      gm_perror ("send failed", status);
#if !GM_KERNEL
      gm_exit (GM_FAILURE);
#endif /* GM_KERNEL */
    }
  
  /* Append the sent packet to the list of sent packets that will be used
     shortly to forge a GM_SENT_EVENT in _gm_unknown(). */
  *port->sent_list_slot++ = gm_hton_up ((gm_up_t) context);
  GM_LOG_EVT(GM_AFTER_SEND_EVENT);
  GM_RETURN_NOTHING ();
}

/* Legacy send function.  Deprecated. */

GM_ENTRY_POINT
void
gm_send (gm_port_t *p, void *message, unsigned int size,
	 gm_size_t len, unsigned int priority,
	 unsigned int target_node_id,
	 unsigned int target_port_id)
{
  gm_send_with_callback (p, message, size, len, priority, target_node_id,
			 target_port_id, _gm_sent, message);
}
