/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_debug_recv_tokens.h"
#include "gm_enable_galvantech_workaround.h"
#include "gm_enable_trace.h"
#include "gm_internal.h"
#include "gm_tick.h"

#define GM_DEBUG_RECEIVE 0

#if GM_DEBUG_RECEIVE
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

GM_ENTRY_POINT
gm_recv_event_t *
gm_receive (gm_port_t * p)
{
  int type;
  gm_recv_event_t *ret = 0;
  static struct gm_recv no_recv;
#if GM_ENABLE_TRACE
  gm_tick_t stamp;
#endif
  
  GM_CALLED ();

#if GM_ENABLE_TRACE
  stamp = gm_tick();
#endif
  
  gm_assert (GM_DMA_ALIGNED (p->recv_queue_slot));

  type = gm_ntoh_u8 (p->recv_queue_slot->event.recv.type);
  if ((unsigned int) type < 128)
    {
#if !GM_CPU_x86 && !GM_CPU_sparc
#  if GM_KERNEL
      /* BAD: No need for this on systems with coherent I/O */
      GM_ARCH_RECV_QUEUE_UPDATE (p);
#  elif defined WIN32
      /* should not need need sync */
#  elif 0
      gm_u32_t slot_num = p->recv_queue_slot - p->recv_queue_start;
      _gm_user_ioctl (p, GM_RECV_QUEUE_UPDATE, &slot_num,
		      sizeof (slot_num));
#  endif
      type = gm_ntoh_u8 (p->recv_queue_slot->event.recv.type);
      if ((unsigned int) type < 128)
	{
	  GM_RETURN_PTR ((gm_recv_event_t *) & no_recv);
	}
#else
      GM_RETURN_PTR ((gm_recv_event_t *) & no_recv);
#endif
    }

  ret = (gm_recv_event_t *) & p->recv_queue_slot->event;
  ret->recv.type = gm_hton_u8 ((gm_u8_t) (type - 128));

  if (++p->recv_queue_slot >= p->recv_queue_limit)
    p->recv_queue_slot = p->recv_queue_start;

  /* debugging code */

  if (GM_DEBUG_RECV_TOKENS)
    {
      switch (type)
	{
	case GM_NEW_RECV_EVENT:
	case GM_NEW_PEER_RECV_EVENT:
	case GM_NEW_FAST_RECV_EVENT:
	case GM_NEW_FAST_PEER_RECV_EVENT:
	  GM_PRINT (GM_DEBUG_RECV_TOKENS,
		    ("gm_debug_recv_token_cnt[%d][0x%x] = 0x%x\n",
		     GM_LOW_PRIORITY,
		     (int) gm_ntoh_u8 (ret->recv.size),
		     --(gm_debug_recv_token_cnt
			[GM_LOW_PRIORITY]
			[gm_ntoh_u8 (ret->recv.size)])));
	  break;
	case GM_NEW_HIGH_RECV_EVENT:
	case GM_NEW_HIGH_PEER_RECV_EVENT:
	case GM_NEW_FAST_HIGH_RECV_EVENT:
	case GM_NEW_FAST_HIGH_PEER_RECV_EVENT:
	  GM_PRINT (GM_DEBUG_RECV_TOKENS,
		    ("gm_debug_recv_token_cnt[%d][0x%x] = 0x%x\n",
		     GM_HIGH_PRIORITY,
		     (int) gm_ntoh_u8 (ret->recv.size),
		     --(gm_debug_recv_token_cnt
			[GM_HIGH_PRIORITY]
			[gm_ntoh_u8 (ret->recv.size)])));
	  break;

	default:
	  break;
	}
    }

#if GM_ENABLE_GALVANTECH_WORKAROUND
  
  switch (ret->recv.type)
    {
    case GM_HIGH_PEER_RECV_EVENT:
    case GM_HIGH_RECV_EVENT:
    case GM_PEER_RECV_EVENT:
    case GM_RECV_EVENT:
      ret->recv.message = ret->recv.buffer;
    case GM_FAST_HIGH_PEER_RECV_EVENT:
    case GM_FAST_HIGH_RECV_EVENT:
    case GM_FAST_PEER_RECV_EVENT:
    case GM_FAST_RECV_EVENT:
      {
	if (_gm_ip_checksum_verify (gm_ntoh_up (ret->recv.message),
				    gm_ntohl (ret->recv.length),
				    gm_ntohs (ret->recv.ip_checksum))
	    != GM_SUCCESS)
	  {
	    gm_u8_t my_name[GM_MAX_HOST_NAME_LEN+1]={0};
	    gm_get_host_name(p, (char *)my_name);

	    GM_WARN (("*** at '%s' bad recv cksum (0x%04x) len=%d from gmID=%d '%s'\n",
		      my_name,
		      gm_ntohs (ret->recv.ip_checksum),
		      gm_ntohl (ret->recv.length),
		      gm_ntohs (ret->recv.sender_node_id),
		      gm_node_id_to_host_name(p,gm_ntohs (ret->recv.sender_node_id))));
	    
	    gm_always_assert (0);
	  }
      }
      break;
      
    case GM_RAW_RECV_EVENT:
      GM_WARN (("*** Not checking raw recv checksum (yet).\n"));
      break;
      
    default:
      break;
    }
#endif /* GM_ENABLE_GALVANTECH_WORKAROUND */

  gm_assert (GM_NUM_RECV_EVENT_TYPES <= 256);
  GM_PRINT (GM_DEBUG_RECEIVE,
		 ("received a %s\n",
		  _gm_recv_event_name ((enum gm_recv_event_type)
				       gm_ntoh_u8 (ret->recv.type))));

  GM_LOG_STAMPED_EVT(GM_RECEIVE_EVENT,stamp,0,0);
  GM_LOG_EVT(type);
  GM_RETURN_PTR (ret);
}

#if GM_DEBUG_BUFFERS
GM_ENTRY_POINT
gm_recv_event_t *
gm_receive_debug_buffers (gm_port_t *port)
{
  gm_recv_event_t *ret;

  ret = gm_receive (port);
  switch (ret->recv.type)
    {
    case GM_SENT_EVENT:
      {
	void **pp;
	void *p;
	struct gm_buf_handle *bh;
	  
	pp = (void **) gm_ntoh_hp (ret->sent.message_list);
	gm_always_assert (pp);
	do {
	  p = gm_ntoh_hp (*pp);
	  gm_always_assert (p);
	  bh = _gm_find_buf (p);
	  if (!bh) {
	    GM_NOTE (
		     ("gm_receive: got unregistered buffer %p from a sent event\n",
		      p));
	    abort();
	  }
	  if (bh->status != gm_in_send) {
	    GM_NOTE (("gm_receive: got buffer in state "));
	    _GM_NOTE (("%s from a sent event, expected gm_in_send\n",
		       gm_get_buf_status_name (bh->status)));
	    abort();
	  }
	  bh->status = gm_in_app;
	  pp++;
	}
	while (gm_ntoh_hp(*pp));
	break;
      }

    case GM_HIGH_RECV_EVENT:
    case GM_FAST_HIGH_RECV_EVENT:
    case GM_RECV_EVENT:
    case GM_FAST_RECV_EVENT:
    case GM_HIGH_PEER_RECV_EVENT:
    case GM_FAST_HIGH_PEER_RECV_EVENT:
    case GM_PEER_RECV_EVENT:
    case GM_FAST_PEER_RECV_EVENT:
    case GM_RAW_RECV_EVENT:
      {
	void *p;
	struct gm_buf_handle *bh;
	int size = gm_ntoh_u8 (ret->recv.size);

	p = gm_ntoh_hp (ret->recv.buffer);
	gm_always_assert (p);
	bh = _gm_find_buf (p);
	gm_always_assert (bh);
	gm_always_assert (bh->status == gm_in_recv);
	gm_always_assert (bh->size == -1 || bh->size == size);
	bh->status = gm_in_app;
	break;
      }
    default:
      break; /* g++ gets a parse error w/o this */
    }
  return ret;
}

#endif
