/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_internal.h"
#include "gm_enable_pio_sends.h"
#include "gm_enable_datagrams.h"
#include "gm_enable_trace.h"

#define GM_DEBUG_SEND 0

#if GM_DEBUG_SEND
#undef GM_LOCALLY_ENABLE_CALL_TRACE
#define GM_LOCALLY_ENABLE_CALL_TRACE 1
#endif

GM_ENTRY_POINT
void
gm_datagram_send (gm_port_t *p, void *message, unsigned int size,
		  unsigned long len, unsigned int priority,
		  unsigned int target_node_id,
		  unsigned int target_port_id,
		  gm_send_completion_callback_t callback,
		  void *context)
{
  GM_CALLED ();

  GM_LOG_EVT (GM_PIO_DATAGRAM_SEND_EVENT);

  gm_assert (p);
  gm_assert (message);
  gm_assert (gm_min_size_for_length (len) <= size);
  /* gm_assert (GM_MIN_PRIORITY <= priority); */
  gm_assert (size >= GM_MIN_MESSAGE_SIZE && size <= 32);
  gm_assert (priority == GM_LOW_PRIORITY ||
	       priority == GM_HIGH_PRIORITY);
  gm_assert (len <= GM_MTU);
  
#if GM_DEBUG_SEND
  gm_printf ("gm_datagram_send_with_callback "
	     "(%p, %p, %d, 0x%lx, %s, %d, %d, %p, %p) called.\n",
	     p, message, size, len,
	     priority ? "GM_HIGH_PRIORITY" : "GM_LOW_PRIORITY",
	     target_node_id, target_port_id, callback, context);
#endif

#if GM_DEBUG_BUFFERS
  {
    struct gm_buf_handle *bh = gm_find_buf (message);
    if (!bh) {
      fprintf (stderr, "gm_send: got unregistered buffer %p\n", message);
      abort();
    }
    if (bh->size != -1 && bh->size != (int) size) {
      fprintf (stderr, "gm_send: buffer %p initialized with size %d, arrived with size %d\n",
	       message, bh->size, size);
      abort();
    }
    if (bh->status != gm_in_app) {
      fprintf (stderr, "gm_send: buffer %p should have status %s not %s\n",
	       message,
	       gm_get_buf_status_name(gm_in_app),
	       gm_get_buf_status_name(bh->status));
      abort();
    }
    bh->status = gm_in_send;
  }
#endif /* GM_DEBUG_BUFFERS */

  GM_SIMULATE_SIMULATE ( );

  /* Pass send token to LANai.  Care must be taken to flush the message to
     memory before flushing the token, and the token should be flushed to
     memory immediately. */

  {
    struct gm_send_send_event volatile *se;
    struct gm_send_send_event batch_write;

    se = GM_SEND_QUEUE_SLOT (p, callback, context, send);

    if (GM_STRUCT_WRITE_COMBINING)
      {
	GM_PRINT (GM_PRINT_LEVEL >= 8,
		  ("gm_send (%p, %p, %d, 0x%lx, %s, %d, %d) called.\n",
		   p, message, size, len,
		   priority ? "GM_HIGH_PRIORITY" : "GM_LOW_PRIORITY",
		   target_node_id, target_port_id));
	GM_PRINT (GM_PRINT_LEVEL >= 8, ("gm_send: port = %p  send = %p\n",
					p, se));
	
	batch_write.message = gm_hton_up(message);
	batch_write.length = gm_htonl (len);
	batch_write.target_node_id = gm_htons ((gm_u16_t) target_node_id);
	batch_write.target_subport_id
	  = gm_htonc (GM_SUBPORT (priority, target_port_id));
	batch_write.type = gm_htonc (GM_DATAGRAM_SEND_EVENT_0 + size);

	if (GM_FAST_SMALL_SEND && len <= GM_FAST_SEND_LEN)
	  {
	    unsigned int i;
	    volatile gm_ptr_t *data;

	    data = (volatile gm_ptr_t *) &se->message;
	    data -= len / sizeof (gm_ptr_t);
	    for (i=0; i<len/GM_SIZEOF_VOID_P; i++)
	      {
		data[i] = ((gm_ptr_t *) message)[i];
	      }
	  }

	GM_COPY_TO_IO_SPACE (*se, batch_write);
      }
    else
      {
	se->message = gm_hton_up (message);
	se->target_node_id = gm_htons ((gm_u16_t) target_node_id);
	se->length = gm_htonl (len);
	se->target_subport_id =
	  gm_htonc (GM_SUBPORT (priority, target_port_id));
	GM_STBAR ();
	se->type = gm_htonc (GM_DATAGRAM_SEND_EVENT_0 + size);
      }
    GM_FLUSH_SEND_EVENT (p, se);
  }
  GM_RETURN_NOTHING ();
}
