/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

/* author: glenn@myri.com */

/* This modules implements a platform-independent page allocation
   interface, with automatic initialization and finalization for use
   in the kernel.

   In some cases, this file uses platform-specific features to
   implement page allocation efficiently, but by default this module
   allocates blocks of pages and keeps a page free list to recycle
   freed pages, but the module never frees the blocks of pages until
   all pages have been freed.  */

#include "gm_call_trace.h"
#include "gm_debug.h"
#include "gm_internal.h"

struct gm_free_page
{
  struct gm_free_page *next;
};

struct gm_page_allocation_record
{
  struct gm_page_allocation_record *next;
  void *buf;
};

static struct gm_free_page *_gm_first_free_page;
static struct gm_page_allocation_record *_gm_first_page_allocation_record;
static unsigned long _gm_page_allocation_cnt;

/* Allocate a page-aligned page. */

GM_ENTRY_POINT void *
gm_page_alloc ()
{
  struct gm_free_page *page;

#if GM_KERNEL && GM_KERNEL_MALLOC_ALIGNS_PAGES
  page = (struct gm_free_page *) gm_malloc (GM_PAGE_LEN);
#elif !GM_KERNEL && GM_HAVE_MEMALIGN
  page = (struct gm_free_page *) memalign (GM_PAGE_LEN, GM_PAGE_LEN);
#else
  if (_gm_first_free_page)
    {
      /* Return a page in the free list. */

      page = _gm_first_free_page;
      _gm_first_free_page = _gm_first_free_page->next;
    }
  else
    {
      int i;
      void *temp;
      struct gm_page_allocation_record *rec;

      /* Allocate more free pages, plus room for the allocation record. */

      if (!GM_PAGE_LEN)
	{
	  GM_NOTE (("gm_alloc_page: do not know GM_PAGE_LEN.\n"));
	  return 0;
	}
      temp = gm_malloc (33 * GM_PAGE_LEN - 1);
      if (!temp)
	return 0;

      /* Determine location of first new page. */

      if (((unsigned long) temp & (GM_PAGE_LEN - 1)) == 0)
	page = temp;
      else
	page = (struct gm_free_page *)
	  ((char *) temp + GM_PAGE_LEN - ((unsigned long) temp & (GM_PAGE_LEN - 1)));

      /* Determine where to store allocation record, utilizing the
         otherwised wasted space allocated to ensure we could align
         32 pages. */

      if ((gm_size_t) ((char *) page - (char *) temp) >= sizeof (*rec))
	rec = (struct gm_page_allocation_record *) temp;
      else
	rec = ((struct gm_page_allocation_record *)
	       ((char *) page + 32 * GM_PAGE_LEN));

      /* Record allocation */

      rec->buf = temp;
      rec->next = _gm_first_page_allocation_record;
      _gm_first_page_allocation_record = rec;

      /* Add 31 of the 32 new free pages to the free list. */

      for (i = 0; i < 31; i++)
	{
	  page->next = _gm_first_free_page;
	  _gm_first_free_page = page;
	  page = (struct gm_free_page *) ((char *) page + GM_PAGE_LEN);
	}
    }
  _gm_page_allocation_cnt++;

  /* Return the remaining page to the caller. */
#endif

  /* allocation succeeded */

  gm_assert (page);
  gm_assert (GM_PAGE_LEN);
  gm_assert (GM_POWER_OF_TWO (GM_PAGE_LEN));
  gm_assert (GM_PAGE_ALIGNED (page));

  GM_PRINT (GM_PRINT_LEVEL >= 5, ("gm_page_alloc() returning 0x%p.\n", page));

  return page;
}

/* Free a page.  If all pages have been freed, free all the memory
   allocated for pages. */

void
gm_page_free (void *addr)
{
  GM_CALLED ();

  gm_assert (GM_PAGE_LEN);
  gm_assert (GM_POWER_OF_TWO (GM_PAGE_LEN));
  gm_assert (GM_PAGE_ALIGNED (addr));

  GM_PRINT (0, ("gm_page_free(%p) called\n", addr));

#if GM_KERNEL && GM_KERNEL_MALLOC_ALIGNS_PAGES
  gm_free (addr);
#elif !GM_KERNEL && GM_HAVE_MEMALIGN
  free (addr);
#else
  {
    struct gm_free_page *page;

    gm_assert ((((long) addr & (GM_PAGE_LEN - 1)) == 0));
    page = (struct gm_free_page *) addr;
    page->next = _gm_first_free_page;
    _gm_first_free_page = page;
    _gm_page_allocation_cnt--;
    if (!_gm_page_allocation_cnt)
      {
	while (_gm_first_page_allocation_record)
	  {
	    struct gm_page_allocation_record *next;

	    next = _gm_first_page_allocation_record->next;
	    gm_free (_gm_first_page_allocation_record->buf);
	    _gm_first_page_allocation_record = next;
	  }
	_gm_first_free_page = 0;
	GM_PRINT (GM_PRINT_LEVEL >= 1,
		  ("Freed all memory reserved for pages.\n"));
      }
  }
#endif
  GM_RETURN_NOTHING ();
}

/*
  This file uses GM standard indentation:

  Local Variables:
  c-file-style:"gnu"
  tab-width:8
  c-backslash-column:72
  End:
*/
