/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation                      *
 * Copyright (c) 1996, 1997 by Myricom, Inc.                             *
 * All rights reserved.  See the file `COPYING' for copyright notice.    *
 *************************************************************************/

#include "gm_call_trace.h"
#include "gm_compiler.h"
#include "gm_debug.h"
#include "gm_internal.h"
#include "gm_enable_ethernet.h"
#include "gm_enable_trace.h"

#if GM_KERNEL
#include "gm_enable_security.h"
#endif

#if !GM_KERNEL
				/* open() */
#  ifdef HAVE_SYS_TYPES_H
#    include <sys/types.h>
#  endif

#  ifdef HAVE_ERRNO_H
/* I don't think there's any such symbol as HAVE_ERRNO_H  - Max, 2000/10/23 */
#    include <errno.h>
#  endif

#  ifdef HAVE_SYS_ERRNO_H
#    include <sys/errno.h>
#    include <errno.h>
#  endif
				/* open() */
#  ifdef HAVE_SYS_STAT_H
#    include <sys/stat.h>
#  endif
#  ifdef HAVE_FCNTL_H
#    include <fcntl.h>
#  endif

				/* sprintf() */
#  ifdef STDC_HEADERS
#    include <stdio.h>
#    include <string.h>
#  endif


				/* close() */
#  ifdef HAVE_UNISTD_H
#    include <unistd.h>
#  endif
#  if GM_OS_SOLARIS
int close (int);
#  endif
#endif /* !GM_KERNEL */

/************************************************************************
 * Driver entry points
 ************************************************************************/

/************************************************************************
 * gm_open family of calls
 ************************************************************************/

#if !GM_KERNEL
#  ifdef _WIN32
#    define GM_DEV_TEMPLATE "\\\\.\\gm%x"
#    define GM_PRIV_DEV_TEMPLATE "\\\\.\\gmp%x"
WSADATA gm_wsadata;
int gm_wsadata_initialized = 0;
#  else
#    define GM_DEV_TEMPLATE "/dev/gm%d"
#    define GM_PRIV_DEV_TEMPLATE "/dev/gmp%d"
#  endif
#endif

#ifdef JOHN_DEBUG
char *glob_sram;
gm_lanai_globals_t *glob_lg;
#endif

#if HAVE_SYS_ERRNO_H && !GM_KERNEL
/* This logic should be in a new libgm module. */
#define GM_STATUS_FROM_ERRNO(status) do { \
      switch (errno) {               \
      case EBUSY:                    \
	 status = GM_BUSY;           \
	 break;                      \
      case ENODEV:                   \
	 status = GM_NO_SUCH_DEVICE; \
	 break;                      \
      case ENOMEM:                   \
	 status = GM_OUT_OF_MEMORY;  \
	 break;                      \
      default:                       \
	 status = GM_FAILURE; }      \
      } while (0)
#endif

#if GM_OS_IRIX && !GM_KERNEL
#if maxstern
/* display the special ioctl args used by IRIX */
static void
display_setpn_args (gm_irix_set_port_num_arg_t *the_args, char *the_comment)
{
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("set_port_num_arg struct %s:\n", the_comment));

  if (!the_args)
    {
      _GM_PRINT (GM_PRINT_LEVEL >= 1,
		("    Args pointer is VOID\n") );
    }
  else
    {
      _GM_PRINT (GM_PRINT_LEVEL >= 1,
		("    Args pointer is 0x%p\n", the_args));
      _GM_PRINT (GM_PRINT_LEVEL >= 1,
		 ("    port_id   = %d\n", the_args->port_id));
      _GM_PRINT (GM_PRINT_LEVEL >= 1,
		 ("    name_size = %d\n", the_args->name_size));
      _GM_PRINT (GM_PRINT_LEVEL >= 1,
		 ("    name      = %s\n", ((the_args->name)
					   ? the_args->name
					   : "<NULL>")));
    }
}
#endif
#endif /* GM_OS_IRIX */

/* Open a LANai port and return a pointer to the gm_internal_port_t structure.

   The gm_internal_port_t structure is assembled by allocating an
   aligned buffer to store the structure and then using mmap() to map
   the appropriate parts of the the /dev/gm* device into this
   structure. */

GM_ENTRY_POINT gm_status_t
_gm_open_common (gm_port_t **port_p, unsigned unit, unsigned port_id,
		 const char *client_type, enum gm_api_version api_version)
{
  gm_status_t status = GM_SUCCESS;
  gm_port_t *port;
  unsigned int i;
  
#if GM_KERNEL

  /****************************************************************
   * Kernel version
   ****************************************************************/

  gm_arch_minor_t minor;
  gm_port_state_t *ps;
  gm_instance_state_t *is;
#else
#  if defined WIN32
  HANDLE temp_fd;
#  endif
#if GM_OS_IRIX
  gm_irix_set_port_num_arg_t irix_pn_args;
  gm_status_t                irix_ioctl_status;
#endif
  char *template, dev[40];
#endif
  GM_PARAMETER_MAY_BE_UNUSED (client_type);

  GM_CALLED_WITH_ARGS (("%p,%d,%d,%s,%d", port_p, unit, port_id, client_type,
			api_version));

  /************
   * Simple Checks
   ************/

  if (api_version < GM_API_VERSION_1_0
      || api_version > GM_API_VERSION_1_3)
    {
      GM_NOTE (("API version 0x%x not supported.\n", api_version));
      _GM_NOTE (("This version of GM supports only"
		 " GM_API_VERSION_1_0 through"
		 " GM_API_VERSION_1_3\n"));
      status = GM_INVALID_PARAMETER;
      goto abort_with_nothing;
    }
  
  if (unit > 9)
    {
      GM_NOTE (("overlarge unit number passed to _gm_open_common()\n"));
      status = GM_INVALID_PARAMETER;
      goto abort_with_nothing;
    }
  
  /************
   * Initialization
   ************/

  status = gm_init ();
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("gm_init() failed in _gm_open_common\n"));
      goto abort_with_nothing;
    }
  
  /************
   * Allocate storage for the port
   ************/

  /* allocate storage for the user port */
  port = (gm_port_t *) gm_malloc (sizeof (*port));
  if (!port)
    {
      GM_NOTE (("Could not allocate storage for user port (in kernel).\n"));
      status = GM_OUT_OF_MEMORY;
      goto abort_with_init;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1,
            ("Allocated Port at %p for unit %d, port %d\n",
             port, unit, port_id));
  gm_bzero (port, sizeof (*port));

  /************
   * Open the device (arrange for ioctl-based API functions to work)
   ************/

#if GM_KERNEL
  /* In the kernel, the ioctl-based API functions don't use IOCTLs, but
     rather access the kernel internals via the port->kernel_port_state
     field.  So, we just need to create initialize this field with
     a new gm_port_state and set its port number to cause it to
     become fully-initialized. */
  
  is = gm_instance_for_id (unit);
  if (!is)
    {
      GM_NOTE (("_gm_open_common: no instance for unit = %d\n",unit));
      status = GM_INTERNAL_ERROR;
      goto abort_with_port;
    }

  /* allocate storage for the kernel port_state */

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating minor node in the kernel.\n"));
  status = gm_minor_alloc (is, &minor);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not allocate storage for kernel port_state\n"));
      goto abort_with_port;
    }
  ps = port->kernel_port_state = gm_minor_get_port_state (minor);
  gm_assert (ps);
  if (!GM_ENABLE_SECURITY)
    ps->privileged = 1;
  ps->is_a_kernel_port = 1;
  
  /* open the port state */

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("opening port state in the kernel.\n"));
  status = gm_port_state_open (ps, port_id);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not open port state in kernel.\n"));
      goto abort_with_minor;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1,
            ("[%d]: opened port state for minor %d, port %d in the kernel.\n",
              unit, minor, port_id));
#else /* !GM_KERNEL */

  /****************************************************************
   * User version
   ****************************************************************/

  /* In user land, we open a new device to get (indirect) access to a
     kernel gm_port_state that allows the api functions to work.

     On unix, we get a new device by opening a clone device and
     setting the port number.

     On NT, we open a pseudo-clone device and set the port number,
     causing the device to create the device we really wanted.  We
     then close the pseudo-clone device and open the device we wanted
     to open in the first place. */

  if (port_id == GM_MAPPER_PORT_ID
      || port_id == GM_DAEMON_PORT_ID
      || port_id == GM_ETHERNET_PORT_ID)
    template = GM_PRIV_DEV_TEMPLATE;
  else
    template = GM_DEV_TEMPLATE;
  sprintf (dev, template, unit);

  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("about to try to open unit %d port %d as %s\n",
	     unit, port_id, dev));

  /****************
   * Open the clone device
   ****************/
  
#ifdef WIN32
#define GM_INVALID_FD INVALID_HANDLE_VALUE
  sprintf (dev, template, unit);
  GM_PRINT (GM_PRINT_LEVEL >= 3, ("Trying to open device \"%s.\"\n", dev));
  port->fd = CreateFile (dev, (GENERIC_READ | GENERIC_WRITE),
			 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_SYSTEM,
			 NULL);
#elif GM_OS_VXWORKS
  GM_NOTE (("VxWorks - no user open defined\n"));
  status = GM_PERMISSION_DENIED;
  goto abort_with_port;
#elif defined HAVE_ERRNO_H
/* I don't think there's any such symbol as HAVE_ERRNO_H  - Max, 2000/10/23 */
  do
    {
      port->fd = open (dev, O_RDWR);
    }
  while (port->fd == -1 && errno == EAGAIN);
#else  /* default open style */
  port->fd = open (dev, O_RDWR);
#endif /* default open style */

  /****
   * Verify the open
   ****/

#ifndef GM_INVALID_FD
#define GM_INVALID_FD (-1)
#endif
  
  if (port->fd == GM_INVALID_FD)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1, ("Could not open device \"%s\"\n", dev));
#if HAVE_SYS_ERRNO_H
      GM_STATUS_FROM_ERRNO(status);
#else      
      status = GM_CLONE_BUSY;
#endif
      goto abort_with_port;
    }

  /****************
   * ???
   ****************/

#if GM_OS_OSF1 || GM_OS_FREEBSD
  {
    int minor_id=-1;
    
    GM_PRINT (GM_PRINT_LEVEL >= 4,("dev %s\n", dev));
    if (ioctl (port->fd, GM_GET_DEV, &minor_id) == -1)
      {
	GM_NOTE (("Could not get_dev ioctl rv=-1  minor = %d errno = %d\n",
		     minor_id, errno));
	perror("GET_DEV failed");
	status = GM_INTERNAL_ERROR;
	goto abort_with_open_device;
      }
    close(port->fd);
    GM_PRINT (GM_PRINT_LEVEL >= 4,("new minor_id %d\n", minor_id));
    sprintf (dev, "/dev/gm%d", minor_id);
#if 0
    switch (minor_id) {
    case 2:
      sprintf (dev, "/dev/gm%d", 1);
      break;
    case 3:
      sprintf (dev, "/dev/gmp%d", 1);
      break;

    case 4:
      sprintf (dev, "/dev/gm%d", 2);
      break;
    case 5:
      sprintf (dev, "/dev/gmp%d", 2);
      break;	
    case 6:
      sprintf (dev, "/dev/gm%d", 3);
      break;
    case 7:
      sprintf (dev, "/dev/gmp%d", 3);
      break;
    default:
      break;
    }
#endif
    port->fd = open(dev, O_RDWR);


    if (port->fd == -1)
      {
	GM_PRINT (GM_PRINT_LEVEL >= 1,
		  ("Round 2 Could not open device \"%s\" %d\n", dev, errno));
	status = GM_BUSY;
	goto abort_with_port;
      } 
  }
#endif
  
  
  

  /****************
   * Set the port number
   ****************/

#if GM_OS_IRIX
/* special use of port-number-set to get new device name to open */

/* Do we need to do this check?  Oh, well, it can't hurt... ////// */
  if (port_id >= GM_NUM_PORTS)
    {
      GM_NOTE (("port_id %d larger than maximum %d\n",
                  port_id, GM_NUM_PORTS-1) );
      status = GM_NO_SUCH_DEVICE;
      goto abort_with_port;
    }

  irix_pn_args.port_id = port_id;
  irix_pn_args.name_size = GM_IRIX_OPEN_NAME_MAX_SZ;
  irix_pn_args.name = gm_malloc(GM_IRIX_OPEN_NAME_MAX_SZ+1);
  if (!irix_pn_args.name)
    {
       GM_PRINT (GM_PRINT_LEVEL >= 1,
		 ("gm_malloc failed to get irix_pn_args.name\n"));
       status = GM_BUSY;
       goto abort_with_port;
    }

  strcpy(irix_pn_args.name, "");

#if maxstern
  display_setpn_args(&irix_pn_args, "before _gm_user_ioctl");
#endif
  irix_ioctl_status = _gm_user_ioctl(port, GM_SET_PORT_NUM,
				     &irix_pn_args, sizeof(irix_pn_args));
  if (irix_ioctl_status != GM_SUCCESS)
    {
      gm_perror ("Special irix ioctl call failed with error",
                 irix_ioctl_status);
      status = GM_BUSY;
      goto abort_with_port;
    }
#if maxstern
  display_setpn_args(&irix_pn_args, "after _gm_user_ioctl");
#endif

  close(port->fd);
  port->fd = open(irix_pn_args.name, O_RDWR);
  if (port->fd == -1)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("Could not open device \"%s\"\n", irix_pn_args.name));
#if HAVE_SYS_ERRNO_H
      GM_STATUS_FROM_ERRNO(status);
#else      
      status = GM_BUSY;
#endif
      goto abort_with_port;
    }

#else  /* !GM_OS_IRIX */

  /* set the port number.  Must be done before most other IOCTLs or MMAPs. */

  GM_PRINT (GM_PRINT_LEVEL >= 4,  ("Setting port number to %d.\n", port_id));
  status = _gm_set_port_num (port, port_id);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Failed to set the port num for the opened port.\n"));
      goto abort_with_open_device;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 4,  ("Did set the port number.\n"));
#endif /* !GM_OS_IRIX */


  /* Print a warning and exit if the installed driver may not be 
     compatible with this version of libgm. */
  if (1)
  {
    char *kernel_build_id;

    kernel_build_id = _gm_get_kernel_build_id (port);
    if (!kernel_build_id)
      {
	GM_WARN (("The GM user library linked with this program may not\n"));
	_GM_WARN (("be compatible with the installed driver.\n"));
	_GM_WARN (("Driver is too old to determine its build ID\n"));
	_GM_WARN (("User library build ID is \"%s.\"\n", _gm_build_id));
	status = GM_FAILURE;
        goto abort_with_open_device;
      }
    else if (strcmp (kernel_build_id, _gm_build_id))
      {
	if (strncmp(kernel_build_id,_gm_version_plus_space,
		strlen(_gm_version_plus_space)))
	  {
	    GM_WARN (("The GM user library linked with this program may not\n"));
	    _GM_WARN (("be compatible with the installed driver.\n"));
	    _GM_WARN (("Driver build ID is     \"%s\".\n", kernel_build_id));
	    _GM_WARN (("User lib build ID is   \"%s\".\n", _gm_build_id));
	    _GM_WARN (("User lib gm_version is \"%s\".\n", 
			_gm_version_plus_space));
	    status = GM_FAILURE;
            goto abort_with_open_device;
	  }
	
	/* HACK: Free the kernel build ID to save space, since the user
	   is unlikely to call _gm_get_kernel_build_id() again */

	if (port->kernel_build_id)
	  {
	    gm_free (port->kernel_build_id);
	    port->kernel_build_id = 0;
	  }
      }
  }


  /****************
   * Open the cloned device, if needed.
   ****************/

#  if defined WIN32
  /* Under NT, close the clone device and open the port device. */
  
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("Closing the clone device and opening the cloned device.\n"));
  temp_fd = port->fd;
  sprintf (dev, "\\\\.\\gm%x-%x", unit, port_id);
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Opening the cloned device \"%s.\"\n", dev));
  port->fd = CreateFile (dev, (GENERIC_READ | GENERIC_WRITE),
			 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_SYSTEM,
			 NULL);
  if (port->fd == INVALID_HANDLE_VALUE)
    {
      GM_NOTE (("Could not open port device \"%s\".\n", dev));
      _GM_NOTE (("    GetLastError() returned 0x%x.\n", GetLastError()));
      status = GM_BUSY;
      goto abort_with_open_device;
    }
  CloseHandle (temp_fd);
  GM_PRINT (GM_PRINT_LEVEL >= 1,
	    ("Succeeded in opening the cloned device.\n"));
#  endif/* WIN32 */
#endif /* !GM_KERNEL */

  /****************
   * Device Mapping
   ****************/

  /****
   * Determine the page length
   ****/

  if (!GM_PAGE_LEN)
    {
#if GM_OS_VXWORKS
      GM_NOTE (("gm_open: Don't know the page length.\n"));
      _GM_NOTE (("gm_open: Setting page length = 4096\n"));
      GM_PAGE_LEN = 4096;
#else
      status = GM_INTERNAL_ERROR;
      goto abort_with_open_device;
#endif
    }

  /****
   * Alloc mappings storage
   ****/

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Allocating space for mappings struct.\n"));
  port->mappings = gm_malloc (sizeof (gm_mapping_specs_t));
  if (!port->mappings) {
    GM_PRINT (GM_PRINT_LEVEL >= 1, ("whoops!  gm_malloc failed\n"));
    status = GM_OUT_OF_MEMORY;
    goto abort_with_open_device;
  }

  /****
   * Get mapping specs
   ****/

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Getting mapping specs\n"));
  status = _gm_get_mapping_specs (port, port->mappings);
  if (status != GM_SUCCESS)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("_gm_get_mapping_specs returned bad status\n"));
      goto abort_with_mappings_mem;
    }

  /****
   * Map the real time clock
   ****/

  GM_PRINT (GM_PRINT_LEVEL >= 1, ("Mapping RTC\n"));
  status = _gm_mmap (port,
		     port->mappings->RTC.offset,
		     port->mappings->RTC.len,
		     GM_MAP_READ,
		     (void **) &port->RTC);
  if (status != GM_SUCCESS)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("gm_open: couldn't _gm_mmap the RTC page\n"));
      goto abort_with_recv_queue_mapping;
    }
  GM_PRINT (GM_PRINT_LEVEL >= 1, ("RTC at %p.\n", port->RTC));
  gm_assert (port->recv_queue_start != (void *)-1);
  if (!GM_PAGE_ALIGNED (port->RTC))
    {
      status = GM_INTERNAL_ERROR;
      goto abort_with_RTC_mapping;
    }

#if GM_ENABLE_TRACE
  /* we need to mmap the special registers to access the "real" RTC */
  if (_gm_mmap (port,
		port->mappings->special_regs.offset,
		port->mappings->special_regs.len,
		GM_MAP_READ,
		(void**)&port->lanai_special_regs) != GM_SUCCESS)
    goto abort_with_RTC_mapping;
#endif

  /* Why don't we treat this port like any other? */
  if (port_id != GM_DAEMON_PORT_ID)
    {

      /****
       * Map the send queue
       ****/

      status = _gm_mmap (port,
			 port->mappings->send_queue.offset,
			 port->mappings->send_queue.len,
			 GM_MAP_READ | GM_MAP_WRITE,
			 (void **) &port->lanai);
      if (status != GM_SUCCESS)
	goto abort_with_real_RTC_mapping;
      gm_assert (port->lanai != 0 && port->lanai != (void *) -1);
      gm_bzero (port->lanai, port->mappings->send_queue.len);

      /****
       * Map the receive queue
       ****/

      /* Map the receive queue, which is in host-side DMAable memory. Must
	 include an extra slot for the "sleep/wake" mechanism. */
      
      gm_assert (GM_MTU > GM_MAX_FAST_RECV_BYTES);
      status = _gm_mmap (port,
			 port->mappings->recv_queue.offset,
			 port->mappings->recv_queue.len,
			 GM_MAP_READ | GM_MAP_WRITE,
			 (void **) &port->recv_queue_start);
      if (status != GM_SUCCESS)
	goto abort_with_lanai_queues_mapping;
      gm_assert (port->recv_queue_start != (void *)-1);
      if (!GM_PAGE_ALIGNED (port->recv_queue_start))
	{
	  status = GM_INTERNAL_ERROR;
	  goto abort_with_recv_queue_mapping;
	}
      gm_bzero ((void *) port->recv_queue_start,
		port->mappings->recv_queue.len);
    }

  /****************
   * Constant caching
   ****************/

  status = gm_max_node_id (port, &port->max_node_id);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not get max node ID.\n"));
      goto abort_with_recv_queue_mapping;
    }

  status = gm_get_node_id (port, &port->this_node_id);
  if (status != GM_SUCCESS)
    {
      if (status == GM_NODE_ID_NOT_YET_SET)
	port->this_node_id = GM_NO_SUCH_NODE_ID;
      else
	{
	  GM_NOTE (("Could not get node id.\n"));
	  goto abort_with_recv_queue_mapping;
	}
    }

  status = gm_get_unique_board_id (port, port->unique_board_id);
  if (status != GM_SUCCESS)
    {
      GM_NOTE (("Could not get unique board id.\n"));
      goto abort_with_recv_queue_mapping;
    }

  {
    gm_pid_t my_pid;

    my_pid = gm_getpid ();
    status = _gm_set_opener_pid (port,my_pid);
    if (status != GM_SUCCESS)
      {
        GM_NOTE (("Could not set the 'opener_pid'.\n"));
      }
  }

  /************
   * Finish initializing port.
   ************/

  port->api_version = api_version;
#if 0
  port->handler.sent = _gm_handle_sent;
  port->handler.recv = _gm_handle_recv;
  port->handler.high_recv = _gm_handle_high_recv;
  port->handler.rejected_send = _gm_handle_rejected_send;
  port->handler.orphaned_send = _gm_handle_orphaned_send;
  port->handler.dropped_send = _gm_handle_dropped_send;
#endif
  /* port->kernel_port_state set above */
  /* port->this_node_id set above */
  /* port->max_node_id set above */
  /* port->sleeping = 0 */

  /*** Initializes pointers to LANai-resident queues ***/

  port->send_queue_start = &port->lanai->send_token_queue[0];
  port->send_queue_limit = (&port->lanai->send_token_queue
			    [GM_NUM_SEND_QUEUE_SLOTS]);

  port->first_free_send_queue_slot_token = &port->send_queue_slot_token[0];
  for (i = 0; i < GM_NUM_ELEM (port->send_queue_slot_token) - 1; i++)
    {
      port->send_queue_slot_token[i].next
	= &port->send_queue_slot_token[i+1];
    }
  port->last_free_send_queue_slot_token = &port->send_queue_slot_token[i];
  port->last_free_send_queue_slot_token->next = 0;
  
  port->recv_token_queue_start = &port->lanai->recv_token_queue[0];
  port->recv_token_queue_slot = port->recv_token_queue_start;
  port->recv_token_queue_limit = (&port->lanai->recv_token_queue
				  [GM_NUM_RECV_TOKEN_QUEUE_SLOTS]);
  
  /* Initialize pointers to host-memory-resident queues. */

  /* recv_queue_start set above */
  port->recv_queue_slot = port->recv_queue_start;
  port->recv_queue_limit = &port->recv_queue_start[GM_NUM_RECV_QUEUE_SLOTS];

  /* port->num_zones = 0 */
  /* port->dma_zone = {0} */
  /* alloced_dma_mem_len = 0 */
  port->unit = unit;
  port->id = port_id;
  /* max_used_id; */
  /* _recv_tokens[GM_NUM_RECV_TOKENS + 1] = {0} */
  /* lanai_globals = 0 */
  /* lanai_special_regs = 0; */
  /* fd set above */
  /* mappings set above */
  /* sleep_pending = 0 */
  /* trash = 0 */
  /* unique_board_id set above */
  /* first_set_alarm = 0 */
  /* first_unset_alarm = 0 */
  /* lanai_alarm_pending = 0 */
  /* lanai_alarm_flush_pending = 0 */
  /* ignore_next_lanai_alarm = 0 */

#if GM_KERNEL && GM_ENABLE_ETHERNET
  if (port_id == GM_ETHERNET_PORT_ID)
    {
      GM_PRINT (GM_PRINT_LEVEL >= 1,
		("doing ethernet port thing in gm_open\n"));
      port->ethernet.recv_token_queue_start
	= port->ethernet.recv_token_queue_slot
	= (void *) ((char *) is->lanai.globals
		    + GM_OFFSETOF (gm_lanai_globals_t,
				   ethernet.recv.token[0]));
      port->ethernet.recv_token_queue_limit
	= (void *) ((char *) is->lanai.globals
		    + GM_OFFSETOF (gm_lanai_globals_t,
				   ethernet.recv.token
				   [GM_NUM_ETHERNET_RECV_TOKENS]));
    }
#endif

  /* commit */

  _gm_open_ports_insert (port);
  *port_p = port;
  GM_RETURN_STATUS (GM_SUCCESS);

abort_with_recv_queue_mapping:
  /* only do this conditionally, since the mapping might have been skipped */
  if (port->recv_queue_start)
    _gm_munmap (port, (void *)port->recv_queue_start,
		GM_STATIC_CAST (unsigned long, port->mappings->recv_queue.len));
abort_with_lanai_queues_mapping:
  /* only do this conditionally, since the mapping might have been skipped */
  if (port->send_queue_start)
    _gm_munmap (port, (void *)&port->send_queue_start,
		GM_STATIC_CAST (unsigned long, port->mappings->send_queue.len));
abort_with_real_RTC_mapping:
#if GM_ENABLE_TRACE
  if (port->lanai_special_regs)
    _gm_munmap (port, (void *)&port->lanai_special_regs,
		GM_STATIC_CAST (unsigned long, port->mappings->special_regs.len));
#endif
abort_with_RTC_mapping:
  if (port->RTC)
    _gm_munmap (port, (void *)port->RTC,
		GM_STATIC_CAST (unsigned long, port->mappings->RTC.len));
abort_with_mappings_mem:
  gm_free (port->mappings);
abort_with_open_device:
#if GM_KERNEL
  gm_port_state_close (ps);
abort_with_minor:
  gm_minor_free (minor);
#else 
#  ifdef _WIN32
  if (port->fd != INVALID_HANDLE_VALUE)
    CloseHandle (port->fd);
#  else
  close (port->fd);
#  endif
#endif
abort_with_port:
  gm_free (port);
abort_with_init:
  gm_finalize ();
abort_with_nothing:
  GM_RETURN_STATUS (status);
}

/* Version of gm_open to be called by clients other than the daemon
   and mapper. */
GM_ENTRY_POINT gm_status_t
gm_open (struct gm_port **port, unsigned unit, unsigned port_id,
	 const char *client_type, enum gm_api_version version)
{
  return _gm_open_common (port, unit, port_id, client_type, version);
}

#if !GM_KERNEL
#ifdef _WIN32
/* Include an entry point if we are building a WIN32 DLL. */
BOOL WINAPI
gm_entry_point(HINSTANCE hinstDLL,  /* handle to DLL module */
	       DWORD fdwReason,     /* reason for calling function */
	       LPVOID lpReserved )  /* reserved */
{
  switch (fdwReason)  
    { 
    case DLL_PROCESS_ATTACH:
    case DLL_PROCESS_DETACH:
      return TRUE;

    default:
      return FALSE;
    }
}
#endif

#if 0 && loic && !GM_KERNEL
/* to be able to use gm_pseudo_global_clone with gdb, referencing it here forces its inclusion in the binary */
void gm_pseudo_global_clone();
static void(*dummy_func)()=gm_pseudo_global_clone;
#endif

#endif
