/******************************************************************-*-c-*-
 * Myricom GM networking software and documentation			 *
 * Copyright (c) 2000 by Myricom, Inc.					 *
 * All rights reserved.	 See the file `COPYING' for copyright notice.	 *
 *************************************************************************/

/* author: glenn@myri.com */

/* This file implements a constant-time system for marking memory
   locations and verifying that they have not been corrupted.  All
   operations have average execution time of O(1).  If enough marks
   were specified in the call to gm_create_mark_set(), then all
   operations have worst-case performance of O(1); otherwise, the
   worst-case performance is O(N) where N is the number of marks in
   the mark set.

   We do not simply use a magic number as a mark, or a mark set
   identifier as a mark, since this technique can lead to false
   positives where unitialized marks appear valid.  With this system,
   any localized corruption of a mark or the mark set database will
   result in the mark being determined to be invalid.  False positives
   in this system require that both the mark and mark set database be
   corrupted in a consistent way, which is basically impossible.

   This implementation was inspired by a challenge from John Regehr,
   who was forwarding a challenge from a UVA CS professor, paraphrased
   as follows: Can you implement the functionality of an array where
   initialization, insertion, removal, and test for presense of an
   array element each has execution time bounded by a constant,
   assuming you are initially provided with a pointer to a sufficient
   amount of uninitialized memory.  You are only allowed O(N) storage,
   where N is the max number of entries."  The anwer is "Yes" and a
   lot of the same tricks are used here. */

#include "gm.h"

/* a reference to a mark, or a free entry.  The reference is used to validate
   the mark.  That mark is considered valid if and only if
   (set->reference[mark->tag] == mark). */

typedef union gm_mark_reference
{
  /* For used mark references */
  gm_mark_t *mark;
  /* For free marks references */
  union gm_mark_reference *next_free;
} gm_mark_reference_t;

/****************
 * Globals
 ****************/

struct gm_mark_set
{
  /* Array of all mark references, including those that are free. */
  gm_mark_reference_t *reference;
  /* One larger than the maximum tag that has been used so far. */
  unsigned long tag_limit;
  /* The number of marks that have been allocated. */
  unsigned long allocated_reference_count;
  /* A list of any marks references freed by gm_unmark(). */
  gm_mark_reference_t *free_references;
  /* The number of references preallocated at init time. */
  unsigned long init_cnt;
};

/* increase the number of preallocated mark references in the mark set */

static
gm_status_t
gm_mark_set_grow (struct gm_mark_set *set, unsigned long cnt)
{
  /* Catch the acceptable initial case of cnt == 0 */
  
  if (cnt == set->allocated_reference_count)
    return GM_SUCCESS;
  
  gm_assert (cnt > set->allocated_reference_count);
  if (!set->reference)
    {
      set->reference = ((gm_mark_reference_t *)
			gm_malloc (cnt * sizeof (gm_mark_reference_t)));
      if (!set->reference)
	{
	  return GM_OUT_OF_MEMORY;
	}
    }
  else
    {
      gm_mark_reference_t *new_references;
      
      new_references = ((gm_mark_reference_t *)
			gm_malloc (cnt * sizeof (gm_mark_reference_t)));
      if (!new_references)
	{
	  return GM_OUT_OF_MEMORY;
	}
      gm_bcopy (set->reference, new_references,
		cnt * sizeof (gm_mark_reference_t));
      gm_free (set->reference);
      set->reference = new_references;
    }
  set->allocated_reference_count = cnt;
  return GM_SUCCESS;
}

/* Allocate a message reference. */

static
gm_status_t
gm_mark_reference_alloc (struct gm_mark_set *set, gm_mark_reference_t **mrp)
{
  /* Try to return a mark reference from a previous unmark.  Failing
     that, allocate one from the end of the mark reference array. */
  
  if (set->free_references)
    {
      /* return reference from a previous unmark */
      
      *mrp = set->free_references;
      set->free_references = (*mrp)->next_free;
    }
  else
    {
      /* If there are no free marks on the end of the array
	 grow the array of valid marks. */
  
      if (set->allocated_reference_count <= set->tag_limit)
	{
	  gm_status_t status;
	  
	  status
	    = gm_mark_set_grow (set, set->allocated_reference_count*2 + 1);
	  if (status != GM_SUCCESS)
	    {
	      return status;
	    }
	}

      /* Allocate a free mark reference from the end of the array */
      
      *mrp = &set->reference[set->tag_limit++];
    }
  return GM_SUCCESS;
}

static
void
gm_mark_reference_free (struct gm_mark_set *set, gm_mark_reference_t *mr)
{
  mr->next_free = set->free_references;
  set->free_references = mr;
}

/****************************************************************
 * Entry points
 ****************************************************************/

GM_ENTRY_POINT
gm_status_t
gm_mark (struct gm_mark_set *set, gm_mark_t *m)
{
  gm_mark_reference_t *mr;
  gm_status_t status;

  if (gm_mark_is_valid (set, m))
    {
      return GM_SUCCESS;
    }
  
  /* allocate a mark reference to buddy with the mark */
  
  status = gm_mark_reference_alloc (set, &mr);
  if (status != GM_SUCCESS)
    {
      return status;
    }

  /* Buddy up the mark and its reference.  That is, make each reference
     the other. */

  *m = mr - &set->reference[0];
  mr->mark = m;
  
  return GM_SUCCESS;
}

/* return 1 if the mark is valid, else 0. */

GM_ENTRY_POINT
int
gm_mark_is_valid (struct gm_mark_set *set, gm_mark_t *m)
{
  unsigned long _m;
  
  gm_assert (m);
  _m = *m;
  return (_m < set->tag_limit
	  && set->reference[_m].mark == m);
}

/* unmark a valid mark */

GM_ENTRY_POINT
void
gm_unmark (struct gm_mark_set *set, gm_mark_t *m)
{
  unsigned long _m;
  gm_mark_reference_t *mr;
  
  /* Prefetch */

  gm_assert (m);
  _m = *m;
  gm_assert (_m < set->tag_limit);
  mr = &set->reference[_m];
  gm_assert (mr->mark == m);

  /* Simply free the mark reference.  Although this does not modify
     the mark, it effectively makes the mark invalid by eliminating
     the legitimizing reference to it. */

  gm_mark_reference_free (set, mr);
}

/* create a mark set */

GM_ENTRY_POINT
gm_status_t
gm_create_mark_set (struct gm_mark_set **msp, unsigned long cnt)
{
  struct gm_mark_set *ret;
  gm_status_t status;
  
  /* Allocate the mark set */

  ret = (struct gm_mark_set *) gm_calloc (1, sizeof (*ret));
  if (!ret)
    {
      status = GM_OUT_OF_MEMORY;
      goto abort_with_nothing;
    }
  ret->init_cnt = cnt;
  
  /* Preallocate the mark references */

  status = gm_mark_set_grow (ret, cnt);
  if (status != GM_SUCCESS)
    {
      goto abort_with_mark_set;
    }

  /* success! */
  
  *msp = ret;
  return GM_SUCCESS;

  /* error handling */

 abort_with_mark_set:
  gm_free (ret);
 abort_with_nothing:
  return status;
}

/* destroy a mark set */

GM_ENTRY_POINT
void
gm_destroy_mark_set (struct gm_mark_set *set)
{
  gm_assert (set);
  if (set->reference)
    {
      gm_free (set->reference);
    }
  gm_free (set);
}

/* unmark all marks for the mark set, freeing all but the initial
   number of preallocated mark references. */

GM_ENTRY_POINT
void
gm_mark_set_unmark_all (struct gm_mark_set *set)
{
  unsigned long init_cnt;
  unsigned long new_alloc_cnt;
  gm_mark_reference_t *new_references;
  
  init_cnt = set->init_cnt;
  new_references = ((gm_mark_reference_t *)
		    gm_malloc (init_cnt * sizeof (gm_mark_reference_t)));
  if (new_references)
    {
      gm_free (set->reference);
      new_alloc_cnt = init_cnt;
    }
  else
    {
      new_references = set->reference;
      new_alloc_cnt = set->allocated_reference_count;
    }
  gm_bzero (set, sizeof (*set));
  set->init_cnt = init_cnt;
  set->reference = new_references;
  set->allocated_reference_count = new_alloc_cnt;
}
